"""
A module for representing NOE data in ARIA's general NOE data format.
Contains methods to read and write NOE data in various formats
(e.g. XEASY, ANSIG, PIPP)
All the NOE data handling is done with this module.
"""
__author__   = "$Author: linge $"
__revision__ = "$Revision: 1.7 $"
__date__     = "$Date: 2000/01/24 18:46:31 $"

import copy, os, re, string

from Aria.Nomenclature import AminoAcid, PseudoAtom
from Aria.Comments import DeleteCnsComments
from Aria.DataIO import Comparisons
from Aria.ThirdParty import  FortranFormat, TextFile

###############################################################################
class NoeList:
    """
    Contains all NOE data of one spectrum in ARIA's general NOE data format
    supplies methods for input and output in various formats
    Author: Jens Linge, EMBL Heidelberg, 1998-1999

    public attributes:
        name           name of the spectrum
        comment        comment for the NOE list
        dimension      dimensionality of the spectrum (2, 3 or 4)
        fileNames      list of fileNames from which the data come from
        peaksdic       dictionary (key: (spectrumName, peakNumber), value: peakobject)
        peakslist      list of NOE peaks (with their contributions)
        intraList      list of NOE peaks (with their contributions)
        seqList        list of NOE peaks (with their contributions)
        mediumList     list of NOE peaks (with their contributions)
        longList       list of NOE peaks (with their contributions)
        unassignedList list of NOE peaks (with their contributions)
        ambiguousList  list of NOE peaks (with their contributions)
    public methods:
        AddPeak               adds one peak (peakobject)
        CountNontrivial       count the number of assignments of each peak
        IntraSeqMediumLong    splits the peak list into intra, medium and long
        ReadAnsigCpe          reads a crosspeaks export file
        ReadAnsigNoe          reads a general NOE restraints file
        ReadAssignPeaksProt   insert a spectrum, reading XEASY .assign, .peaks,
                              and .prot files
         ReadAureliaUserPeak   reads an AURELIA user peak list
        ReadList              insert a spectrum, reading ARIA's .list file
        ReadListPeaksProt     insert a spectrum, reading XEASY .peaks, .prot,
                              and ARIA's .list files
         ReadLolUpl            reads DYANA .lol and .upl files
        ReadOldListPeaksProt  insert a spectrum, reading XEASY .peaks, .prot,
                              and ARIA's old .list files
        ReadPeaksProt         insert a spectrum, reading XEASY .peaks and .prot
        ReadPippPck           insert a spectrum, reading PIPP's .PCK file
        ReadRegine            insert a spectrum, reading a Regine file
        ReadTbl               insert a spectrum, reading ARIA's .tbl files
        RemovePeak            removes one peak (specified by its peakNumber)
        Sort                  sorting after spectrumName and peakNumber
        Stdout                writes the data in the new .list format to stdout
        WriteAnsigNoe         writes an ANSIG general NOE restraints file
        WriteAureliaUserPeak  writes an AURELIA user peak list
        WriteList             writes a list in ARIA's .list format
        WriteOldList          writes a list in ARIA's old .list format
        WriteTbl              writes .tbl in ARIA format
        WriteUplLol           writes .upl and .lol files for MolMol
        WriteXeasyAssignPeaks writes .peaks and .assign files
        WriteXml              writes an XML file

    NOTE:
        -all fields are treated internally as strings, except:
          atomnames are always tuples of strings
          dimension is always an integer
        -empty fields are set to None, showns as '-' in stdout
        -there are internal objects for XEASY .assign, .peaks and .prot
    """
    def __init__(self, name = '', dimension = 0, comment = 'ARIA peak list'):
        self.fileNames = []
        self.name = name
        self.comment = comment
        self.dimension = dimension
        self.peakslist = []
        self.intraList = []
        self.seqList = []
        self.mediumList = []
        self.longList = []
        self.unassignedList = []
        self.ambiguousList = []
        self.peaksdic = {}
        #internal attributes for XEASY .peaks files:
        self._iname1 = '?'
        self._iname2 = '?'
        self._iname3 = '?'
        self._iname4 = '?'
        

    def __repr__(self):
        return self.name

    def _cleanUp(self):
        #delete all the existing peaks:
        self.peakslist = []
        self.intraList = []
        self.seqList = []
        self.mediumList = []
        self.longList = []
        self.unassignedList = []
        self.ambiguousList = []
    
    def AddPeak(self, peakobject):
        self.peakslist.append(peakobject)
        self.peaksdic[(peakobject.spectrumName, peakobject.peakNumber)] = peakobject
        

    def CountNontrivial(self):
        if len(self.peakslist) == 0:
            print 'peakslist is empty. CountNontrivial method aborted.'
            return
        print 'counting the contributions for each peak and setting the attribute nta for each contribution.'
        for EACHP in self.peakslist:
            for EACHCON in EACHP.contributions:
                #convert to string:
                EACHCON.nta = str(len(EACHP.contributions))
        

    def IntraSeqMediumLong(self):
        """
        splits self.peakslist into shorter lists:
          self.intraList          |i-j| = 0
          self.seqList            |i-j| = 1
          self.mediumList         1 < |i-j| < 5
          self.longList           |i-j| > 4
          self.unassignedList
          self.ambiguousList

        the self.peakslist attribute remains untouched
        """
        print 'sorting the peaks: intra, seq, medium, long, unassigned, ambiguous'
        for eachPeak in self.peakslist:
            if len(eachPeak.contributions) > 1:
                self.ambiguousList.append(eachPeak)
            elif len(eachPeak.contributions) == 0:
                continue
            else:
                if eachPeak.contributions[0].residue1 != None and \
                   eachPeak.contributions[0].residue2 != None:
                    if abs(string.atoi(eachPeak.contributions[0].residue1) -\
                           string.atoi(eachPeak.contributions[0].residue2)) == 0:
                        self.intraList.append(eachPeak)
                    elif abs(string.atoi(eachPeak.contributions[0].residue1) -\
                           string.atoi(eachPeak.contributions[0].residue2)) == 1:
                        self.seqList.append(eachPeak)
                    elif abs(string.atoi(eachPeak.contributions[0].residue1) -\
                             string.atoi(eachPeak.contributions[0].residue2)) < 5:
                        self.mediumList.append(eachPeak)
                    else:
                        self.longList.append(eachPeak)
                else:
                    self.unassignedList.append(eachPeak)

        #some messages for stdout:
        print len(self.intraList), len(self.seqList), len(self.mediumList),\
              len(self.longList), len(self.unassignedList), len(self.ambiguousList)
              


    def ReadAnsigCpe(self, fileName, het1, pro1, het2, pro2):
        """
        reads an ANSIG crosspeaks export file or an ANSIG storage crosspeaks file
        """
        if _DoesFileExist(fileName) == 0:
            return
        print 'reading the ANSIG crosspeaks export file:\n ', fileName
        
        #important - delete all the existing peaks:
        self._cleanUp()

        
        cpHandle = TextFile.TextFile(fileName)
        
        bang = re.compile('#')

        #some default values:
        peakType = None
        segid1 = None
        segid2 = None
        volumeError = None
        distance = None
        distanceError = None
        lowerBound = None
        upperBound = None
        dh1ppm = None
        dp1ppm = None
        dh2ppm = None
        dp2ppm = None

        #get total number of crosspeaks and dimensionality from the first two lines:
        eachLine = cpHandle.readline()
        eachLine = cpHandle.readline()
        
        totAndDim = string.split(eachLine)
        totalNumber = totAndDim[-2]
        self.dimension = int(totAndDim[-1])

        #there are two possible cases:
        #1. ANSIG v3.3 export crosspeaks file
        #   second line contains 'totalNumber dimensionality'
        #   spectrumName appears in every line
        #2. ANSIG v3.3 storage crosspeaks file
        #   second line contains 'spectrumName totalNumber dimensionality'
        #   spectrumName appears only in the header

        if len(totAndDim) == 2:  #for the ANSIG crosspeaks files           
            format2 = FortranFormat.FortranFormat('3E13.6,A12,7I6,6A4')
            format3 = FortranFormat.FortranFormat('4E13.6,A12,9I6,9A4')
            format4 = FortranFormat.FortranFormat('5E13.6,A12,11I6,12A4')

            #use (lineNumber - 2) as peak number:
            lineNumber = 2

            for eachLine in cpHandle:
                lineNumber = lineNumber + 1
                if bang.match(eachLine) or len(eachLine) < 40:
                    continue

                #some more default values:
                atomnameh1 = (None,)
                atomnameh2 = (None,)
                h1ppm = None
                h2ppm = None

                exec 'lineList = FortranFormat.FortranLine(eachLine, format' + str(self.dimension) +')'
                outS = ''
                if het1 != 'N':
                    outS = outS + 'h1ppm = string.strip(str(lineList[' + het1 + '-1]));'
                    outS = outS + 'atomnameh1 = (string.strip(lineList[' + str(self.dimension * 5 + 4) + '+' + het1 + ']),);' 
                if het2 != 'N':
                    outS = outS + 'h2ppm = string.strip(str(lineList[' + het2 + '-1]));'
                    outS = outS + 'atomnameh2 = (string.strip(lineList[' + str(self.dimension * 5 + 4) + '+' + het1 + ']),);'

                outS = outS + 'p1ppm = string.strip(str(lineList[' + pro1 + '-1]));'
                outS = outS + 'p2ppm = string.strip(str(lineList[' + pro2 + '-1]));'

                #ANSIG intensity => ARIA volume:
                outS = outS + 'volume = string.strip(str(lineList[' + str(self.dimension) + ']));' 
                outS = outS + 'spectrumName = string.strip(lineList[' + str(self.dimension + 1) + ']);' 

                outS = outS + 'residue1 = string.strip(str(lineList[' + str(self.dimension * 3 + 4) + '+' + pro1 + ']));' 
                outS = outS + 'residue2 = string.strip(str(lineList[' + str(self.dimension * 3 + 4) + '+' + pro2 + ']));' 

                outS = outS + 'aa1 = string.upper(string.strip(lineList[' +\
                     str(self.dimension * 4 + 4) + '+' + pro1 + ']));' 
                outS = outS + 'aa2 = string.upper(string.strip(lineList[' +\
                     str(self.dimension * 4 + 4) + '+' + pro2 + ']));' 

                outS = outS + 'atomnamep1 = (string.strip(lineList[' + str(self.dimension * 5 + 4) + '+' + pro1 + ']),);' 
                outS = outS + 'atomnamep2 = (string.strip(lineList[' + str(self.dimension * 5 + 4) + '+' + pro2 + ']),);' 
                exec outS

                if h1ppm == '': h1ppm = None
                if p1ppm == '': p1ppm = None
                if h2ppm == '': h2ppm = None
                if p2ppm == '': p2ppm = None
                if volume == '': volume = None
                if spectrumName == '': spectrumName = None
                if residue1 == '': residue1 = None
                if residue2 == '': residue2 = None
                if aa1 == '': aa1 = None
                if aa2 == '': aa2 = None
                if atomnameh1 == ('',): atomnameh1 = (None,)
                if atomnameh2 == ('',): atomnameh2 = (None,)
                if atomnamep1 == ('',): atomnamep1 = (None,)
                if atomnamep2 == ('',): atomnamep2 = (None,)

                #for the strange residue numbers, e.g. '72?'
                if residue1:
                    if residue1[-1] == '?':
                        residue1 = residue1[:-1]
                if residue2:
                    if residue2[-1] == '?':
                        residue2 = residue2[:-1]
                
                #put it in a contribution instance:
                CONT = NoeContribution(peakNumber = str(lineNumber - 2),\
                                       peakType = peakType,\
                                       residue1 = residue1,\
                                       aa1 = aa1 ,\
                                       segid1 = segid1,\
                                       atomnameh1 = atomnameh1,\
                                       atomnamep1 = atomnamep1,\
                                       residue2 = residue2,\
                                       aa2 = aa2,\
                                       segid2 = segid2,\
                                       atomnameh2 = atomnameh2,\
                                       atomnamep2 = atomnamep2,\
                                       volume = volume,\
                                       volumeError = volumeError,\
                                       distanceAve = distance,\
                                       distanceStd = distanceError,\
                                       lowerBound = lowerBound,\
                                       upperBound = upperBound,\
                                       h1ppm = h1ppm,\
                                       dh1ppm = dh1ppm,\
                                       p1ppm = p1ppm,\
                                       dp1ppm = dp1ppm,\
                                       h2ppm = h2ppm,\
                                       dh2ppm = dh2ppm,\
                                       p2ppm = p2ppm,\
                                       dp2ppm = dp2ppm)

                #ANSIG specific attributes:
                CONT.spectrumName = spectrumName

                #create a peak instance and add contribution:
                NOE = NoePeak()
                NOE.AddContribution(CONT)

                #add peak to the NOE list:
                self.AddPeak(NOE)
            
        else:  #for the ANSIG storage files
            format2 = FortranFormat.FortranFormat('3E13.6,5I6,6A4')
            format3 = FortranFormat.FortranFormat('4E13.6,7I6,9A4')
            format4 = FortranFormat.FortranFormat('5E13.6,9I6,12A4')
            spectrumName = totAndDim[0]
            
            #use (lineNumber - 2) as peak number:
            lineNumber = 2

            for eachLine in cpHandle:
                lineNumber = lineNumber + 1
                if bang.match(eachLine) or len(eachLine) < 40:
                    continue

                #some more default values:
                atomnameh1 = (None,)
                atomnameh2 = (None,)
                h1ppm = None
                h2ppm = None

                exec 'lineList = FortranFormat.FortranLine(eachLine, format' + str(self.dimension) +')'
                outS = ''
                if het1 != 'N':
                    outS = outS + 'h1ppm = string.strip(str(lineList[' + het1 + '-1]));'
                    outS = outS + 'atomnameh1 = (string.strip(lineList[' + str(self.dimension * 5 + 1) + '+' + het1 + ']),);' 
                if het2 != 'N':
                    outS = outS + 'h2ppm = string.strip(str(lineList[' + het2 + '-1]));'
                    outS = outS + 'atomnameh2 = (string.strip(lineList[' + str(self.dimension * 5 + 1) + '+' + het1 + ']),);'

                outS = outS + 'p1ppm = string.strip(str(lineList[' + pro1 + '-1]));'
                outS = outS + 'p2ppm = string.strip(str(lineList[' + pro2 + '-1]));'

                #ANSIG intensity => ARIA volume:
                outS = outS + 'volume = string.strip(str(lineList[' + str(self.dimension) + ']));' 
                outS = outS + 'residue1 = string.strip(str(lineList[' + str(self.dimension * 3 + 1) + '+' + pro1 + ']));' 
                outS = outS + 'residue2 = string.strip(str(lineList[' + str(self.dimension * 3 + 1) + '+' + pro2 + ']));' 

                outS = outS + 'aa1 = string.upper(string.strip(lineList[' +\
                     str(self.dimension * 4 + 1) + '+' + pro1 + ']));' 
                outS = outS + 'aa2 = string.upper(string.strip(lineList[' +\
                     str(self.dimension * 4 + 1) + '+' + pro2 + ']));' 

                outS = outS + 'atomnamep1 = (string.strip(lineList[' + str(self.dimension * 5 + 1) + '+' + pro1 + ']),);' 
                outS = outS + 'atomnamep2 = (string.strip(lineList[' + str(self.dimension * 5 + 1) + '+' + pro2 + ']),);' 
                exec outS

                if h1ppm == '': h1ppm = None
                if p1ppm == '': p1ppm = None
                if h2ppm == '': h2ppm = None
                if p2ppm == '': p2ppm = None
                if volume == '': volume = None
                if residue1 == '': residue1 = None
                if residue2 == '': residue2 = None
                if aa1 == '': aa1 = None
                if aa2 == '': aa2 = None
                if atomnameh1 == ('',): atomnameh1 = (None,)
                if atomnameh2 == ('',): atomnameh2 = (None,)
                if atomnamep1 == ('',): atomnamep1 = (None,)
                if atomnamep2 == ('',): atomnamep2 = (None,)

                #put it in a contribution instance:
                CONT = NoeContribution(peakNumber = str(lineNumber - 2),\
                                       peakType = peakType,\
                                       residue1 = residue1,\
                                       aa1 = aa1 ,\
                                       segid1 = segid1,\
                                       atomnameh1 = atomnameh1,\
                                       atomnamep1 = atomnamep1,\
                                       residue2 = residue2,\
                                       aa2 = aa2,\
                                       segid2 = segid2,\
                                       atomnameh2 = atomnameh2,\
                                       atomnamep2 = atomnamep2,\
                                       volume = volume,\
                                       volumeError = volumeError,\
                                       distanceAve = distance,\
                                       distanceStd = distanceError,\
                                       lowerBound = lowerBound,\
                                       upperBound = upperBound,\
                                       h1ppm = h1ppm,\
                                       dh1ppm = dh1ppm,\
                                       p1ppm = p1ppm,\
                                       dp1ppm = dp1ppm,\
                                       h2ppm = h2ppm,\
                                       dh2ppm = dh2ppm,\
                                       p2ppm = p2ppm,\
                                       dp2ppm = dp2ppm)

                #ANSIG specific attributes:
                CONT.spectrumName = spectrumName

                #create a peak instance and add contribution:
                NOE = NoePeak()
                NOE.AddContribution(CONT)

                #add peak to the NOE list:
                self.AddPeak(NOE)


    def ReadAnsigNoe(self, noefile, het1, pro1, het2, pro2):
        """
        reads an ANSIG general NOE restraint file
        
        I use the same convention as in the .html file for ANSIG:
        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)

        I use the ANSIG 'absolute intensity' field for the ARIA volumes
        
        """
        #some messages:
        print 'reading the ANSIG general NOE file:'
        print '  ' + noefile

        #check if the files exist:
        _DoesFileExist(noefile)
        self._cleanUp()

        
        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read the general NOE file:
        noeHandle = TextFile.TextFile(noefile)
        bang = re.compile('!')
        format2 = FortranFormat.FortranFormat('I2,3A4,F8.3,3A4,F8.3,I6,A12,F6.1,2E11.4')
        format3 = FortranFormat.FortranFormat('I2,3A4,F8.3,3A4,F8.3,3A4,F8.3,I6,A12,F6.1,2E11.4')
        format4 = FortranFormat.FortranFormat('I2,3A4,F8.3,3A4,F8.3,3A4,F8.3,3A4,F8.3,I6,A12,F6.1,2E11.4')
        
        #unavailable data set to default:
        peakType = None
        segid1 = None
        segid2 = None
        volumeError = None
        distance = None
        distanceError = None
        lowerBound = None
        upperBound = None
        dh1ppm = None
        dp1ppm = None
        dh2ppm = None
        dp2ppm = None

        for eachLine in noeHandle:
            if bang.match(eachLine) or len(eachLine) < 10:
                continue
##            #replace tabs with whitespace (circumvent TR1 problem):
##            print eachLine, #test
##            eachLine = re.sub('\t', ' ', eachLine)
            #get the right format for the corresponding dimensionality:
            exec 'lineList = FortranFormat.FortranLine(eachLine, format' + eachLine[0:1] +')'
##            print lineList, lineList[3] #test
            self.dimension = lineList[0]
            outS = ''
            #residue1:
            outS = outS + 'residue1 = lineList[' + str(4 * string.atoi(pro1) - 3) + '];'
            outS = outS + 'aa1 = lineList[' + str(4 * string.atoi(pro1) - 2) + '];'
            if het1 != 'N':
                outS = outS + 'atomnameh1 = lineList[' + str(4 * string.atoi(het1) - 1) + '];'
                outS = outS + 'h1ppm = lineList[' + str(4 * string.atoi(het1)) + '];'
            else:
                atomnameh1 = None
                h1ppm = None
            outS = outS + 'atomnamep1 = lineList[' + str(4 * string.atoi(pro1) - 1) + '];'
            outS = outS + 'p1ppm = lineList[' + str(4 * string.atoi(pro1)) + '];'
            #residue2:
            outS = outS + 'residue2 = lineList[' + str(4 * string.atoi(pro2) - 3) + '];'
            outS = outS + 'aa2 = lineList[' + str(4 * string.atoi(pro2) - 2) + '];'
            if het2 != 'N':
                outS = outS + 'atomnameh2 = lineList[' + str(4 * string.atoi(het2) - 1) + '];'
                outS = outS + 'h2ppm = lineList[' + str(4 * string.atoi(het2)) + '];'
            else:
                atomnameh2 = None
                h2ppm = None
            outS = outS + 'atomnamep2 = lineList[' + str(4 * string.atoi(pro2) - 1) + '];'
            outS = outS + 'p2ppm = lineList[' + str(4 * string.atoi(pro2)) + '];'
            outS = outS + 'peakNumber = lineList[' + str(self.dimension * 4 + 1) + '];'
            outS = outS + 'spectrumName = lineList[' + str(self.dimension * 4 + 2) + '];'
            outS = outS + 'mixtime = lineList[' + str(self.dimension * 4 + 3) + '];'
            #use absolute intensities for the volumes:
            outS = outS + 'volume = lineList[' + str(self.dimension * 4 + 4) + '];'
            outS = outS + 'relativeint = lineList[' + str(self.dimension * 4 + 5) + '];'

            exec outS
            
            #put it in a contribution instance:
            CONT = NoeContribution(peakNumber = string.strip(str(peakNumber)),\
                                   peakType = peakType,
                                   residue1 = string.strip(str(residue1)),\
                                   aa1 = string.upper(string.strip(aa1)),\
                                   segid1 = segid1,\
                                   atomnameh1 = (string.strip(atomnameh1),),\
                                   atomnamep1 = (string.strip(atomnamep1),),\
                                   residue2 = string.strip(str(residue2)),\
                                   aa2 = string.upper(string.strip(aa2)),\
                                   segid2 = segid2,\
                                   atomnameh2 = (string.strip(atomnameh2),),\
                                   atomnamep2 = (string.strip(atomnamep2),),\
                                   volume = string.strip(str(volume)),\
                                   volumeError = volumeError,\
                                   distanceAve = distance,\
                                   distanceStd = distanceError,\
                                   lowerBound = lowerBound,\
                                   upperBound = upperBound,\
                                   h1ppm = string.strip(str(h1ppm)),\
                                   dh1ppm = dh1ppm,\
                                   p1ppm = string.strip(str(p1ppm)),\
                                   dp1ppm = dp1ppm,\
                                   h2ppm = string.strip(str(h2ppm)),\
                                   dh2ppm = dh2ppm,\
                                   p2ppm = string.strip(str(p2ppm)),\
                                   dp2ppm = dp2ppm)
            
            #ANSIG specific attributes:
            CONT.spectrumName = string.strip(str(spectrumName))
            CONT.mixtime = string.strip(str(mixtime))
            CONT.relativeint = string.strip(str(relativeint))
            
            #create a peak instance and add contribution:
            NOE = NoePeak()
            NOE.AddContribution(CONT)

            #add peak to the NOE list:
            self.AddPeak(NOE)
    
    def ReadAssignPeaksProt(self, assignfile, peaksfile, protfile,\
                            het1, pro1, het2, pro2):
        """
        reads Xeasy .assign, .peaks and .prot files
        
        I use the same convention as in the .html file for XEASY:
        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)

        I use pro1 and pro2 for the matrix -> binary format conversion

        the strategy for combining the information is:
        1. take all the peaks in the .peaks files
        2. all the peaks, for which there is an assignment
           in the .assign file and no complete assignment in the
           .peaks file, will get the possible assignments from the
           .assign file
           that means that fully assigned peaks from .peaks
           stay untouched
        """
        #some messages:
        print 'reading the XEASY files:'
        print '  ' + assignfile
        print '  ' + peaksfile
        print '  ' + protfile

        #for XEASY I need:
        from Aria.DataIO import ReadXeasy
        
        #internal XEASY data stores:
        self._ASSIGN = ReadXeasy.XeasyAssign() #internal store of XEASY .assign data
        self._PEAKS = ReadXeasy.XeasyPeaks()   #internal store of XEASY .peaks data
        self._PROT = ReadXeasy.XeasyProt()     #internal store of XEASY .prot data
        
        #check if the files exist:
        _DoesFileExist(assignfile)
        _DoesFileExist(peaksfile)
        _DoesFileExist(protfile)
        
        #for the documentation:
        self.fileNames.append(assignfile)
        self.fileNames.append(peaksfile)
        self._cleanUp()

        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read in .assign, use pro1 and pro2 for the matrix:
        #.first = pro1 and .second = pro2 !!!
        self._ASSIGN.ReadAssign(assignfile, pro1, pro2)
        
        #read in .peaks and .prot:
        self._PEAKS.ReadPeaks(peaksfile)
        self._PROT.ReadProt(protfile)
        
        #get the dimensionality from the .peaks file:
        self.dimension = self._PEAKS.dimension
        
        #get INAME entries from .peaks:
        self._iname1 = self._PEAKS.iname1
        self._iname2 = self._PEAKS.iname2
        self._iname3 = self._PEAKS.iname3
        self._iname4 = self._PEAKS.iname4
        
        #put all the information together (like described above):
        #start with the peaks from the .peaks file:
        for EACHP in self._PEAKS.peakslist:
            EACHP.spectrumName = None #for the spectrumName
            NEWP = NoeContribution()
            NEWP.peakNumber = EACHP.peakNumber

            #copy the user-defined spectrum type:
            getudst = copy.copy(EACHP.spectrumType)
            getinte = copy.copy(EACHP.integration)
            
            #get the right columns for the atomnumbers (0 => None):
            if het1 != 'N':
                NEWP.atomnumberh1 = getattr(EACHP, 'atomnumber' + het1)
                if NEWP.atomnumberh1 == '0':
                    NEWP.atomnumberh1 = None
            if pro1 != 'N':
                NEWP.atomnumberp1 = getattr(EACHP, 'atomnumber' + pro1)
                if NEWP.atomnumberp1 == '0':
                    NEWP.atomnumberp1 = None
            if het2 != 'N':
                NEWP.atomnumberh2 = getattr(EACHP, 'atomnumber' + het2)
                if NEWP.atomnumberh2 == '0':
                    NEWP.atomnumberh2 = None
            if pro2 != 'N':
                NEWP.atomnumberp2 = getattr(EACHP, 'atomnumber' + pro2)
                if NEWP.atomnumberp2 == '0':
                    NEWP.atomnumberp2 = None
            
            NEWP.peakType = EACHP.peakType
            NEWP.volume = EACHP.volume
            NEWP.volumeError = EACHP.volumeError
            NEWP.xeasyudst = EACHP.spectrumType
            NEWP.xeasyinte = EACHP.integration
            
            #get the right columns for the frequencies:
            if het1 != 'N':
                NEWP.h1ppm = getattr(EACHP, 'w' + het1)
            if pro1 != 'N':
                NEWP.p1ppm = getattr(EACHP, 'w' + pro1)
            if het2 != 'N':
                NEWP.h2ppm = getattr(EACHP, 'w' + het2)
            if pro2 != 'N':
                NEWP.p2ppm = getattr(EACHP, 'w' + pro2)
            
            #now copy this contribution to the peak:
            WHOLEP = NoePeak()
            WHOLEP.AddContribution(NEWP)
            
            #now copy the whole peak to the self.peakslist attribute:
            self.AddPeak(WHOLEP)
            
        #now have a look at the .assign file:
        print 'ARIA uses all the peaks in the .peaks files.'
        print '  all the peaks, for which there is an assignment',\
              'in the .assign file'
        print '  and no complete assignment in the .peaks file, will get'
        print '  the possible assignments from the .assign file:'

        for EACHP in self._ASSIGN.assignments:
            EACHP.spectrumName = None #for the spectrumName
            #if peak is unambiguously assigned in .peaks, keep this peak:
            if self.peaksdic.has_key((EACHP.spectrumName, EACHP.peakNumber)):
                #check if all the assignments are there:
                if len(self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)].contributions) == 1:
                    if (het1 != 'N') and \
                       (self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)].contributions[0].atomnameh1 != (None, )):
                        continue
                    if (pro1 != 'N') and \
                       (self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)].contributions[0].atomnamep1 != (None, )):
                        continue
                    if (het2 != 'N') and \
                       (self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)].contributions[0].atomnameh2 != (None, )):
                        continue
                    if (pro2 != 'N') and \
                       (self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)].contributions[0].atomnamep2 != (None, )):
                        continue

##                #the old stuff:                
##                if len(self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)].contributions) > 1:
##                    continue


                
            #if ambiguous peak is in .peaks, it will be removed now:
            if self.peaksdic.has_key((EACHP.spectrumName, EACHP.peakNumber)):
#                print '    replacing peak', EACHP.peakNumber, 'from .peaks with',\
#                      'the corresponding peak from .assign'
                OLDPEAK = self.peaksdic[(EACHP.spectrumName, EACHP.peakNumber)]
                self.RemovePeak(EACHP.spectrumName, EACHP.peakNumber)
##             print EACHP.peakNumber, EACHP.dimension, EACHP.binlist, EACHP.first,\
##                   EACHP.second, EACHP.possible1, EACHP.possible2, EACHP.possible3,\
##                   EACHP.possible4   #test
            else:
                OLDPEAK = None
                
            #I have to know which column belongs to h1, p1, h2, p2
            #.first = pro1 and .second = pro2 !!! see above

            #initialize the lists:
            h1list = []
            p1list = []
            h2list = []
            p2list = []

            #find the right lists for h1, p1, h2, p2:
            if het1 != 'N':
                h1list = getattr(EACHP, 'possible' + het1)
            if pro1 != 'N':
                p1list = getattr(EACHP, 'possible' + pro1)
            if het2 != 'N':
                h2list = getattr(EACHP, 'possible' + het2)
            if pro2 != 'N':
                p2list = getattr(EACHP, 'possible' + pro2)

            #set the default to [None] instead of []:
            if h1list == []:
                h1list = [None]
            if p1list == []:
                p1list = [None]
            if h2list == []:
                h2list = [None]
            if p2list == []:
                p2list = [None]    
                
            #create a NoePeak instance and get the peak number:
            NOEP = NoePeak()
            NOEP.peakNumber = EACHP.peakNumber
                
            #set the attributes of NoeContribution,
            #loop through all possible assignments:

            #add some of the peak attributes from the removed peak of .peaks:
            def _addAtt(oldPeak, noeC):
                noeC.volume = oldPeak.contributions[0].volume
                noeC.volumeError = oldPeak.contributions[0].volumeError
                noeC.h1ppm = oldPeak.contributions[0].h1ppm
                noeC.dh1ppm = oldPeak.contributions[0].dh1ppm
                noeC.p1ppm = oldPeak.contributions[0].p1ppm
                noeC.dp1ppm = oldPeak.contributions[0].dp1ppm
                noeC.h2ppm = oldPeak.contributions[0].h2ppm
                noeC.dh2ppm = oldPeak.contributions[0].dh2ppm
                noeC.p2ppm = oldPeak.contributions[0].p2ppm
                noeC.dp2ppm = oldPeak.contributions[0].dp2ppm
                noeC.peakType = oldPeak.contributions[0].peakType

            #use a ReadXeasy XeasyAssign instance:
            print '###TEST'
            XA = ReadXeasy.XeasyAssign()
            #read in .assign, use pro1 and pro2 for the matrix:
            #.first = pro1 and .second = pro2 !!!
            first = pro1   #test: must be set to proton1
            second = pro2  #test: must be set to proton2
            XA.ReadAssign(assignfile, first, second)
            for eachAP in XA.assignments:
                print eachAP.possible1
                print eachAP.possible2
                print eachAP.assimatrix
                print eachAP.binlist
            #2D case:
            if (het1 == 'N') and (pro1 != 'N') and (het2 == 'N') \
               and (pro2 != 'N'):
                for eachp1 in p1list:
                    for eachp2 in p2list:
                        NOEC = NoeContribution()
                        NOEC.peakNumber = EACHP.peakNumber
                        NOEC.atomnumberp1 = eachp1
                        NOEC.atomnumberp2 = eachp2
                        NOEC.xeasyudst = getudst
                        NOEC.xeasyinte = getinte
                        #add old attributes:
                        if OLDPEAK:
                            _addAtt(OLDPEAK, NOEC)
                        #pool the contributions in NoePeak:
                        NOEP.AddContribution(NOEC)
                            
            #3D case, heteronucleus in het1:
            if (het1 != 'N') and (pro1 != 'N') and (het2 == 'N') \
               and (pro2 != 'N'):
                for eachh1 in h1list:
                    for eachp1 in p1list:
                        for eachp2 in p2list:
                            NOEC = NoeContribution()
                            NOEC.peakNumber = EACHP.peakNumber
                            NOEC.atomnumberh1 = eachh1
                            NOEC.atomnumberp1 = eachp1
                            NOEC.atomnumberp2 = eachp2
                            NOEC.xeasyudst = getudst
                            NOEC.xeasyinte = getinte
                            #add old attributes:
                            if OLDPEAK:
                                _addAtt(OLDPEAK, NOEC)
                            #pool the contributions in NoePeak:
                            NOEP.AddContribution(NOEC)
                            
            #3D case, heteronucleus in het2:
            if (het1 == 'N') and (pro1 != 'N') and (het2 != 'N') \
               and (pro2 != 'N'):
                for eachh2 in h2list:
                    for eachp1 in p1list:
                        for eachp2 in p2list:
                            NOEC = NoeContribution()
                            NOEC.peakNumber = EACHP.peakNumber
                            NOEC.atomnumberp1 = eachp1
                            NOEC.atomnumberh2 = eachh2
                            NOEC.atomnumberp2 = eachp2
                            NOEC.xeasyudst = getudst
                            NOEC.xeasyinte = getinte
                            #add old attributes:
                            if OLDPEAK:
                                _addAtt(OLDPEAK, NOEC)
                            #pool the contributions in NoePeak:
                            NOEP.AddContribution(NOEC)
                            
            #4D case:
            if (het1 != 'N') and (pro1 != 'N') and (het2 != 'N') \
               and (pro2 != 'N'):
                for eachh1 in h1list:
                    for eachh2 in h2list:
                        for eachp1 in p1list:
                            for eachp2 in p2list:
                                NOEC = NoeContribution()
                                NOEC.peakNumber = EACHP.peakNumber
                                NOEC.atomnumberh1 = eachh1
                                NOEC.atomnumberp1 = eachp1
                                NOEC.atomnumberh2 = eachh2
                                NOEC.atomnumberp2 = eachp2
                                NOEC.xeasyudst = getudst
                                NOEC.xeasyinte = getinte
                                #add old attributes:
                                if OLDPEAK:
                                    _addAtt(OLDPEAK, NOEC)
                                #pool the contributions in NoePeak:
                                NOEP.AddContribution(NOEC)
                                
            del OLDPEAK
            #add the peak to NoeList:
            self.AddPeak(NOEP)
            
        #now get the assigned shifts, shifterrors, atomnames and residue numbers
        #from .prot, loop through the whole NoeList:
        for EPEAK in self.peakslist:
            for ECON in EPEAK.contributions:
                if self._PROT.atomdican.has_key(ECON.atomnumberh1):
                    ECON.ah1ppm = self._PROT.atomdican[ECON.atomnumberh1].shift
                    ECON.dah1ppm = self._PROT.atomdican[ECON.atomnumberh1].shifterror
                    ECON.atomnameh1 = self._PROT.atomdican[ECON.atomnumberh1].ariaatomname
                    ECON.residue1 = self._PROT.atomdican[ECON.atomnumberh1].fragmentnumber
                else:
                    if not ((ECON.atomnumberh1 == None) or (ECON.atomnumberh1 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberh1, 'not found in .prot'
                if self._PROT.atomdican.has_key(ECON.atomnumberp1):
                    ECON.ap1ppm = self._PROT.atomdican[ECON.atomnumberp1].shift
                    ECON.dap1ppm = self._PROT.atomdican[ECON.atomnumberp1].shifterror
                    ECON.atomnamep1 = self._PROT.atomdican[ECON.atomnumberp1].ariaatomname
                    ECON.residue1 = self._PROT.atomdican[ECON.atomnumberp1].fragmentnumber
                else:
                    if not ((ECON.atomnumberp1 == None) or (ECON.atomnumberp1 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberp1, 'not found in .prot'
                if self._PROT.atomdican.has_key(ECON.atomnumberh2):
                    ECON.ah2ppm = self._PROT.atomdican[ECON.atomnumberh2].shift
                    ECON.dah2ppm = self._PROT.atomdican[ECON.atomnumberh2].shifterror
                    ECON.atomnameh2 = self._PROT.atomdican[ECON.atomnumberh2].ariaatomname
                    ECON.residue2 = self._PROT.atomdican[ECON.atomnumberh2].fragmentnumber
                else:
                    if not ((ECON.atomnumberh2 == None) or (ECON.atomnumberh2 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberh2, 'not found in .prot'
                if self._PROT.atomdican.has_key(ECON.atomnumberp2):
                    ECON.ap2ppm = self._PROT.atomdican[ECON.atomnumberp2].shift
                    ECON.dap2ppm = self._PROT.atomdican[ECON.atomnumberp2].shifterror
                    ECON.atomnamep2 = self._PROT.atomdican[ECON.atomnumberp2].ariaatomname
                    ECON.residue2 = self._PROT.atomdican[ECON.atomnumberp2].fragmentnumber
                else:
                    if not ((ECON.atomnumberp2 == None) or (ECON.atomnumberp2 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberp2, 'not found in .prot'
        

    def ReadList(self, fileName):
        """
        reads an ARIA .list file
        """
        self.fileNames.append(fileName)

        #check, if file exists:
        _DoesFileExist(fileName)
        
        #important: delete all the existing peaks
        self._cleanUp()

        #fileHandle:
        fileHandle = TextFile.TextFile(fileName)

        #for the first peak (see if statment below):
        PEAK=None

        for eachLine in fileHandle:
            lineList = string.split(eachLine)
            while '-' in lineList:
                index = lineList.index('-')
                lineList.remove('-')
                lineList.insert(index, None)
            if len(lineList) < 4:
                continue
            if lineList[0] == 'p':
                #add the last peak to peakslist:
                if PEAK:
                    self.AddPeak(PEAK)
                #new peak instance:
                PEAK = NoePeak()
                #get the data:
                spectrumName = lineList[1]
                peakNumber = lineList[2]
                testSet = lineList[3]
                inWeight = lineList[4]
                volume = lineList[5]
                volumeError = lineList[6]
                intensity = lineList[7]
                intensityError = lineList[8]
                h1ppm = lineList[9]
                dh1ppm = lineList[10]
                p1ppm = lineList[11]
                dp1ppm = lineList[12]
                h2ppm = lineList[13]
                dh2ppm = lineList[14]
                p2ppm = lineList[15]
                dp2ppm = lineList[16]
            elif lineList[0] == 'a':
                #get the data:
                peakType = lineList[1]
                fomAria = lineList[2]
                curWeight = lineList[3]
                lowerBound = lineList[4]
                upperBound = lineList[5]
                sumDistanceAve = lineList[6]
                sumDistanceStd = lineList[7]
                backVolumeAria = lineList[8]
                backVolumeAriaStd = lineList[9]
                allAssi = lineList[10]
                nta = lineList[11]
            elif lineList[0] == 'c':
                #get the data for one contribution:
                contribution = lineList[1]
                fomContribution = lineList[2]
                distanceAve = lineList[3]
                distanceStd = lineList[4]
                backVolume = lineList[5]
                backVolumeStd = lineList[6]
                segid1 = lineList[7]
                residue1 = lineList[8]
                aa1 = lineList[9]
                atomnameh1 = (lineList[10],) #atomname is a tuple
                assih1ppm = lineList[11]
                assidh1ppm = lineList[12]
                atomnamep1 = (lineList[13],) #atomname is a tuple
                assip1ppm = lineList[14]
                assidp1ppm = lineList[15]
                segid2 = lineList[16]
                residue2 = lineList[17]
                aa2 = lineList[18]
                atomnameh2 = (lineList[19],) #atomname is a tuple
                assih2ppm = lineList[20]
                assidh2ppm = lineList[21]
                atomnamep2 = (lineList[22],) #atomname is a tuple
                assip2ppm = lineList[23]
                assidp2ppm  = lineList[24]
                #put it into contribution instance:
                CONT = NoeContribution(spectrumName = spectrumName,\
                                       peakNumber = peakNumber,\
                                       testSet = testSet,\
                                       inWeight = inWeight,\
                                       volume = volume,\
                                       volumeError = volumeError,\
                                       intensity = intensity,\
                                       intensityError = intensityError,\
                                       h1ppm = h1ppm,\
                                       dh1ppm = dh1ppm,\
                                       p1ppm = p1ppm,\
                                       dp1ppm = dp1ppm,\
                                       h2ppm = h2ppm,\
                                       dh2ppm = dh2ppm,\
                                       p2ppm = p2ppm,\
                                       dp2ppm = dp2ppm,\
                                       peakType = peakType,\
                                       fomAria = fomAria,\
                                       curWeight = curWeight,\
                                       lowerBound = lowerBound,\
                                       upperBound = upperBound,\
                                       sumDistanceAve = sumDistanceAve,\
                                       sumDistanceStd = sumDistanceStd,\
                                       backVolumeAria = backVolumeAria,\
                                       backVolumeAriaStd = backVolumeAriaStd,\
                                       allAssi = allAssi,\
                                       nta = nta,\
                                       contribution = contribution,\
                                       fomContribution = fomContribution,\
                                       distanceAve = distanceAve,\
                                       distanceStd = distanceStd,\
                                       backVolume = backVolume,\
                                       backVolumeStd = backVolumeStd,\
                                       segid1 = segid1,\
                                       residue1 = residue1,\
                                       aa1 = aa1,\
                                       atomnameh1 = atomnameh1,\
                                       assih1ppm = assih1ppm,\
                                       assidh1ppm = assidh1ppm,\
                                       atomnamep1 = atomnamep1,\
                                       assip1ppm = assip1ppm,\
                                       assidp1ppm = assidp1ppm,\
                                       segid2 = segid2,\
                                       residue2 = residue2,\
                                       aa2 = aa2,\
                                       atomnameh2 = atomnameh2,\
                                       assih2ppm = assih2ppm,\
                                       assidh2ppm = assidh2ppm,\
                                       atomnamep2 = atomnamep2,\
                                       assip2ppm = assip2ppm,\
                                       assidp2ppm = assidp2ppm)
                
                #add contribution to peak:
                PEAK.AddContribution(CONT)
            else:
                print 'WARNING: incorrect format'
            
    
    def ReadListPeaksProt(self, listfile, peaksfile, protfile, het1, pro1, het2, pro2):
        """
        reads Xeasy .peaks and .prot file and an ARIA .list file
        this method is used to read the whole data set after a
        complete ARIA iteration

        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)
        """
        self.fileNames.append(peaksfile)
        self.fileNames.append(protfile)
        self.fileNames.append(listfile)

        #for XEASY I need:
        from Aria.DataIO import ReadXeasy
        
        #internal XEASY data stores:
        self._PEAKS = ReadXeasy.XeasyPeaks()   #internal store of XEASY .peaks data
        self._PROT = ReadXeasy.XeasyProt()     #internal store of XEASY .prot data

        #read .peaks and .prot:
        self._PEAKS.ReadPeaks(peaksfile)
        self._PROT.ReadProt(protfile)
        
        #take dimensionality from .peaks:
        self.dimension = self._PEAKS.dimension

        #XEASY inames for the .peaks header:
        self._iname1 = self._PEAKS.iname1
        self._iname2 = self._PEAKS.iname2
        self._iname3 = self._PEAKS.iname3
        self._iname4 = self._PEAKS.iname4

        #important: delete all the existing peaks
        self._cleanUp()
        
        #check, if files exist:
        _DoesFileExist(peaksfile)
        _DoesFileExist(protfile)
        _DoesFileExist(listfile)

        #insert spectrum from the .list file:
        self.ReadList(listfile)
        
        #update the data with the .peaks and .prot data:
        for PEAK in self.peakslist:
            for CON in PEAK.contributions:
##                 #get the peakType:
##                 CON.peakType = self._PEAKS.peaksdic[CON.peakNumber].peakType

##                 #if the spectrumName doesn't fit, don't use it!!!
##                 if CON.spectrumName != inSpectrumName:
##                     continue

                #get the XEASY integration method:
                CON.xeasyinte = self._PEAKS.peaksdic[CON.peakNumber].integration

                #get the XEASY user-defined spectrum-type:
                CON.xeasyudst = self._PEAKS.peaksdic[CON.peakNumber].spectrumType

                #get the missing heteronucleus atomnumbers from the .peaks file:
                if het1 != 'N':
                    CON.atomnumberh1 = getattr(self._PEAKS.peaksdic[CON.peakNumber], 'atomnumber' + het1)
#                    print  CON.atomnumberh1 #test
                    if CON.atomnumberh1 == '0':
                        CON.atomnumberh1 = None
                if het2 != 'N':
                    CON.atomnumberh2 = getattr(self._PEAKS.peaksdic[CON.peakNumber], 'atomnumber' + het2)
                    if CON.atomnumberh2 == '0':
                        CON.atomnumberh2 = None
                
                #get the missing proton atomnumbers, look up the atoms in the
                #.prot file, otherwise try pseudoatoms:
                #pro1:
                if self._PROT.atomdicfa.has_key((CON.residue1, CON.atomnamep1)): #normal case
                    CON.atomnumberp1 = self._PROT.atomdicfa[(CON.residue1, CON.atomnamep1)].atomnumber
                else: #now the pseudoatom case
                    pseudos = PseudoAtom.Atom2Pseudo(CON.atomnamep1[0], CON.aa1)
                    for pseudo in pseudos:
                        if pseudo in self._PROT.atomdicre[CON.residue1]:
                            CON.atomnumberp1 = self._PROT.atomdicfx[(CON.residue1, pseudo)].atomnumber
                            continue #29.1.99 always get the first hit
                #pro2:
                if self._PROT.atomdicfa.has_key((CON.residue2, CON.atomnamep2)): #normal case
                    CON.atomnumberp2 = self._PROT.atomdicfa[(CON.residue2, CON.atomnamep2)].atomnumber
                else: #now the pseudoatom case
                    pseudos = PseudoAtom.Atom2Pseudo(CON.atomnamep2[0], CON.aa2)
                    for pseudo in pseudos:
                        if pseudo in self._PROT.atomdicre[CON.residue2]:
                            CON.atomnumberp2 = self._PROT.atomdicfx[(CON.residue2, pseudo)].atomnumber
                            continue #29.1.99 always get the first hit


    def ReadOldListPeaksProt(self, listfile, peaksfile, protfile, het1, pro1, het2, pro2):
        """
        insert a spectrum, reading XEASY .peaks, .prot and ARIA's old .list files
        the data from the old .list file are complemented with the data from .peaks
        and .prot
        """
        #some messages:
        print 'reading the files:'
        print '  ' + listfile
        print '  ' + peaksfile
        print '  ' + protfile

        #for XEASY I need:
        from Aria.DataIO import ReadXeasy
        
        #internal XEASY data stores:
        self._ASSIGN = ReadXeasy.XeasyAssign() #internal store of XEASY .assign data
        self._PEAKS = ReadXeasy.XeasyPeaks()   #internal store of XEASY .peaks data
        self._PROT = ReadXeasy.XeasyProt()     #internal store of XEASY .prot data
        
        #check if the files exist:
        _DoesFileExist(listfile)        
        _DoesFileExist(peaksfile)
        _DoesFileExist(protfile)
        
        #for the documentation:
        self.fileNames.append(listfile)
        self.fileNames.append(peaksfile)
        self.fileNames.append(protfile)
        
        #read the old .list file:
        self._ReadOldList(listfile)

        #read .peaks and .prot:
        self._PEAKS.ReadPeaks(peaksfile)
        self._PROT.ReadProt(protfile)

        #get the dimensionality from the .peaks file:
        self.dimension = self._PEAKS.dimension
        
        #update the data with the .peaks and .prot data:
        for PEAK in self.peakslist:
            for CON in PEAK.contributions:
                #get the heteronucleus frequency:
                if het1 != 'N':
                    CON.h1ppm = getattr(self._PEAKS.peaksdic[CON.peakNumber], 'w' +\
                                        het1)
                if het2 != 'N':
                    CON.h2ppm = getattr(self._PEAKS.peaksdic[CON.peakNumber], 'w' +\
                                        het2)
                    
                #get the peakType:
                CON.peakType = self._PEAKS.peaksdic[CON.peakNumber].peakType

                #get the missing heteronucleus atomnumbers from the .peaks file:
                if het1 != 'N':
                    CON.atomnumberh1 = getattr(self._PEAKS.peaksdic[CON.peakNumber], 'atomnumber' + het1)
                    if CON.atomnumberh1 == '0':
                        CON.atomnumberh1 = None
                if het2 != 'N':
                    CON.atomnumberh2 = getattr(self._PEAKS.peaksdic[CON.peakNumber], 'atomnumber' + het2)
                    if CON.atomnumberh2 == '0':
                        CON.atomnumberh2 = None
                
                #get the missing proton atomnumbers, look up the atoms in the
                #.prot file, otherwise try pseudoatoms:
                #pro1:
                if self._PROT.atomdicfa.has_key((CON.residue1, CON.atomnamep1)): #normal case
                    CON.atomnumberp1 = self._PROT.atomdicfa[(CON.residue1, CON.atomnamep1)].atomnumber
                else: #now the pseudoatom case
                    pseudos = PseudoAtom.Atom2Pseudo(CON.atomnamep1[0], CON.aa1)
                    for pseudo in pseudos:
                        if pseudo in self._PROT.atomdicre[CON.residue1]:
                            CON.atomnumberp1 = self._PROT.atomdicfx[(CON.residue1, pseudo)].atomnumber
                            continue #29.1.99 always get the first hit
                #pro2:
                if self._PROT.atomdicfa.has_key((CON.residue2, CON.atomnamep2)): #normal case
                    CON.atomnumberp2 = self._PROT.atomdicfa[(CON.residue2, CON.atomnamep2)].atomnumber
                else: #now the pseudoatom case
                    pseudos = PseudoAtom.Atom2Pseudo(CON.atomnamep2[0], CON.aa2)
                    for pseudo in pseudos:
                        if pseudo in self._PROT.atomdicre[CON.residue2]:
                            CON.atomnumberp2 = self._PROT.atomdicfx[(CON.residue2, pseudo)].atomnumber
                            continue #29.1.99 always get the first hit

                #get the missing heteronucleus atomnames from the .prot file:
                if het1 != 'N' and self._PROT.atomdican.has_key(CON.atomnumberh1):
                    CON.atomnameh1 = (self._PROT.atomdican[CON.atomnumberh1].xeasyatomname,)
                if het2 != 'N' and self._PROT.atomdican.has_key(CON.atomnumberh2):
                    CON.atomnameh2 = (self._PROT.atomdican[CON.atomnumberh2].xeasyatomname,)

    
    def ReadPeaksProt(self, peaksfile, protfile, het1, pro1, het2, pro2):
        """
        reads Xeasy .peaks and .prot files
        
        I use the same convention as in the .html file for XEASY:
        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)
        """
        #some messages:
        print 'reading the XEASY files:'
        print '  ' + peaksfile
        print '  ' + protfile

        #check if the files exist:
        _DoesFileExist(peaksfile)
        _DoesFileExist(protfile)
        
        #for XEASY I need:
        from Aria.DataIO import ReadXeasy
        
        #internal XEASY data stores:
        self._ASSIGN = ReadXeasy.XeasyAssign() #internal store of XEASY .assign data
        self._PEAKS = ReadXeasy.XeasyPeaks()   #internal store of XEASY .peaks data
        self._PROT = ReadXeasy.XeasyProt()     #internal store of XEASY .prot data
        
        #for the documentation:
        self.fileNames.append(peaksfile)
        self.fileNames.append(protfile)
        
        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read in .peaks and .prot:
        self._PEAKS.ReadPeaks(peaksfile)
        self._PROT.ReadProt(protfile)

        #get the dimensionality from the .peaks file:
        self.dimension = self._PEAKS.dimension
        
        #put all the information together (like described above):
        #start with the peaks from the .peaks file:
        for EACHP in self._PEAKS.peakslist:
            NEWP = NoeContribution()
            NEWP.peakNumber = EACHP.peakNumber

            #get the right columns for the atomnumbers (0 => None):
            if het1 != 'N':
                NEWP.atomnumberh1 = getattr(EACHP, 'atomnumber' + het1)
                if NEWP.atomnumberh1 == '0':
                    NEWP.atomnumberh1 = None
            if pro1 != 'N':
                NEWP.atomnumberp1 = getattr(EACHP, 'atomnumber' + pro1)
                if NEWP.atomnumberp1 == '0':
                    NEWP.atomnumberp1 = None
            if het2 != 'N':
                NEWP.atomnumberh2 = getattr(EACHP, 'atomnumber' + het2)
                if NEWP.atomnumberh2 == '0':
                    NEWP.atomnumberh2 = None
            if pro2 != 'N':
                NEWP.atomnumberp2 = getattr(EACHP, 'atomnumber' + pro2)
                if NEWP.atomnumberp2 == '0':
                    NEWP.atomnumberp2 = None
            
            NEWP.peakType = EACHP.peakType
            NEWP.volume = EACHP.volume
            NEWP.volumeError = EACHP.volumeError
            
            #get the right columns for the frequencies:
            if het1 != 'N':
                NEWP.h1ppm = getattr(EACHP, 'w' + het1)
            if pro1 != 'N':
                NEWP.p1ppm = getattr(EACHP, 'w' + pro1)
            if het2 != 'N':
                NEWP.h2ppm = getattr(EACHP, 'w' + het2)
            if pro2 != 'N':
                NEWP.p2ppm = getattr(EACHP, 'w' + pro2)
            
            #now copy this contribution to the peak:
            WHOLEP = NoePeak()
            WHOLEP.AddContribution(NEWP)
            
            #now copy the whole peak to the self.peakslist attribute:
            self.AddPeak(WHOLEP)
            
        #now get the assigned shifts, shifterrors, atomnames and residue numbers
        #from .prot, loop through the whole NoeList:
        for EPEAK in self.peakslist:
            for ECON in EPEAK.contributions:
                if self._PROT.atomdican.has_key(ECON.atomnumberh1):
                    ECON.ah1ppm = self._PROT.atomdican[ECON.atomnumberh1].shift
                    ECON.dah1ppm = self._PROT.atomdican[ECON.atomnumberh1].shifterror
                    ECON.atomnameh1 = self._PROT.atomdican[ECON.atomnumberh1].ariaatomname
                    ECON.residue1 = self._PROT.atomdican[ECON.atomnumberh1].fragmentnumber
                else:
                    if not ((ECON.atomnumberh1 == None) or (ECON.atomnumberh1 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberh1, 'not found in .prot'
                if self._PROT.atomdican.has_key(ECON.atomnumberp1):
                    ECON.ap1ppm = self._PROT.atomdican[ECON.atomnumberp1].shift
                    ECON.dap1ppm = self._PROT.atomdican[ECON.atomnumberp1].shifterror
                    ECON.atomnamep1 = self._PROT.atomdican[ECON.atomnumberp1].ariaatomname
                    ECON.residue1 = self._PROT.atomdican[ECON.atomnumberp1].fragmentnumber
                else:
                    if not ((ECON.atomnumberp1 == None) or (ECON.atomnumberp1 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberp1, 'not found in .prot'
                if self._PROT.atomdican.has_key(ECON.atomnumberh2):
                    ECON.ah2ppm = self._PROT.atomdican[ECON.atomnumberh2].shift
                    ECON.dah2ppm = self._PROT.atomdican[ECON.atomnumberh2].shifterror
                    ECON.atomnameh2 = self._PROT.atomdican[ECON.atomnumberh2].ariaatomname
                    ECON.residue2 = self._PROT.atomdican[ECON.atomnumberh2].fragmentnumber
                else:
                    if not ((ECON.atomnumberh2 == None) or (ECON.atomnumberh2 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberh2, 'not found in .prot'
                if self._PROT.atomdican.has_key(ECON.atomnumberp2):
                    ECON.ap2ppm = self._PROT.atomdican[ECON.atomnumberp2].shift
                    ECON.dap2ppm = self._PROT.atomdican[ECON.atomnumberp2].shifterror
                    ECON.atomnamep2 = self._PROT.atomdican[ECON.atomnumberp2].ariaatomname
                    ECON.residue2 = self._PROT.atomdican[ECON.atomnumberp2].fragmentnumber
                else:
                    if not ((ECON.atomnumberp2 == None) or (ECON.atomnumberp2 == '0')):
                        print 'WARNING: atomnumber', ECON.atomnumberp2, 'not found in .prot'
        


    def ReadPippPck(self, noeFile, het1, pro1, het2, pro2, assign1, assign2):
        """
        reads an PIPP .PCK file
        
        I use the same convention as in the .html file for PIPP:
        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)

        assign1 and assign2 are used to determine which 'Assign'
        column belongs to residue 1 or residue 2:
        assign1 = '1' or '2'
        assign2 = '1' or '2'

        all columns are numbered from left to right
        """
        #some messages:
        print 'reading the PIPP .PCK file:'
        print '  ' + noeFile

        #check if the files exist:
        _DoesFileExist(noeFile)
        self.fileNames.append(noeFile)
        
        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read the general NOE file:
        noeHandle = TextFile.TextFile(noeFile)

        #unavailable data set to default:
        peakType = None
        segid1 = None
        segid2 = None
        volumeError = None
        distance = None
        distanceError = None
        lowerBound = None
        upperBound = None
        h1ppm = None
        h2ppm = None
        dh1ppm = None
        dp1ppm = None
        dh2ppm = None
        dp2ppm = None
        residue1 = None
        residue2 = None
        aa1 = None
        aa2 = None
        atomnameh1 = None
        atomnamep1 = None
        atomnameh2 = None
        atomnamep2 = None
            
        pipe = re.compile('|')
        pound = re.compile('#')
        semiColon = re.compile(';')
        startList = 0
        formatCol = []
        varsCol = []
        
        for eachLine in noeHandle:
            if pound.match(eachLine):
                continue
#            print eachLine, #test
            lineList = string.split(eachLine)
            if len(lineList) == 0:
                continue
            elif lineList[0] == 'FORMAT':
                formatCol = lineList[1:]
                continue
            elif lineList[0] == 'VARS':
                startList = 1
                varsCol = lineList[1:]
                continue
            elif lineList[0] == 'DATA' and lineList[1] == 'DIMCOUNT':
                self.dimension = lineList[2]
                continue
            elif startList == 0:
                continue

            #pkID (=peakNumber):
            peakNumber = lineList[varsCol.index('PkID')]
            
            #Sl.Z (=the Z slice the data was found on):
            sliceZ = lineList[varsCol.index('Sl.Z')]
            
            #X, Y, Z (=ppm 3D):
            ppmList = [0, lineList[varsCol.index('X')], lineList[varsCol.index('Y')], \
                       lineList[varsCol.index('Z')]]
            
            if het1 != 'N':
                exec 'h1ppm = ppmList[' + het1 + ']'
            if het2 != 'N':
                exec 'h2ppm = ppmList[' + het2 + ']'
            exec 'p1ppm = ppmList[' + pro1 + ']'
            exec 'p2ppm = ppmList[' + pro2 + ']'
            
            #Intensity:
            volume = lineList[varsCol.index('Intensity')]

            asterisk = re.compile('\*')
            
            #Assign1, Assign2 (not mandatory!):
            if len(varsCol) > 6:
                #get the whole entry (e.g. 'E68.CB,HB1|HB2'):
                get1 = semiColon.sub('', lineList[varsCol.index('Assign1')])
                try:
                    get2 = lineList[varsCol.index('Assign2')]
                except IndexError:
                    get2 = '****'
                if not asterisk.search(get1):
                    #for the first column (Assign1):
                    #split after the '.' (e.g. E68.CB,HB1|HB2' => ['E68', 'CB,HB1|HB2']):
                    get1List = string.split(get1, '.')
                    #get 3-letter code and residue number:
                    outS = 'aa' + assign1 + ' = AminoAcid.AminoAcid(get1List[0][0])[1];' +\
                           'residue' + assign1 + ' = get1List[0][1:];'
                    #split after the ',' (e.g. 'CB,HB1|HB2' => ['CB', 'HB1|HB2']):
                    atom1List = string.split(get1List[1], ',')
                    #form tuples to get all the possibilities within the same list
                    #e.g. ['CB', 'HB1|HB2']  => [('CB'), ('HB1', 'HB2')]:
                    atom1List[0] = tuple(string.split(atom1List[0], '|'))
                    if len(atom1List) == 2:
                        atom1List[1] = tuple(string.split(atom1List[1], '|'))
                    #now assign the stuff according to the specified columns:
                    outS = outS + 'atomnamep' + assign1 + ' = atom1List[-1];'
                    if len(atom1List) == 2:
                        outS = outS + 'atomnameh' + assign1 + ' = atom1List[-2]'
                    exec outS

                if not asterisk.search(get2):
                    #for the second column (Assign2):
                    #split after the '.' (e.g. E68.CB,HB1|HB2' => ['E68', 'CB,HB1|HB2']):
                    get2List = string.split(get2, '.')
                    #get 3-letter code and residue number:
                    outS = 'aa' + assign2 + ' = AminoAcid.AminoAcid(get2List[0][0])[1];' +\
                           'residue' + assign2 + ' = get2List[0][1:];'
                    #split after the ',' (e.g. 'CB,HB1|HB2' => ['CB', 'HB1|HB2']):
                    atom2List = string.split(get2List[1], ',')
                    #form tuples to get all the possibilities within the same list
                    #e.g. ['CB', 'HB1|HB2']  => [('CB'), ('HB1', 'HB2')]:
                    atom2List[0] = tuple(string.split(atom2List[0], '|'))
                    if len(atom2List) == 2:
                        atom2List[1] = tuple(string.split(atom2List[1], '|'))
                    #now assign the stuff according to the specified columns:
                    outS = outS + 'atomnamep' + assign2 + ' = atom2List[-1];'
                    if len(atom2List) == 2:
                        outS = outS + 'atomnameh' + assign2 + ' = atom2List[-2]'
                    exec outS
                    
            #put it in a contribution instance:
            CONT = NoeContribution(peakNumber = peakNumber,\
                                   peakType = peakType,\
                                   residue1 = residue1,\
                                   aa1 = aa1,\
                                   segid1 = segid1,\
                                   atomnameh1 = atomnameh1,\
                                   atomnamep1 = atomnamep1,\
                                   residue2 = residue2,\
                                   aa2 = aa2,\
                                   segid2 = segid2,\
                                   atomnameh2 = atomnameh2,\
                                   atomnamep2 = atomnamep2,\
                                   volume = volume,\
                                   volumeError = volumeError,\
                                   distanceAve = distance,\
                                   distanceStd = distanceError,\
                                   lowerBound = lowerBound,\
                                   upperBound = upperBound,\
                                   h1ppm = h1ppm,\
                                   dh1ppm = dh1ppm,\
                                   p1ppm = p1ppm,\
                                   dp1ppm = dp1ppm,\
                                   h2ppm = h2ppm,\
                                   dh2ppm = dh2ppm,\
                                   p2ppm = p2ppm,\
                                   dp2ppm = dp2ppm)
            
            #create a peak instance and add contribution:
            NOE = NoePeak()
            NOE.AddContribution(CONT)

            #add peak to the NOE list:
            self.AddPeak(NOE)

        
    def ReadRegine(self, fileName):
        """
        reads an Regine 2D, 3D or 4D peaks file

        the number of columns specifies the dimensionality
        
        2D format: whitespace separated fields
                   peakNumber ppmProt1 ppmProt2 Intensity Volume
    
        3D format: whitespace separated fields
                   peakNumber ppmHet1 ppmProt1 ppmProt2 Intensity Volume
    
        4D format: whitespace separated fields
                   peakNumber ppmHet1 ppmProt1 ppmHet2 ppmProt2 Intensity Volume

        2D example (5 columns):
        1      8.535    5.528     13449000    310650000
        2      8.534    1.953     16301000    385720000
        3      9.405    5.329      7526900    160030000

        3D example (6 columns):
        1  117.566    8.535    5.528     13449000    310650000
        2  117.582    8.534    1.953     16301000    385720000
        3  123.610    9.405    5.329      7526900    160030000

        4D example (7 columns):
        1  117.566    8.535    51.435   5.528     13449000    310650000
        2  117.582    8.534    45.235   1.953     16301000    385720000
        3  123.610    9.405    50.435   5.329      7526900    160030000
        

        We always use the volumes for the calibration in ARIA!
        Edit the code if you want to change this behaviour.
        """
        self.fileNames.append(fileName)

        #check, if file exists:
        _DoesFileExist(fileName)
        
        #important: delete all the existing peaks
        self._cleanUp()

        #fileHandle:
        fileHandle = TextFile.TextFile(fileName)

        getDim = 0
        lineCounter = 0
        for eachLine in fileHandle:
            lineCounter = lineCounter + 1
            lineList = string.split(eachLine)
            if len(lineList) < 4:
                continue
            #get the dimensionality from the first data line:
            if not getDim:
                if len(lineList) == 7:
                    getDim = 4
                elif len(lineList) == 6:
                    getDim = 3
                else:
                    getDim = 2
            #print warnings:
            if getDim != (len(lineList) - 3):
                print 'WARNING: number of columns not correct in shifts file!'
                print '         row no.', str(lineCounter), 'incorrect'
            #new peak instance:
            PEAK = NoePeak()
            #get the data:
            if getDim == 2:
                peakNumber = lineList[0]
                h1ppm = None
                p1ppm = lineList[1]
                h2ppm = None
                p2ppm = lineList[2]
                intensity = lineList[3]
                volume = lineList[4]
            elif getDim == 3:
                peakNumber = lineList[0]
                h1ppm = lineList[1]
                p1ppm = lineList[2]
                h2ppm = None
                p2ppm = lineList[3]
                intensity = lineList[4]
                volume = lineList[5]
            elif getDim == 4:
                peakNumber = lineList[0]
                h1ppm = lineList[1]
                p1ppm = lineList[2]
                h2ppm = lineList[3]
                p2ppm = lineList[4]
                intensity = lineList[5]
                volume = lineList[6]
            #put it into contribution instance:
            CONT = NoeContribution(spectrumName = None,\
                                   peakNumber = peakNumber,\
                                   testSet = None,\
                                   inWeight = None,\
                                   volume = volume,\
                                   volumeError = None,\
                                   intensity = intensity,\
                                   intensityError = None,\
                                   h1ppm = h1ppm,\
                                   dh1ppm = None,\
                                   p1ppm = p1ppm,\
                                   dp1ppm = None,\
                                   h2ppm = h2ppm,\
                                   dh2ppm = None,\
                                   p2ppm = p2ppm,\
                                   dp2ppm = None,\
                                   peakType = None,\
                                   fomAria = None,\
                                   curWeight = None,\
                                   lowerBound = None,\
                                   upperBound = None,\
                                   sumDistanceAve = None,\
                                   sumDistanceStd = None,\
                                   backVolumeAria = None,\
                                   backVolumeAriaStd = None,\
                                   allAssi = None,\
                                   nta = None,\
                                   contribution = None,\
                                   fomContribution = None,\
                                   distanceAve = None,\
                                   distanceStd = None,\
                                   backVolume = None,\
                                   backVolumeStd = None,\
                                   segid1 = None,\
                                   residue1 = None,\
                                   aa1 = None,\
                                   atomnameh1 = (None,),\
                                   assih1ppm = None,\
                                   assidh1ppm = None,\
                                   atomnamep1 = (None,),\
                                   assip1ppm = None,\
                                   assidp1ppm = None,\
                                   segid2 = None,\
                                   residue2 = None,\
                                   aa2 = None,\
                                   atomnameh2 = (None,),\
                                   assih2ppm = None,\
                                   assidh2ppm = None,\
                                   atomnamep2 = (None,),\
                                   assip2ppm = None,\
                                   assidp2ppm = None)

            #add contribution to peak:
            PEAK.AddContribution(CONT)
            
            #create a peak instance:
            NOE = NoePeak()
            
            #add contribution to NOE instance:
            NOE.AddContribution(CONT)

            #add peak to the NOE list:
            self.AddPeak(NOE)
    
    def ReadTbl(self, noeFile):
        """
        reads an ARIA .tbl file

        typical lines can look like that:

        ASSI {    2}
           (( segid "    " and resid 3    and name HN  ))
           (( segid "    " and resid 3    and name HA  ))
           2.600 0.800 0.800 peak 2 weight 0.10000E+01 volume 0.96210E+07 ppm1 7.997 ppm2 4.846

        assi ( attr store1 < 2.177 and attr store1 > 2.097 )
             ( attr store1 < 2.407 and attr store1 > 2.367 )
             6.0 0.1 0.1 peak 12 volume 0.000e+00 ppm1 2.137 ppm2 2.387

        assign (resid 1 and name ha# ) (resid 2 and name HN ) 2.55 0.75 0.75
        assign (resid 1 and name ha# ) (resid 2 and name HA ) 3.65 1.85 1.85

        the 'OR' is also understood.
        It's always a better idea to read the corresponding .list files because
        they contain more information than the .tbl files.
        The contents of the .list file are stored in the attribute _tblFileContent
        This is important for the free R factor refinement.
        """
        #some messages:
        print 'reading the ARIA .tbl file:'
        print '  ' + noeFile

        #check if the files exist:
        _DoesFileExist(noeFile)
        self.fileNames.append(noeFile)
        
        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read the general NOE file:
        bigString = DeleteCnsComments.GetString(noeFile)
        assiList = string.split(bigString, 'assign')[1:]
        if len(assiList) == 0:
            assiList = string.split(bigString, 'ASSI')[1:]
        if len(assiList) == 0:
            assiList = string.split(bigString, 'assi')[1:]
        if len(assiList) == 0:
            assiList = string.split(bigString, 'Assi')[1:]


        #IMPORTANT:
        #store the original content of the peak list in the attribute
        #_tblFileContent (e.g. important for free R factor refinement): 
        self._tblFileContent = assiList
        
        #compile some patterns:
        segidP = re.compile('segid\s*"(.{0,4})"\s*')
        residP = re.compile('resid\s*([0-9]+)[\s\)]+')
        nameP = re.compile('name\s*([\w!@#$%&*?|]+)\s*\)')
        store1LessP = re.compile('store1\s*<\s*(\d+)')
        store1GreatP = re.compile('store1\s*>\s*(\d+)')
        distancesP = re.compile('\)\s+([0-9.\-eE+]+)\s+([0-9.\-eE+]+)\s+([0-9.\-eE+]+)\s+')
        peakP = re.compile('\s+peak\s+([0-9.\-eE+]+)\s+')
        weightP = re.compile('\s+weight\s+([0-9.\-eE+]+)\s+')
        volumeP = re.compile('\s+volume\s+([0-9.\-eE+]+)\s+')
        ppm1P = re.compile('\s+ppm1\s+([0-9.\-eE+]+)\s+')
        ppm2P = re.compile('\s+ppm2\s+([0-9.\-eE+]+)\s+')

        #for creating peaknumbers (if there are no peaknumbers already):
        countPeaks = 1
        
        for eachAssi in assiList:
##             print eachAssi #test
            #unavailable data set to default:
            peakNumber = None
            peakType = None
            segid1 = None
            segid2 = None
            volume = None
            volumeError = None
            distance = None
            distanceError = None
            lowerBound = None
            upperBound = None
            h1ppm = None
            h2ppm = None
            p1ppm = None
            p2ppm = None
            dh1ppm = None
            dp1ppm = None
            dh2ppm = None
            dp2ppm = None
            residue1 = None
            residue2 = None
            aa1 = None
            aa2 = None
            atomnameh1 = (None,)
            atomnamep1 = (None,)
            atomnameh2 = (None,)
            atomnamep2 = (None,)
            
            #consider the 'OR':
            eachList = string.split(eachAssi, 'OR ')
            if len(eachList) == 1:
                eachList = string.split(eachAssi, 'or ')

            #create a peak instance:
            NOE = NoePeak()

            for eachCont in eachList:
                #mandatory:
                distancesL = distancesP.search(eachCont)
                if distancesL:
                    distance = distancesL.group(1)
                    upperBound = distancesL.group(2)
                    lowerBound = distancesL.group(3)
                peakM = peakP.search(eachCont)
                if peakM:
                    peakNumber = peakM.group(1)
                else:
                    peakNumber = str(countPeaks)
                    countPeaks = countPeaks + 1
                volumeM = volumeP.search(eachCont)
                if volumeM:
                    volume = volumeM.group(1)
                ppm1M = ppm1P.search(eachCont)
                if ppm1M:
                    p1ppm = ppm1M.group(1)
                ppm2M = ppm2P.search(eachCont)
                if ppm2M:
                    p2ppm = ppm2M.group(1)

                #others:
                segidL = segidP.findall(eachCont)
                if len(segidL) > 0:
                    segid1 = segidL[0]
                    segid2 = segidL[1]
                residL = residP.findall(eachCont)
                if len(residL) > 0:
                    residue1 = residL[0]
                    residue2 = residL[1]
                nameL  = nameP.findall(eachCont)
                if len(nameL) > 0:
                    atomnamep1 = (nameL[0],)
                    atomnamep2 = (nameL[1],)
                weightMatch = weightP.search(eachCont)
                if weightMatch:
                    weight = weightMatch.group(1)

                #put it in a contribution instance:
                CONT = NoeContribution(peakNumber = peakNumber,\
                                       peakType = peakType,\
                                       residue1 = residue1,\
                                       aa1 = aa1,\
                                       segid1 = segid1,\
                                       atomnameh1 = atomnameh1,\
                                       atomnamep1 = atomnamep1,\
                                       residue2 = residue2,\
                                       aa2 = aa2,\
                                       segid2 = segid2,\
                                       atomnameh2 = atomnameh2,\
                                       atomnamep2 = atomnamep2,\
                                       volume = volume,\
                                       volumeError = volumeError,\
                                       distanceAve = distance,\
                                       distanceStd = distanceError,\
                                       lowerBound = lowerBound,\
                                       upperBound = upperBound,\
                                       h1ppm = h1ppm,\
                                       dh1ppm = dh1ppm,\
                                       p1ppm = p1ppm,\
                                       dp1ppm = dp1ppm,\
                                       h2ppm = h2ppm,\
                                       dh2ppm = dh2ppm,\
                                       p2ppm = p2ppm,\
                                       dp2ppm = dp2ppm)

                #add contribution to NOE instance:
                NOE.AddContribution(CONT)

            #add peak to the NOE list:
            self.AddPeak(NOE)


    def ReadAureliaUserPeak(self):
        """
        bla
        """
        return
    

    def ReadAureliaTenFields(self, noeFile, pro1, pro2):
        """
        reads an AURELIA file with a format like that:
        hypo  peak_id  cluster  center(w1,w2)  match(xy,z)  intensity(abs)  name

     1     1      428     2.732, 10.138   1.00, 1.00      277766   40HB2 41HE1
     2     1       79     7.281, 10.138   1.00, 1.00      119324   
     3     1       71     7.431, 10.138   1.00, 1.00     1462283   41HZ2 41HE1
     4     1      497     2.367, 10.138   1.00, 1.00      324854   40HB3 41HE1.s  

     the .s symmetry identifiers are thrown away
     pro1: proton1 found in column '1' or '2'
     pro2: proton2 found in column '1' or '2'
     """
        #some messages:
        print 'reading the file:'
        print '  ' + noeFile

        #check if the files exist:
        _DoesFileExist(noeFile)
        self.fileNames.append(noeFile)
        
        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read the general NOE file:
        noeHandle = TextFile.TextFile(noeFile)

        #unavailable data set to default:
        peakType = None
        segid1 = None
        segid2 = None
        volumeError = None
        distance = None
        distanceError = None
        lowerBound = None
        upperBound = None
        h1ppm = None
        h2ppm = None
        dh1ppm = None
        dp1ppm = None
        dh2ppm = None
        dp2ppm = None
        residue1 = None
        residue2 = None
        aa1 = None
        aa2 = None
        atomnameh1 = None
        atomnamep1 = None
        atomnameh2 = None
        atomnamep2 = None
            
        startList = 0
        formatCol = []
        varsCol = []

        number = re.compile('(^\d+)(\S+)')
        
        for eachLine in noeHandle:
            if string.strip(eachLine) == '':
                continue
            lineList = string.split(eachLine)
            if len(lineList) < 8 or lineList[0] == 'hypo':
                continue
            
            peakNumber = lineList[0]
            peakId= lineList[1]
            cluster= lineList[2]
            outS = ''
            outS = 'p' + pro1 + 'ppm =  lineList[3][:-1];' #without trailing ','
            outS = outS + 'p' + pro2 + 'ppm = lineList[4]'
            exec outS
            matchxy = lineList[5][:-1]  #without trailing ','
            matchz = lineList[6]
            volume = lineList[7]

            #defaults for the atomnames:
            atomnamep1 = (None,)
            atomnamep2 = (None,)
            if len(lineList) > 9:
                name1 = lineList[8]
                name2 = lineList[9]
                if name1[-2:] == '.s': name1 = name1[:-2]
                if name2[-2:] == '.s': name2 = name2[:-2]
                #get the residue numbers and atomname tuples:
                outS = 'getNumber1 = number.search(name' + pro1 + ');'
                outS = outS + 'getNumber2 = number.search(name' + pro2 + ')'
                exec outS
                residue1 = getNumber1.group(1)
                residue2 = getNumber2.group(1)
                atomnamep1 = PseudoAtom.Pseudo2Tuple(getNumber1.group(2))
                atomnamep2 = PseudoAtom.Pseudo2Tuple(getNumber2.group(2))
                
            #put it in a contribution instance:
            CONT = NoeContribution(peakNumber = peakNumber,\
                                   peakType = peakType,\
                                   residue1 = residue1,\
                                   aa1 = aa1,\
                                   segid1 = segid1,\
                                   atomnameh1 = atomnameh1,\
                                   atomnamep1 = atomnamep1,\
                                   residue2 = residue2,\
                                   aa2 = aa2,\
                                   segid2 = segid2,\
                                   atomnameh2 = atomnameh2,\
                                   atomnamep2 = atomnamep2,\
                                   volume = volume,\
                                   volumeError = volumeError,\
                                   distanceAve = distance,\
                                   distanceStd = distanceError,\
                                   lowerBound = lowerBound,\
                                   upperBound = upperBound,\
                                   h1ppm = h1ppm,\
                                   dh1ppm = dh1ppm,\
                                   p1ppm = p1ppm,\
                                   dp1ppm = dp1ppm,\
                                   h2ppm = h2ppm,\
                                   dh2ppm = dh2ppm,\
                                   p2ppm = p2ppm,\
                                   dp2ppm = dp2ppm)
            
            #create a peak instance and add contribution:
            NOE = NoePeak()
            NOE.AddContribution(CONT)

            #add peak to the NOE list:
            self.AddPeak(NOE)
            

    def ReadLolUpl(self, lolFileName, uplFileName):
        """
        reads DYANA .lol and .upl files

        ...NOT YET IMPLEMENTED!

        
        """
        #some messages:
        print 'reading the ANSIG general NOE file:'
        print '  ' + noefile

        #check if the files exist:
        _DoesFileExist(noefile)
        self.fileNames.append(noefile)
        
        #important - delete all the existing peaks:
        self._cleanUp()
        
        #read the .upl file:
        noeHandle = TextFile.TextFile(noefile)

        print 'NOT YET IMPLEMENTED!'
   
        
    def RemovePeak(self, spectrumName, peakNumber):
        """
        removes the whole peakobject for the peak with a given peakNumber
        """
        if self.peaksdic.has_key(spectrumName, peakNumber):
            #get the object from the dictionary:
            whichobject = self.peaksdic[spectrumName, peakNumber]
            #remove it from the list and the dictionary:
            self.peakslist.remove(whichobject)
            del(self.peaksdic[(spectrumName, peakNumber)])
        
    
    def Sort(self):
        """
        sorts all the data:   1. after spectrumName
                              2. after peakNumber
        """
        allLists = ['peakslist',
                    'intraList',
                    'seqList',
                    'mediumList',
                    'longList',
                    'unassignedList',
                    'ambiguousList']
        for eachList in allLists:
            getattr(self, eachList).sort(Comparisons.CmpComposite(Comparisons.CmpAttr('spectrumName', 0),\
                                                                  Comparisons.CmpAttr('peakNumber', 1)))

        
    def Stdout(self):
        """writes the attributes to stdout in the new .list format"""
        print 'name:', self.name, '\ndimension:', self.dimension,\
              '\ncomment:', self.comment
        print 'p spec pnum test WIn volu volE int intE h1ppm dh1ppm p1ppm dp1ppm h2ppm dh2ppm p2ppm dp2ppm'
        print 'a type fom Wcur low up SDA SDS bVol bVolS all nta'
        print 'c con fom dAve dStd bVol bVolS seg1 res1 aa1 Nh1 ah1ppm adh1ppm Np1 ap1ppm adp1ppm seg2 res2 aa2 Nh2 ah2ppm adh2ppm Np2 ap2ppm adp2ppm'
        for PEAK in self.peakslist:
            forTheFirstCon = 1
            for CON in PEAK.contributions:
                if forTheFirstCon:
                    #the 'p' line:
                    print 'p',
                    for each in (CON.spectrumName, CON.peakNumber,\
                                 CON.testSet, CON.inWeight,\
                                 CON.volume, CON.volumeError,\
                                 CON.intensity, CON.intensityError,\
                                 CON.h1ppm,  CON.dh1ppm,\
                                 CON.p1ppm, CON.dp1ppm,\
                                 CON.h2ppm, CON.dh2ppm,\
                                 CON.p2ppm, CON.dp2ppm):
                        if each == None:
                            print '-',
                        elif each == (None,):
                            print ('-',),
                        else:
                            print each,
                    print ''
                    #the 'a' line:
                    print 'a',
                    for each in (CON.peakType, CON.fomAria,\
                                 CON.curWeight, CON.lowerBound,\
                                 CON.upperBound, CON.sumDistanceAve,\
                                 CON.sumDistanceStd, CON.backVolume,\
                                 CON.backVolumeStd, CON.allAssi,\
                                 CON.nta):
                        if each == None:
                            print '-',
                        elif each == (None,):
                            print ('-',),
                        else:
                            print each,
                    print ''
                #the 'c' line:
                forTheFirstCon = 0
                print 'c',
                for each in (CON.contribution, CON.fomContribution,\
                             CON.distanceAve, CON.distanceStd,\
                             CON.backVolume, CON.backVolumeStd,\
                             CON.segid1, CON.residue1,\
                             CON.aa1, CON.atomnameh1,\
                             CON.assih1ppm, CON.assidh1ppm,\
                             CON.atomnamep1, CON.assip1ppm,\
                             CON.assidp1ppm, CON.segid2,\
                             CON.residue2, CON.aa2,\
                             CON.atomnameh2, CON.assih2ppm,\
                             CON.assidh2ppm, CON.atomnamep2,\
                             CON.assip2ppm, CON.assidp2ppm):
##                              CON.atomnumberh1, CON.atomnumberp1,\
##                              CON.atomnumberh2, CON.atomnumberp2):
                    if each == None:
                        print '-',
                    elif each == (None,):
                        print ('-',),
                    else:
                        print each,
                print ''
    
    def WriteAnsigNoe(self, fileName, het1, pro1, het2, pro2):
        """
        writes an ANSIG general NOE file
        
        I use the same convention as in the .html file for ANSIG:
        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)):
        """
        noeHandle = TextFile.TextFile(fileName, 'w')
        print 'writing  an ANSIG general NOE file:\n  ', fileName
        
        if len(self.peakslist) == 0:
            print 'peaklist empty. WriteAnsigNoe method aborted.'
            return
        
        #the first line contains only comments:
        noeHandle.write('! ANSIG v3.3 general format NOE file\n')

        #the output format as explained in the ANSIG manual:
        format2 = FortranFormat.FortranFormat('I1,A1,3A4,F8.3,3A4,F8.3,I6,A12,F6.1,2E11.4')
        format3 = FortranFormat.FortranFormat('I1,A1,3A4,F8.3,3A4,F8.3,3A4,F8.3,I6,A12,F6.1,2E11.4')
        format4 = FortranFormat.FortranFormat('I1,A1,3A4,F8.3,3A4,F8.3,3A4,F8.3,3A4,F8.3,I6,A12,F6.1,2E11.4')

        #order the dimensions with a list of tuples called outlist:
        outlist = range(5)  #zero-element is not used
        
        for PEAK in self.peakslist:
            for CON in PEAK.contributions:
                #convert 3-lettercode from XXX to Xxx and create outlist:
                if het1 != 'N':
                    outlist[string.atoi(het1)] = (CON.residue1, CON.aa1[:1] + \
                                                  string.lowerBound(CON.aa1[1:3]), \
                                                  CON.atomnameh1, CON.h1ppm)
                outlist[string.atoi(pro1)] = (CON.residue1, CON.aa1[:1] + \
                                              string.lowerBound(CON.aa1[1:3]), \
                                              CON.atomnamep1, CON.p1ppm)
                if het2 != 'N':
                    outlist[string.atoi(het2)] = (CON.residue2, CON.aa2[:1] + \
                                                  string.lowerBound(CON.aa2[1:3]), \
                                                  CON.atomnameh2, CON.h2ppm)
                outlist[string.atoi(pro2)] = (CON.residue2, CON.aa2[:1] + \
                                              string.lowerBound(CON.aa2[1:3]), \
                                              CON.atomnamep2, CON.p2ppm)

                #write the stuff with FortranFormat:
                #set some default values if field is None:
                if CON.peakNumber == None:
                    outpeakNumber = 0
                else:
                    outpeakNumber = string.atoi(CON.peakNumber)
                if CON.mixtime == None:
                    outmixtime = 0.0
                else:
                    outmixtime = string.atof(CON.mixtime)
                if CON.volume == None:
                    outvolume = 0.0
                else:
                    outvolume = string.atof(CON.volume)
                if CON.relativeint == None:
                    outrelativeint = 0.0
                else:
                    outrelativeint = string.atof(CON.relativeint)
                
                if self.dimension == 2:
                    #use attributes spectrumName, mixtime, relative,
                    #take the first atomname from the tuple:
                    line = FortranFormat.FortranLine([self.dimension, ' ',\
                                                      outlist[1][0], outlist[1][1],\
                                                      outlist[1][2][0], string.atof(outlist[1][3]),\
                                                      outlist[2][0], outlist[2][1],\
                                                      outlist[2][2][0], string.atof(outlist[2][3]),\
                                                      outpeakNumber, CON.spectrumName,\
                                                      outmixtime, outvolume,\
                                                      outrelativeint], format2)
                elif self.dimension == 3:
                    #use attributes spectrumName, mixtime, relative,
                    #take the first atomname from the tuple:
                    line = FortranFormat.FortranLine([self.dimension, ' ',\
                                                      outlist[1][0], outlist[1][1],\
                                                      outlist[1][2][0], string.atof(outlist[1][3]),\
                                                      outlist[2][0], outlist[2][1],\
                                                      outlist[2][2][0], string.atof(outlist[2][3]),\
                                                      outlist[3][0], outlist[3][1],\
                                                      outlist[3][2][0], string.atof(outlist[3][3]),\
                                                      outpeakNumber, CON.spectrumName,\
                                                      outmixtime, outvolume,\
                                                      outrelativeint], format3)
                elif self.dimension == 4:
                    #use attributes spectrumName, mixtime, relative,
                    #take the first atomname from the tuple:
                    line = FortranFormat.FortranLine([self.dimension, ' ',\
                                                      outlist[1][0], outlist[1][1],\
                                                      outlist[1][2][0], string.atof(outlist[1][3]),\
                                                      outlist[2][0], outlist[2][1],\
                                                      outlist[2][2][0], string.atof(outlist[2][3]),\
                                                      outlist[3][0], outlist[3][1],\
                                                      outlist[3][2][0], string.atof(outlist[3][3]),\
                                                      outlist[4][0], outlist[4][1],\
                                                      outlist[4][2][0], string.atof(outlist[4][3]),\
                                                      outpeakNumber, CON.spectrumName,\
                                                      outmixtime, outvolume,\
                                                      outrelativeint], format4)
                #write the stuff to disk:
                noeHandle.write(str(line) +'\n')
        noeHandle.close()
        

    def WriteAureliaUserPeak(self, fileName):
        """
        writes a list in AURELIA user peak list format
        """
        file = TextFile.TextFile(fileName, 'w')
        print 'writing the data in AURELIA user peak list format to', fileName
        print 'We always use the following format for the user peak lists:'
        print '  # 110.23 4.56 15.8 2.05 N HN CA HC 3 76 ABCD EFGH'
        print '  # h1ppm p1ppm h2ppm p2ppm h1 p1 h2 p2 res1 res2 segid1 segid2'
        print '  segid should contain 4 letters or 4 spaces'
        print '  other formats can not be read in by this method!'
        if len(self.peakslist) == 0:
            print 'peaklist empty. WriteList method aborted.'
            return
        for PEAK in self.peakslist:
            for CON in PEAK.contributions:
                file.write('# ' + CON.h1ppm + ' ' + CON.p1ppm + ' ' +\
                           CON.h2ppm + ' ' + CON.p2ppm + ' ' +\
                           CON.atomnameh1[0] + ' ' + CON.atomnamep1[0] + ' ' +\
                           CON.atomnameh2[0] + ' ' + CON.atomnamep2[0] + ' ' +\
                           CON.residue1 + ' ' + CON.residue2 +\
                           CON.segid1 + ' ' + CON.segid2 + '\n')
        file.close()
        

    def WriteList(self, fileName):
        """
        writes a list in ARIA's .list format
        """
        file = TextFile.TextFile(fileName, 'w')
        print 'writing the data in .list format to', fileName
        if len(self.peakslist) == 0:
            print 'peaklist empty. WriteList method aborted.'
            return
        for PEAK in self.peakslist:
            for CON in PEAK.contributions:
                file.write(CON.peakNumber + ' ' + CON.peakType + ' ' +\
                           CON.residue1 + ' ' + CON.aa1 + ' ' +\
                           CON.segid1 + ' ' + CON.atomnameh1[0] + ' ' +\
                           CON.atomnamep1[0] + ' ' +\
                           CON.residue2 + ' ' + CON.aa2 + ' ' +\
                           CON.segid2 + ' ' + CON.atomnameh2[0] + ' ' +\
                           CON.atomnamep2[0] + ' ' +\
                           CON.volume + ' ' + CON.volumeError + ' ' +\
                           CON.distanceAve + ' ' + CON.distanceStd + ' ' +\
                           CON.lowerBound + ' ' + CON.upperBound + ' ' +\
                           CON.h1ppm + ' ' + CON.dh1ppm + ' ' +\
                           CON.p1ppm + ' ' + CON.dp1ppm + ' ' +\
                           CON.h2ppm + ' ' + CON.dh2ppm + ' ' +\
                           CON.p2ppm + ' ' + CON.dp2ppm + ' ' +\
                           CON.weight + ' ' + CON.contribution + ' ' +\
                           CON.nta + '\n')
        file.close()


    def WriteTbl(self, fileName, ppmdhet1='0.5', ppmdpro1='0.02',\
                 ppmdhet2='0.5', ppmdpro2='0.02', whichpeaks='all'):
        """
        writes a list in ARIA's .tbl format

        you have to specify the deltas for the frequency windows:
        ppmdhet1
        ppmdpro1
        ppmdhet2
        ppmdpro2
        The given values are just default values. It's advisable
        to find the optimal values for each dataset.
        
        
        whichpeaks offers the choices:
        all         all peaks
        allw1       all peaks, set weight > 1
        allassiw1   all peaks, set weight > 1 for fully assigned peaks 
        onlyassi    only fully assigned peaks
        onlyassiw1  only fully assigned peaks, set weight > 1

        this method looks if the chemical shifts are existent.
        then the assi statement is written out
        """
        print 'writing the .tbl file', fileName
        tblhandle = TextFile.TextFile(fileName, 'w')
        tblhandle.write('! derived from the ' + str(self.dimension) +\
                        'D files:\n')
        for eachfile in self.fileNames:
            tblhandle.write('! ' + eachfile + '\n')
        for EACHP in self.peakslist:
            contCounter = 0
            for EACHC in EACHP.contributions:
                #set lowerBound and upper bounds:
                if EACHC.h1ppm != None:
                    lowerBoundh1 = str(float(EACHC.h1ppm) - float(ppmdhet1))
                    higherh1 = str(float(EACHC.h1ppm) + float(ppmdhet1))
                if EACHC.p1ppm != None:
                    lowerBoundp1 = str(float(EACHC.p1ppm) - float(ppmdpro1))
                    higherp1 = str(float(EACHC.p1ppm) + float(ppmdpro1))
                if EACHC.h2ppm != None:   
                     lowerBoundh2 = str(float(EACHC.h2ppm) - float(ppmdhet2))
                     higherh2 = str(float(EACHC.h2ppm) + float(ppmdhet2))
                if EACHC.p2ppm != None:   
                    lowerBoundp2 = str(float(EACHC.p2ppm) - float(ppmdpro2))
                    higherp2 = str(float(EACHC.p2ppm) + float(ppmdpro2))
                
                #distance is always set to 6 by default:
                dist = '6.0'
                if EACHC.volume == 0.0:
                    errp = '0.0'
                    errm = '6.0'
                else:
                    errp = '0.1'
                    errm = '0.1'
                #set the segids (with keyword, braces and 'and') if they exist:
                segid1 = ''
                segid2 = ''
                outsegid1 = ''
                outsegid2 = ''
                if EACHC.segid1 != None:
                    segid1 = ' and segid "' + EACHC.segid1 + '"'
                if EACHC.segid2 != None:
                    segid2 = ' and segid "' + EACHC.segid2 + '"'

                #either 'ASSI' or 'OR':
                assiOr = 'assi'
                if contCounter > 0:
                    assiOr = '  or'
                    
                #check all the cases for 2D, 3D and 4D:
                #4D case:
                if (EACHC.h1ppm != None) and\
                   (EACHC.p1ppm != None) and\
                   (EACHC.h2ppm != None) and\
                   (EACHC.p2ppm != None):
                    #look if it's fully assigned:
                    if EACHC.atomnameh1 != (None,) and\
                       EACHC.atomnamep1 != (None,) and\
                       EACHC.atomnameh2 != (None,) and\
                       EACHC.atomnamep2 != (None,) and\
                       EACHC.aa1 != None and\
                       EACHC.aa2 != None:
                        #for the fully assigned peaks:
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassi':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassiw1':
                            weight = ' weight 1.1'                            
                        #write all the stuff to disk (assigned format):
                        tblhandle.write(assiOr + ' ( resid ' + EACHC.residue1 + \
                                        ' and name ' + EACHC.atomnamep1[0] + \
                                        segid1 + ' )\n')
                        tblhandle.write('     ( resid ' + EACHC.residue2 + \
                                        ' and name ' + EACHC.atomnamep2[0] + \
                                        segid2 + ' )\n')
                        if conCounter > 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume +\
                                            ' hpm1 ' + EACHC.h1ppm +\
                                            ' ppm1 ' + EACHC.p1ppm +\
                                            ' hpm2 ' + EACHC.h2ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                    else:
                        #for the other peaks (not fully assigned):
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ''
                        elif whichpeaks == 'onlyassi':
                            continue  #don't use this contribution!
                        elif whichpeaks == 'onlyassiw1':
                            continue  #don't use this contribution!
                        #write all the stuff to disk (frequency window format):
                        tblhandle.write(assiOr + ' (( attr store1 < ' + higherp1 + \
                                        ' and attr store1 > ' + lowerBoundp1 + ' )\n')
                        tblhandle.write('     and bondedto ( attr store1 < ' + \
                                        higherh1 + ' and attr store1 > ' + \
                                        lowerBoundh1 + ' ) ' + segid1 + ' )\n')
                        tblhandle.write('     (( attr store1 < ' + higherp2 + \
                                        ' and attr store1 > ' + lowerBoundp2 + ' )\n')
                        tblhandle.write('     and bondedto ( attr store1 < ' + \
                                        higherh2 + ' and attr store1 > ' + \
                                        lowerBoundh2 + ' ) ' + segid2 + ' )\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume +\
                                            ' hpm1 ' + EACHC.h1ppm +\
                                            ' ppm1 ' + EACHC.p1ppm +\
                                            ' hpm2 ' + EACHC.h2ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                
                #3D case, heteronucleus in het1:
                elif (EACHC.h1ppm != None) and\
                     (EACHC.p1ppm != None) and\
                     (EACHC.h2ppm == None) and\
                     (EACHC.p2ppm != None):
                    #look if it's fully assigned:
                    if EACHC.atomnameh1 != (None,) and\
                       EACHC.atomnamep1 != (None,) and\
                       EACHC.atomnamep2 != (None,) and\
                       EACHC.aa1 != None and\
                       EACHC.aa2 != None:
                        #for the fully assigned peaks:
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassi':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassiw1':
                            weight = ' weight 1.1'
                        #write all the stuff to disk (assigned format):
                        tblhandle.write(assiOr + ' ( resid ' + EACHC.residue1 + \
                                        ' and name ' + EACHC.atomnamep1[0] + \
                                        segid1 + ' )\n')
                        tblhandle.write('     ( resid ' + EACHC.residue2 + \
                                        ' and name ' + EACHC.atomnamep2[0] + \
                                        segid2 + ' )\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume +\
                                            ' hpm1 ' + EACHC.h1ppm +\
                                            ' ppm1 ' + EACHC.p1ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                    else:
                        #for the other peaks (not fully assigned):
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ''
                        elif whichpeaks == 'onlyassi':
                            continue  #don't use this contribution!
                        elif whichpeaks == 'onlyassiw1':
                            continue  #don't use this contribution!
                        #write all the stuff to disk (frequency window format):
                        tblhandle.write(assiOr + ' (( attr store1 < ' + higherp1 + \
                                        ' and attr store1 > ' + lowerBoundp1 + ' )\n')
                        tblhandle.write('     and bondedto ( attr store1 < ' + \
                                        higherh1 + ' and attr store1 > ' + \
                                        lowerBoundh1 + ' ) ' + segid1 + ' )\n')
                        tblhandle.write('     ( attr store1 < ' + higherp2 + \
                                        ' and attr store1 > ' + lowerBoundp2 + ' )' + segid2 + '\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume +\
                                            ' hpm1 ' + EACHC.h1ppm +\
                                            ' ppm1 ' + EACHC.p1ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                #3D case, heteronucleus in het2:
                elif (EACHC.h1ppm == None) and\
                     (EACHC.p1ppm != None) and\
                     (EACHC.h2ppm != None) and\
                     (EACHC.p2ppm != None):
                    #look if it's fully assigned:
                    if EACHC.atomnamep1 != (None,) and\
                       EACHC.atomnameh2 != (None,) and\
                       EACHC.atomnamep2 != (None,) and\
                       EACHC.aa1 != None and\
                       EACHC.aa2 != None:
                        #for the fully assigned peaks:
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassi':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassiw1':
                            weight = ' weight 1.1'
                        #write all the stuff to disk (assigned format):
                        tblhandle.write(assiOr + ' ( resid ' + EACHC.residue1 + \
                                        ' and name ' + EACHC.atomnamep1[0] + \
                                        segid1 + ' )\n')
                        tblhandle.write('     ( resid ' + EACHC.residue2 + \
                                        ' and name ' + EACHC.atomnamep2[0] + \
                                        segid2 + ' )\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume + ' ppm1 ' + EACHC.p1ppm +\
                                            ' hpm2 ' + EACHC.h2ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                    else:
                        #for the other peaks (not fully assigned):
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ''
                        elif whichpeaks == 'onlyassi':
                            continue  #don't use this contribution!
                        elif whichpeaks == 'onlyassiw1':
                            continue  #don't use this contribution!
                        #write all the stuff to disk (frequency window format):
                        tblhandle.write('     ( attr store1 < ' + higherp1 + \
                                        ' and attr store1 > ' + lowerBoundp1 + ' )' + segid1 + '\n')
                        tblhandle.write(assiOr + ' (( attr store1 < ' + higherp2 + \
                                        ' and attr store1 > ' + lowerBoundp2 + ' )\n')
                        tblhandle.write('     and bondedto ( attr store1 < ' + \
                                        higherh2 + ' and attr store1 > ' + \
                                        lowerBoundh2 + ' ) ' + segid2 + ' )\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume + ' ppm1 ' + EACHC.p1ppm +\
                                            ' hpm2 ' + EACHC.h2ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                #2D case:
                elif (EACHC.h1ppm == None) and\
                     (EACHC.p1ppm != None) and\
                     (EACHC.h2ppm == None) and\
                     (EACHC.p2ppm != None):
                    #look if it's fully assigned:
                    if EACHC.atomnamep1 != (None,) and\
                       EACHC.atomnamep2 != (None,) and\
                       EACHC.aa1 != None and\
                       EACHC.aa2 != None:
                        #for the fully assigned peaks:
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassi':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'onlyassiw1':
                            weight = ' weight 1.1'
                        #write all the stuff to disk (assigned format):
##                        print EACHC.residue1,EACHC.atomnamep1[0], segid1 #test
                        tblhandle.write(assiOr + ' ( resid ' + EACHC.residue1 + \
                                        ' and name ' + EACHC.atomnamep1[0] + \
                                        segid1 + ' )\n')
                        tblhandle.write('     ( resid ' + EACHC.residue2 + \
                                        ' and name ' + EACHC.atomnamep2[0] + \
                                        segid2 + ' )\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume + ' ppm1 ' + EACHC.p1ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                    else:
                        #for the other peaks (not fully assigned):
                        if whichpeaks == 'all':
                            weight = ''
                        elif whichpeaks == 'allw1':
                            weight = ' weight 1.1'
                        elif whichpeaks == 'allassiw1':
                            weight = ''
                        elif whichpeaks == 'onlyassi':
                            continue  #don't use this contribution!
                        elif whichpeaks == 'onlyassiw1':
                            continue  #don't use this contribution!
                        #write all the stuff to disk (frequency window format):
                        tblhandle.write(assiOr + ' ( attr store1 < ' + higherp1 + \
                                        ' and attr store1 > ' + lowerBoundp1 + ' )' + segid1 + '\n')
                        tblhandle.write('     ( attr store1 < ' + higherp2 + \
                                        ' and attr store1 > ' + lowerBoundp2 + ' )' + segid2 + '\n')
                        if contCounter == 0:
                            tblhandle.write('     ' + dist + ' ' + errm + ' ' +\
                                            errp + ' peak ' + \
                                            EACHC.peakNumber + weight + ' volume ' +\
                                            EACHC.volume + ' ppm1 ' + EACHC.p1ppm +\
                                            ' ppm2 ' + EACHC.p2ppm + '\n')
                contCounter = contCounter + 1
        #closing file handles and bye:
        tblhandle.close()
        


    def WriteOldList(self, fileName):
        """
        writes a list in ARIA's old .list format
        """
        file = TextFile.TextFile(fileName, 'w')
        print 'writing the data in .list format to', fileName
        if len(self.peakslist) == 0:
            print 'peaklist empty. WriteList method aborted.'
            return
        for PEAK in self.peakslist:
            for CON in PEAK.contributions:
                if (CON.peakType == '3') or (CON.peakType == '4'):
                    sign = '-'
                else:
                    sign = '+'
                file.write(sign + ' ' + CON.peakNumber + ' ' +\
                           CON.residue1 + ' ' + CON.aa1 + ' ' +\
                           CON.atomnamep1[0] + ' ' +\
                           CON.residue2 + ' ' + CON.aa2 + ' ' +\
                           CON.atomnamep2[0] + ' ' + CON.distanceAve + ' ' +\
                           CON.volume + ' ' + CON.lowerBound + ' ' + CON.upperBound +\
                           CON.p1ppm + ' ' + CON.p2ppm + ' ' +\
                           CON.atomnumberp1 + ' ' + CON.atomnumberp2 + ' ' +\
                           CON.contribution + ' ' + CON.weight + ' ' +\
                           CON.nta + '\n')
        file.close()
        

    def WriteUplLol(self, fileroot):
        """
        writes out .upl and .lol files for MolMol
        pseudoatoms are not used, the atomnames should fit to the pdb files.
        important:
        I use the contribution to the peak as relative limit in these files!
        If more than one atomname is possible, all the possibilities are
        written out!
        
        accepted peaks:
          _amb+.upl
          _unamb+.upl
          _amb+.lol
          _unamb+.lol
        rejected peaks:
          _amb-.upl
          _unamb-.upl
          _amb-.lol
          _unamb-.lol

        e.g. WriteUplLol('/home/linge/vasp')
             will create vasp.upl and vasp.lol in /home/linge
        
        """
        uplAmbAccFile = fileroot + '_amb+.upl'
        lolAmbAccFile = fileroot + '_amb+.lol'
        uplUnambAccFile = fileroot + '_unamb+.upl'
        lolUnambAccFile = fileroot + '_unamb+.lol'
        uplAmbRejFile = fileroot + '_amb-.upl'
        lolAmbRejFile = fileroot + '_amb-.lol'
        uplUnambRejFile = fileroot + '_unamb-.upl'
        lolUnambRejFile = fileroot + '_unamb-.lol'
        print '  writing all data to the following MolMol files:\n   ',\
              uplAmbAccFile, '\n   ', lolAmbAccFile, '\n   ',\
              uplUnambAccFile, '\n   ', lolUnambAccFile, '\n   ',\
              uplAmbRejFile, '\n   ', lolAmbRejFile, '\n   ',\
              uplUnambRejFile, '\n   ', lolUnambRejFile
        uplAmbAccHandle = TextFile.TextFile(uplAmbAccFile, 'w')
        lolAmbAccHandle = TextFile.TextFile(lolAmbAccFile, 'w')
        uplUnambAccHandle = TextFile.TextFile(uplUnambAccFile, 'w')
        lolUnambAccHandle = TextFile.TextFile(lolUnambAccFile, 'w')
        uplAmbRejHandle = TextFile.TextFile(uplAmbRejFile, 'w')
        lolAmbRejHandle = TextFile.TextFile(lolAmbRejFile, 'w')
        uplUnambRejHandle = TextFile.TextFile(uplUnambRejFile, 'w')
        lolUnambRejHandle = TextFile.TextFile(lolUnambRejFile, 'w')
        
        for PEAK in self.peakslist:
##             if PEAK.contributions == []:  #test
##                 continue                  #test
            for CON in PEAK.contributions:
                #I have to loop over the proton atomname tuples:
                for each1 in CON.atomnamep1:
                    for each2 in CON.atomnamep2:
                        if CON.upperBound == None:
                            outupperBound = '6'  #default: 6 Angstrom
                        else:
                            outupperBound = CON.upperBound
                        if CON.lowerBound == None:
                            outlowerBound = '0'  #default: 0 Angstrom
                        else:
                            outlowerBound = CON.lowerBound
                        if CON.contribution == None:
                            outcontribution = '1'
                        else:
                            outcontribution = CON.contribution
                        if CON.residue1 == None:
                            outResidue1 = 0
                        else:
                            outResidue1 = string.atoi(CON.residue1)
                        if CON.residue2 == None:
                            outResidue2 = 0
                        else:
                            outResidue2 = string.atoi(CON.residue2)
                        if len(PEAK.contributions) == 1 and \
                           string.atoi(CON.peakType) < 3:
                            uplUnambAccHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                    (outResidue1,\
                                                     CON.aa1,\
                                                     each1,\
                                                     outResidue2,\
                                                     CON.aa2,\
                                                     each2,\
                                                     string.atof(outupperBound),\
                                                     string.atof(outcontribution)))
                            lolUnambAccHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                    (outResidue1,\
                                                     CON.aa1,\
                                                     each1,\
                                                     outResidue2,\
                                                     CON.aa2,\
                                                     each2,\
                                                     string.atof(outlowerBound),\
                                                     string.atof(outcontribution)))
                        if len(PEAK.contributions) > 1 and \
                           string.atoi(CON.peakType) < 3:
                            uplAmbAccHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                  (outResidue1,\
                                                   CON.aa1,\
                                                   each1,\
                                                   outResidue2,\
                                                   CON.aa2,\
                                                   each2,\
                                                   string.atof(outupperBound),\
                                                   string.atof(outcontribution)))
                            lolAmbAccHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                  (outResidue1,\
                                                   CON.aa1,\
                                                   each1,\
                                                   outResidue2,\
                                                   CON.aa2,\
                                                   each2,\
                                                   string.atof(outlowerBound),\
                                                   string.atof(outcontribution)))
                        if len(PEAK.contributions) == 1 and \
                           string.atoi(CON.peakType) > 2:
                            uplUnambRejHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                    (outResidue1,\
                                                     CON.aa1,\
                                                     each1,\
                                                     outResidue2,\
                                                     CON.aa2,\
                                                     each2,\
                                                     string.atof(outupperBound),\
                                                     string.atof(outcontribution)))
                            lolUnambRejHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                    (outResidue1,\
                                                     CON.aa1,\
                                                     each1,\
                                                     outResidue2,\
                                                     CON.aa2,\
                                                     each2,\
                                                     string.atof(outlowerBound),\
                                                     string.atof(outcontribution)))
                        if len(PEAK.contributions) > 1 and \
                           string.atoi(CON.peakType) > 2:
                            uplAmbRejHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                  (outResidue1,\
                                                   CON.aa1,\
                                                   each1,\
                                                   outResidue2,\
                                                   CON.aa2,\
                                                   each2,\
                                                   string.atof(outupperBound),\
                                                   string.atof(outcontribution)))
                            lolAmbRejHandle.write('%3i %-4s %-5s %3i %-4s %-7s %-5.2f %-5.2f \n' %\
                                                  (outResidue1,\
                                                   CON.aa1,\
                                                   each1,\
                                                   outResidue2,\
                                                   CON.aa2,\
                                                   each2,\
                                                   string.atof(outlowerBound),\
                                                   string.atof(outcontribution)))

        #close all the filehandles:
        uplAmbAccHandle.close()
        lolAmbAccHandle.close()
        uplUnambAccHandle.close()
        lolUnambAccHandle.close()
        uplAmbRejHandle.close()
        lolAmbRejHandle.close()
        uplUnambRejHandle.close()
        lolUnambRejHandle.close()


    def WriteXeasyAssignPeaks(self, fileroot, het1, pro1, het2, pro2):
        """
        writes out the XEASY .assign and .peaks files
        fileroot should be a name without any extensions
        (no .prot file is created, just copy the .prot file of the last iteration)
        
        I use the same convention as in the .html file for XEASY:
        het1 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro1 = '1', '2', '3', '4' or 'N' (column number or not used)
        het2 = '1', '2', '3', '4' or 'N' (column number or not used)
        pro2 = '1', '2', '3', '4' or 'N' (column number or not used)

        I use pro1 and pro2 for the matrix -> binary format conversion

        IMPORTANT: it is only possible to write out these files if the
                   atomnumbers have been read in from other XEASY files!
                   also the spectrum type, integration method must be
                   available!

        NOTE: the matrix is set up with the protons!
              more than 10 ambiguous assignments are not possible in XEASY
        """
##         #have a look if there are atomnumbers defined:
##         print 'checking the atomnumbers:',
##         print 'o.k.' #test - I have to work on that!
        
        #creating the fileNames from fileroot:
        peaksfile = fileroot + '.peaks'
        assignfile = fileroot + '.assign'

        #only for XEASY stuff I need:
        import Numeric
        from Aria.DataIO import ReadXeasy
        
        #message to stdout:
        print 'writing all data to the following XEASY files:\n  ',\
              peaksfile, '\n  ', assignfile
        
        #opening filehandles:
        peakshandle = TextFile.TextFile(peaksfile, 'w')
        assignhandle = TextFile.TextFile(assignfile, 'w')
        
        #write some comments (dimension and INAME) for .peaks:
        peakshandle.write('# Number of dimensions ' + str(self.dimension) + '\n')
        if self.dimension >= 1:
            peakshandle.write('# INAME 1 ' + self._iname1 + '\n')
        if self.dimension >= 2:
            peakshandle.write('# INAME 2 ' + self._iname2 + '\n')
        if self.dimension >= 3:
            peakshandle.write('# INAME 3 ' + self._iname3 + '\n')
        if self.dimension >= 4:
            peakshandle.write('# INAME 4 ' + self._iname4 + '\n')
            
        #write first line with the dimension for .assign:
        assignhandle.write(str(self.dimension) + '\n')
        
        #first loop over all the peaks, only one contribution possible in .peaks:
        for PEAK in self.peakslist:
            #for empty fields:
            if PEAK.contributions == []: 
                continue
                
            #get the assigned atom numbers from the first contribution:
            if het1 != 'N':
                outnumberh1 = PEAK.contributions[0].atomnumberh1
            outnumberp1 = PEAK.contributions[0].atomnumberp1
            if het2 != 'N':
                outnumberh2 = PEAK.contributions[0].atomnumberh2
            outnumberp2 = PEAK.contributions[0].atomnumberp2

            #looking whether the assigned atomnumbers are the same for every contribution,
            #because only one contribution is allowed in .peaks:
            for iii in range(len(PEAK.contributions) - 1):
                if het1 != 'N':
                    if PEAK.contributions[iii].atomnumberh1 == PEAK.contributions[iii+1].atomnumberh1:
                        outnumberh1 = PEAK.contributions[iii].atomnumberh1
                    else:
                        outnumberh1 = '0'   #default
                if PEAK.contributions[iii].atomnumberp1 == PEAK.contributions[iii+1].atomnumberp1:
                    outnumberp1 = PEAK.contributions[iii].atomnumberp1
                else:
                    outnumberp1 = '0'   #default
                if het2 != 'N':
                    if PEAK.contributions[iii].atomnumberh2 == PEAK.contributions[iii+1].atomnumberh2:
                        outnumberh2 = PEAK.contributions[iii].atomnumberh2
                    else:
                        outnumberh2 = '0'   #default
                if PEAK.contributions[iii].atomnumberp2 == PEAK.contributions[iii+1].atomnumberp2:
                    outnumberp2 = PEAK.contributions[iii].atomnumberp2
                else:
                    outnumberp2 = '0'   #default    

            #arrange the outnumbers in a list in the correct order (for the columns):
            outnumberlist = [0, None, None, None, None]  #zero-element is not used
            if het1 != 'N':
                outnumberlist[string.atoi(het1)] = outnumberh1
            outnumberlist[string.atoi(pro1)] = outnumberp1
            if het2 != 'N':
                outnumberlist[string.atoi(het2)] = outnumberh2
            outnumberlist[string.atoi(pro2)] = outnumberp2

            #arrange the ppms in a list in the correct order (for the columns):
            outppmlist = [0, None, None, None, None]  #zero-element is not used
            if het1 != 'N':
                outppmlist[string.atoi(het1)] = PEAK.contributions[0].assih1ppm  #12:09:99
            outppmlist[string.atoi(pro1)] = PEAK.contributions[0].assip1ppm      #12:09:99
            if het2 != 'N':
                outppmlist[string.atoi(het2)] = PEAK.contributions[0].assih2ppm  #12:09:99
            outppmlist[string.atoi(pro2)] = PEAK.contributions[0].assip2ppm      #12:09:99

            #replace None by 0 for every ppm in outppmlist, for atof():
            for www in range(0, 4):
                if None in outppmlist:
                    index = outppmlist.index(None)
                    outppmlist.remove(None)
                    outppmlist.insert(index, '0')
                else:
                    break
            
##             print outppmlist #test
            
            #replace None by 0 for volume and volumeError, for atof():
            outvolume = PEAK.contributions[0].volume
            outvolumeError = PEAK.contributions[0].volumeError
            if outvolume == None:
                outvolume = '0'
            if outvolumeError == None:
                outvolumeError = '0'
            if PEAK.contributions[0].peakNumber == None:
                outPeakNumber = 0
            else:
                outPeakNumber = string.atoi(PEAK.contributions[0].peakNumber)
            if PEAK.contributions[0].peakType == None:
                outPeakType = 6
            else:
                outPeakType = string.atoi(PEAK.contributions[0].peakType)   
            if outnumberlist[1] == None:
                outNumber1 = 0
            else:
                outNumber1 = string.atoi(outnumberlist[1])
            if outnumberlist[2] == None:
                outNumber2 = 0
            else:
                outNumber2 = string.atoi(outnumberlist[2])
            if outnumberlist[3] == None:
                outNumber3 = 0
            else:
                outNumber3 = string.atoi(outnumberlist[3])
            if outnumberlist[4] == None:
                outNumber4 = 0
            else:
                outNumber4 = string.atoi(outnumberlist[4])

            #replace integration method by 'm' (if None):
            integrationM = PEAK.contributions[0].xeasyinte
            if not integrationM:
                integrationM = 'm'

            #write out with the correct order:
            if self.dimension == 4:
                   peakshandle.write('%4i %7.3f %7.3f %7.3f %7.3f %1i %-9s %10.3e %9.2e %1s %3i %4i %4i %4i %4i %1i \n' %\
                                  (outPeakNumber,\
                                   string.atof(outppmlist[1]),\
                                   string.atof(outppmlist[2]),\
                                   string.atof(outppmlist[3]),\
                                   string.atof(outppmlist[4]),\
                                   outPeakType,\
                                   PEAK.contributions[0].xeasyudst,\
                                   string.atof(outvolume),\
                                   string.atof(outvolumeError),\
                                   integrationM,\
                                   0,          #unused1, always 0
                                   outNumber1,\
                                   outNumber2,\
                                   outNumber3,\
                                   outNumber4,\
                                   0))         #unused2, always 0
            elif self.dimension == 3:
                peakshandle.write('%4i %7.3f %7.3f %7.3f %1i %-9s %10.3e %9.2e %1s %3i %4i %4i %4i %1i \n' %\
                                  (outPeakNumber,\
                                   string.atof(outppmlist[1]),\
                                   string.atof(outppmlist[2]),\
                                   string.atof(outppmlist[3]),\
                                   outPeakType,\
                                   PEAK.contributions[0].xeasyudst,\
                                   string.atof(outvolume),\
                                   string.atof(outvolumeError),\
                                   integrationM,\
                                   0,          #unused1, always 0
                                   outNumber1,\
                                   outNumber2,\
                                   outNumber3,\
                                   0))         #unused2, always 0
            elif self.dimension == 2:
               peakshandle.write('%4i %7.3f %7.3f %1i %-9s %10.3e %9.2e %1s %3i %4i %4i %1i \n' %\
                                  (outPeakNumber,\
                                   string.atof(outppmlist[1]),\
                                   string.atof(outppmlist[2]),\
                                   outPeakType,\
                                   PEAK.contributions[0].xeasyudst,\
                                   string.atof(outvolume),\
                                   string.atof(outvolumeError),\
                                   integrationM,\
                                   0,          #unused1, always 0
                                   outNumber1,\
                                   outNumber2,\
                                   0))         #unused2, always 0
               
            #initialize the lists of possible assignments in all dimensions:
            possibleh1 = []
            possiblep1 = []
            possibleh2 = []
            possiblep2 = []
                
            #loop through all the contributions, set up the matrix:
            if len(PEAK.contributions) > 1:
                for CON in PEAK.contributions:
                    if PEAK.contributions == []:
                        continue
                    if not (CON.atomnumberh1 in possibleh1):
                        possibleh1.append(CON.atomnumberh1)
                    if not (CON.atomnumberp1 in possiblep1):
                        possiblep1.append(CON.atomnumberp1)
                    if not (CON.atomnumberh2 in possibleh2):
                        possibleh2.append(CON.atomnumberh2)
                    if not (CON.atomnumberp2 in possiblep2):
                        possiblep2.append(CON.atomnumberp2)
                if possibleh1.count(None) != 0:
                    possibleh1.remove(None)
                if possiblep1.count(None) != 0:
                    possiblep1.remove(None)
                if possibleh2.count(None) != 0:
                    possibleh2.remove(None)
                if possiblep2.count(None) != 0:
                    possiblep2.remove(None)
##                 print possibleh1, possiblep1, possibleh2, possiblep2  #test
                if (len(possiblep1)) > 0 and (len(possiblep2) > 0):
                    matrix = Numeric.zeros((len(possiblep1), len(possiblep2)))
                for CON in PEAK.contributions:
                    if PEAK.contributions == []:  #test
                        continue  #test
##                        print matrix, CON.atomnumberp1, CON.atomnumberp2  #test
                    matrix[possiblep1.index(CON.atomnumberp1),\
                           possiblep2.index(CON.atomnumberp2)] = 1

                #calculate the 4 binary numbers from the matrix:
                XAP = ReadXeasy.XeasyAssignPeak()
                #print warnings, if a peak has more than 10 contributions:
                if len(possibleh1) > 11:
                    print 'NOTE: peak', PEAK.peakNumber, 'has', len(possibleh1), 'contributions,',\
                          'only 10 can be used in XEASY'
                if len(possiblep1) > 11:
                    print 'NOTE: peak', PEAK.peakNumber, 'has', len(possiblep1), 'contributions,',\
                          'only 10 can be used in XEASY'
                if len(possibleh2) > 11:
                    print 'NOTE: peak', PEAK.peakNumber, 'has', len(possiblep2), 'contributions,',\
                          'only 10 can be used in XEASY'
                if len(possiblep2) > 11:
                    print 'NOTE: peak', PEAK.peakNumber, 'has', len(possiblep2), 'contributions,',\
                          'only 10 can be used in XEASY'
                #just use the first 10 contributions,
                #use the protons (set: 2 and 4) to set up the matrix:
                XAP.AddMatrix(PEAK.peakNumber, matrix, self.dimension,\
                              '2', '4',\
                              possibleh1[0:10], possiblep1[0:10],\
                              possibleh2[0:10], possiblep2[0:10])
                
                #write output to .assign file:
                assignhandle.write('# ' + CON.peakNumber + '\n')
                for qqq in range(1, 5):
                    qqq = str(qqq)
                    if het1 == qqq:
                        assignhandle.write(str(len(possibleh1[0:10])))
                        for eachatom in possibleh1[0:10]:
                            assignhandle.write(' ' + eachatom)
                        assignhandle.write('\n')
                    if pro1 == qqq:
                        assignhandle.write(str(len(possiblep1[0:10])))
                        for eachatom in possiblep1[0:10]:
                            assignhandle.write(' ' + eachatom)
                        assignhandle.write('\n')
                    if het2 == qqq:
                        assignhandle.write(str(len(possibleh2[0:10])))                        
                        for eachatom in possibleh2[0:10]:
                            assignhandle.write(' ' + eachatom)
                        assignhandle.write('\n')
                    if pro2 == qqq:
                        assignhandle.write(str(len(possiblep2[0:10])))
                        for eachatom in possiblep2[0:10]:
                            assignhandle.write(' ' + eachatom)
                        assignhandle.write('\n')
                for eachbin in XAP.binlist:
                    eachbin = str(eachbin)
                    eachbin = eachbin[:-1]   #get rid of the 'L' of the long integer
                    assignhandle.write(str(eachbin) + ' ')
                assignhandle.write('\n')
            else:
                #now for the case that you only have one contribution:
                assignhandle.write('# ' + PEAK.peakNumber + '\n')
                for qqq in range(1, 5):
                    qqq= str(qqq)
                    if het1 == qqq:
                        if PEAK.contributions[0].atomnumberh1 == None:
                            assignhandle.write('0\n')
                        else:
                            assignhandle.write('1 ' + PEAK.contributions[0].atomnumberh1 + '\n')
                    if pro1 == qqq:
                        if PEAK.contributions[0].atomnumberp1 == None:
                            assignhandle.write('0\n')
                        else:
                            assignhandle.write('1 ' + PEAK.contributions[0].atomnumberp2 + '\n')
                    if het2 == qqq:
                        if PEAK.contributions[0].atomnumberh2 == None:
                            assignhandle.write('0\n')
                        else:
                            assignhandle.write('1 ' + PEAK.contributions[0].atomnumberh1 + '\n')
                    if pro2 == qqq:
                        if PEAK.contributions[0].atomnumberp2 == None:
                            assignhandle.write('0\n')
                        else:
                            assignhandle.write('1 ' + PEAK.contributions[0].atomnumberp2 + '\n')
                #now write the binary numbers for fully assigned peaks:
                assignhandle.write('-1 -1 -1 -1\n')

        #close all the filehandles:
        peakshandle.close()
        assignhandle.close()
        
     
    def WriteXml(self, xmlFileName):
        """
        writes a XML file containing the peaklist
        pseudoatoms are not used, the atomnames should fit to the pdb files.
        """
        print 'writing all data to the following XML file:\n  ', xmlFileName
        xmlHandle = TextFile.TextFile(xmlFileName, 'w')
        
        for PEAK in self.peakslist:
            xmlHandle.write('<NOE>\n')
            peakString="""  <PEAK_PARAMETERS>
    <SPECTRUM_NAME>%s</SPECTRUM_NAME>
    <PEAK_NUMBER>%s</PEAK_NUMBER>
    <TEST_SET>%s</TEST_SET>
    <INPUT_WEIGHT>%s</INPUT_WEIGHT>
    <VOLUME>%s</VOLUME>
    <VOLUME_ERROR>%s</VOLUME_ERROR>
    <INTENSITY>%s</INTENSITY>
    <INTENSITY_ERROR>%s</INTENSITY_ERROR>
    <PPM_HETERO_1>%s</PPM_HETERO_1>
    <DELTA_PPM_HETERO_1>%s</DELTA_PPM_HETERO_1>
    <PPM_PROTON_1>%s</PPM_PROTON_1>
    <DELTA_PPM_PROTON_1>%s</DELTA_PPM_PROTON_1>
    <PPM_HETERO_2>%s</PPM_HETERO_2>
    <DELTA_PPM_HETERO_2>%s</DELTA_PPM_HETERO_2>
    <PPM_PROTON_2>%s</PPM_PROTON_2>
    <DELTA_PPM_PROTON_2>%s</DELTA_PPM_PROTON_2>
  </PEAK_PARAMETERS>
""" % ('',   #spectrum name
       PEAK.contributions[0].peakNumber,  #peak number
       '1',  #test set
       '0',  #input weight
       PEAK.contributions[0].volume,      #volume
       PEAK.contributions[0].volumeError, #volume error
       '0',  #intensity
       '0',  #intensity error
       PEAK.contributions[0].h1ppm,  #PPM_HETERO_1
       PEAK.contributions[0].dh1ppm, #DELTA_PPM_HETERO_1
       PEAK.contributions[0].p1ppm,  #PPM_PROTON_1
       PEAK.contributions[0].dp1ppm, #DELTA_PPM_PROTON_1
       PEAK.contributions[0].h2ppm,  #PPM_HETERO_2
       PEAK.contributions[0].dh2ppm, #DELTA_PPM_HETERO_2
       PEAK.contributions[0].p2ppm,  #PPM_PROTON_2
       PEAK.contributions[0].dp2ppm) #DELTA_PPM_PROTON_2
            ariaString="""  <ARIA_PARAMETERS>
    <PEAK_TYPE>%s</PEAKTYPE>
    <FIGURE_OF_MERIT>%s</FIGURE_OF_MERIT>
    <CURRENT_WEIGHT>%s</CURRENT_WEIGHT>
    <LOWERBOUND_BOUND>%s</LOWER_BOUND>
    <UPPER_BOUND>%s</UPPER_BOUND>
    <SUM_DISTANCE_AVERAGE>%s</SUM_DISTANCE_AVERAGE>
    <SUM_DISTANCE_STANDARD_DEVIATION>%s</SUM_DISTANCE_STANDARD_DEVIATION>
    <ALL_ASSIGNED>%s</ALL_ASSIGNED>
    <NON_TRIVIAL_ASSIGNMENTS>%s</NON_TRIVIAL_ASSIGNMENTS>
  </ARIA_PARAMETERS>
""" % (PEAK.contributions[0].peakType,  #PEAK_TYPE
       '0',  #FIGURE_OF_MERIT
       PEAK.contributions[0].weight,  #CURRENT_WEIGHT
       PEAK.contributions[0].lowerBound,   #LOWER_BOUND
       PEAK.contributions[0].upperBound,   #UPPER_BOUND
       '0',  #SUM_DISTANCE_AVERAGE
       '0',  #SUM_DISTANCE_STANDARD_DEVIATION
       '0',  #ALL_ASSIGNED
       PEAK.contributions[0].nta)      #NON_TRIVIAL_ASSIGNMENTS
            xmlHandle.write(peakString + ariaString)
            for CON in PEAK.contributions:
                contributionString="""  <CONTRIBUTION_PARAMETERS>
    <CONTRIBUTION>%s</CONTRIBUTION>
    <FIGURE_OF_MERIT>%s</FIGURE_OF_MERIT>
    <DISTANCE_AVERAGE>%s</DISTANCE_AVERAGE>
    <DISTANCE_STANDARD_DEVIATAION>%s</DISTANCE_STANDARD_DEVIATAION>
    <SEGID_1>%s</SEGID_1>
    <RESIDUE_1>%s</RESIDUE_1>
    <AMINOACID_1>%s</AMINOACID_1>
    <ATOMNAME_HETERO_1>%s</ATOMNAME_HETERO_1>
    <ASSIGNED_PPM_HETERO_1>%s</ASSIGNED_PPM_HETERO_1>
    <ASSIGNED_DELTA_PPM_HETERO_1>%s</ASSIGNED_DELTA_PPM_HETERO_1>
    <ATOMNAME_PROTON_1>%s</ATOMNAME_PROTON_1>
    <ASSIGNED_PPM_PROTON_1>%s</ASSIGNED_PPM_PROTON_1>
    <ASSIGNED_DELTA_PPM_PROTON_1>%s</ASSIGNED_DELTA_PPM_PROTON_1>
    <SEGID_2>%s</SEGID_2>
    <RESIDUE_2>%s</RESIDUE_2>
    <AMINOACID_2>%s</AMINOACID_2>
    <ATOMNAME_HETERO_2>%s</ATOMNAME_HETERO_2>
    <ASSIGNED_PPM_HETERO_2>%s</ASSIGNED_PPM_HETERO_2>
    <ASSIGNED_DELTA_PPM_HETERO_2>%s</ASSIGNED_DELTA_PPM_HETERO_2>
    <ATOMNAME_PROTON_2>%s</ATOMNAME_PROTON_2>
    <ASSIGNED_PPM_PROTON_2>%s</ASSIGNED_PPM_PROTON_2>
    <ASSIGNED_DELTA_PPM_PROTON_2>%s</ASSIGNED_DELTA_PPM_PROTON_2>
    <ATOMNUMBER_HETERO_1>%s</ATOMNUMBER_HETERO_1>
    <ATOMNUMBER_PROTON_1>%s</ATOMNUMBER_PROTON_1>
    <ATOMNUMBER_HETERO_2>%s</ATOMNUMBER_HETERO_2>
    <ATOMNUMBER_PROTON_2>%s</ATOMNUMBER_PROTON_2>
  </CONTRIBUTION_PARAMETERS>
""" % (CON.contribution,  #CONTRIBUTION
       '0',               #FIGURE_OF_MERIT  #test
       CON.distanceAve,      #DISTANCE_AVERAGE
       '0',               #DISTANCE_STANDARD_DEVIATAION  #test
       CON.segid1,        #SEGID_1
       CON.residue1,      #RESIDUE_1
       CON.aa1,           #AMINOACID_1
       CON.atomnameh1,    #ATOMNAME_HETERO_1
       CON.ah1ppm,        #ASSIGNED_PPM_HETERO_1
       '0',               #ASSIGNED_DELTA_PPM_HETERO_1  #test
       CON.atomnamep1,    #ATOMNAME_PROTON_1
       CON.ap1ppm,        #ASSIGNED_PPM_PROTON_1
       '0',               #ASSIGNED_DELTA_PPM_PROTON_1
       CON.segid2,        #SEGID_2
       CON.residue2,      #RESIDUE_2
       CON.aa2,           #AMINOACID_2
       CON.atomnameh2,    #ATOMNAME_HETERO_2
       CON.ah2ppm,        #ASSIGNED_PPM_HETERO_2
       '0',               #ASSIGNED_DELTA_PPM_HETERO_2  #test
       CON.atomnamep2,    #ATOMNAME_PROTON_2
       CON.ap2ppm,        #ASSIGNED_PPM_PROTON_2
       '0',               #ASSIGNED_DELTA_PPM_PROTON_2
       CON.atomnumberh1,  #ATOMNUMBER_HETERO_1
       CON.atomnumberp1,  #ATOMNUMBER_PROTON_1
       CON.atomnumberh2,  #ATOMNUMBER_HETERO_2
       CON.atomnumberp2)  #ATOMNUMBER_PROTON_2
                xmlHandle.write(contributionString)
            xmlHandle.write('</NOE>\n')
            
        #close all the filehandles:
        xmlHandle.close()
      
    
    def _ReadOldList(self, listfile):
        """
        reads in an old ARIA .list file and writes a list of peak
        objects to the attribute .peakslist
        """
        #get a clean list:
        self.peakslist = []
        NP = NoePeak()
        CON = NoeContribution()
        oldpeakNumber = 0
        withsegid = 0      #if 0, segid is empty
        #set some variables:
        segid1 = None
        segid2 = None
        atomnameh1 = (None,)
        atomnameh2 = (None,)
        atomnamep1 = []
        atomnamep2 = []
        volumeError = None
        distanceError = None
        for line in TextFile.TextFile(listfile):
            #for the negative volumes, add one space character:
            #add spaces for heteronucleus - this is a ugly hack!!!
            line = line[:54] + ' ' + line[54:]
            line = line[:77] + ' ' + line[77:]
            line = line[:87] + ' ' + line[87:]

            linelist = string.split(line)

            #for non-data lines:
            if len(linelist) < 5:
                continue

            #for non-data lines:
            sign = linelist[0]
            if sign != '+' and sign != None:
                continue   #throws out all non-data lines

            #for the segid field:
            try:
                testatoi = string.atoi(linelist[2])
            except ValueError:
                withsegid = 1
                
            if withsegid == 0:
                peakNumber = linelist[1]
                residue1 = linelist[2]
                aa1 = linelist[3]
                atomnamep1 = (linelist[4], )
                residue2 = linelist[5]
                aa2 = linelist[6]
                atomnamep2 = (linelist[7], )
                distance = linelist[8]
                volume = linelist[9]
                lowerBound = linelist[10]
                upperBound = linelist[11]
                hppm1 = linelist[12]
                hppm2 = linelist[13]
                pppm1 = linelist[14]
                pppm2 = linelist[15]
                contribution = linelist[16]
                weight = linelist[17]
                nontrivial = '1' #linelist[18]
            else:
                peakNumber = linelist[1]
                segid1 = linelist[2]
                residue1 = linelist[3]
                aa1 = linelist[4]
                atomnamep1 = (linelist[5], )
                segid2 = linelist[6]
                residue2 = linelist[7]
                aa2 = linelist[8]
                atomnamep2 = (linelist[9], )
                distance = linelist[10]
                volume = linelist[11]
                lowerBound = linelist[12]
                upperBound = linelist[13]
                hppm1 = linelist[14]
                hppm2 = linelist[15]
                pppm1 = linelist[16]
                pppm2 = linelist[17]
                contribution = linelist[18]
                weight = linelist[19]
                nontrivial = linelist[20]

            oldpeakNumber = CON.peakNumber
            
            #convert +/- to peakTypes:
            if string.atoi(nontrivial) > 1:
                if sign == '+':
                    peakType = '2'
                else:
                    peakType = '4'
            else:
                if sign == '+':
                    peakType = '1'
                else:
                    peakType = '3'
                    
            CON = NoeContribution(peakNumber = peakNumber,\
                                  peakType = peakType,\
                                  residue1 = residue1,\
                                  aa1 = aa1,\
                                  segid1 = segid1,\
                                  atomnameh1 = atomnameh1,\
                                  atomnamep1 = atomnamep1,\
                                  residue2 = residue2,\
                                  aa2 = aa2,\
                                  segid2 = segid2,\
                                  atomnameh2 = atomnameh2,\
                                  atomnamep2 = atomnamep2,\
                                  volume = volume,\
                                  volumeError = volumeError,\
                                  distanceAve = distance,\
                                  distanceStd = distanceError,\
                                  lowerBound = lowerBound,\
                                  upperBound = upperBound,\
                                  pppm1 = pppm1,\
                                  pppm2 = pppm2,\
                                  inWeight = weight,\
                                  contribution = contribution,\
                                  nta = nontrivial)

            #add contribution to peak:
            if oldpeakNumber == peakNumber:
                NP.AddContribution(CON)
            else:
                #the output:
                self.AddPeak(NP)
                #create a new instance of a NOE peak:
                NP = NoePeak()
                NP.AddContribution(CON)
        #for the last peak:
        self.AddPeak(NP)
        
###############################################################################
class NoePeak:
    """
    Represents one NOE peak with all possible contributions (like several
    lines for one peak in an ARIA .list file)
    
    attributes:
        contributions    a list of NoeContribution objects
        peakNumber       peakNumber (e.g. from XEASY)
        spectrumName     spectrum name
    methods:
        AddContribution  adds a contribution object to the peak
    """
    def __init__(self):
        self.contributions = []
        self.peakNumber = None
        self.spectrumName = None
    
    def AddContribution(self, newContribution):
        self.contributions.append(newContribution)
        if self.peakNumber == None:
            self.peakNumber = newContribution.peakNumber
        if self.spectrumName == None:
            self.spectrumName = newContribution.spectrumName

    
###############################################################################
class NoeContribution:
    """
    Represents one contribution to an NOE peak (like one line in an ARIA
    .list file).
    This object just contains all the data as attributes, there are no
    methods available.    
    """
    def __init__(self,\
                 spectrumName = None,\
                 peakNumber = None,\
                 testSet = None,\
                 inWeight = None,\
                 volume = None,\
                 volumeError = None,\
                 intensity = None,\
                 intensityError = None,\
                 h1ppm = None,\
                 dh1ppm = None,\
                 p1ppm = None,\
                 dp1ppm = None,\
                 h2ppm = None,\
                 dh2ppm = None,\
                 p2ppm = None,\
                 dp2ppm = None,\
                 peakType = None,\
                 fomAria = None,\
                 curWeight = None,\
                 lowerBound = None,\
                 upperBound = None,\
                 sumDistanceAve = None,\
                 sumDistanceStd = None,\
                 backVolumeAria = None,\
                 backVolumeAriaStd = None,\
                 allAssi = None,\
                 nta = None,\
                 contribution = None,\
                 fomContribution = None,\
                 distanceAve = None,\
                 distanceStd = None,\
                 backVolume = None,\
                 backVolumeStd = None,\
                 segid1 = None,\
                 residue1 = None,\
                 aa1 = None,\
                 atomnameh1 = (None, ),\
                 assih1ppm = None,\
                 assidh1ppm = None,\
                 atomnamep1 = (None, ),\
                 assip1ppm = None,\
                 assidp1ppm = None,\
                 segid2 = None,\
                 residue2 = None,\
                 aa2 = None,\
                 atomnameh2 = (None, ),\
                 assih2ppm = None,\
                 assidh2ppm = None,\
                 atomnamep2 = (None, ),\
                 assip2ppm = None,\
                 assidp2ppm = None,\
                 xeasyudst = 'T',\
                 xeasyinte = None,\
                 atomnumberh1 = None,\
                 atomnumberp1 = None,\
                 atomnumberh2 = None,\
                 atomnumberp2 = None,\
                 comment = None):

## the old constructor contained:                 
##                 peakNumber = None, peakType = None,\
##                 residue1 = None, aa1 = None, segid1 = None,\
##                 atomnameh1 = (None, ), atomnamep1 = (None, ),\
##                 residue2 = None, aa2 = None, segid2 = None,\
##                 atomnameh2 = (None, ), atomnamep2 = (None, ),\
##                 volume = None, volumeError = None,\
##                 distance = None, distanceError = None,\
##                 lowerBound = None, upperBound = None,\
##                 h1ppm = None, dh1ppm = None, p1ppm = None, dp1ppm = None,\
##                 h2ppm = None, dh2ppm = None, p2ppm = None, dp2ppm = None,\
##                 weight = None, contribution = None, nta = None, comment = None,\
##                 ah1ppm = None, dah1ppm = None, ap1ppm = None, dap1ppm = None,\
##                 ah2ppm = None, dah2ppm = None, ap2ppm = None, dap2ppm = None,\
##                 xeasyudst = 'T', xeasyinte = None):

        #peak line:
        self.spectrumName = spectrumName #spectrum name (not longer than 4 characters)
        self.peakNumber = peakNumber     #peak number
        self.testSet = testSet           #either 1 or 0 (1=used for the calculation, 0=not used)
        self.inWeight = inWeight         #initial weight of the peak
        self.volume = volume             #NOE volume
        self.volumeError = volumeError   #NOE volume error
        self.intensity = intensity       #NOE intensity
        self.intensityError = intensityError       #NOE intensity error
        self.h1ppm = h1ppm               #heteronucleus1 chemical shift
        self.dh1ppm = dh1ppm             #delta of heteronucleus1 chemical shift
        self.p1ppm = p1ppm               #proton1 chemical shift
        self.dp1ppm = dp1ppm             #delta of proton1 chemical shift
        self.h2ppm = h2ppm               #heteronucleus2 chemical shift
        self.dh2ppm = dh2ppm             #delta of heteronucleus2 chemical shift
        self.p2ppm = p2ppm               #proton2 chemical shift
        self.dp2ppm = dp2ppm             #delta of proton2 chemical shift
        
        #aria line:
        self.peakType = peakType         #colour coding in the range [1, 6]
        self.fomAria = fomAria           #figure of merit
        self.curWeight = curWeight       #current weight
        self.lowerBound = lowerBound               #lower distance bound
        self.upperBound = upperBound               #upper distance bound
        self.sumDistanceAve = sumDistanceAve       #sum of all distances
        self.sumDistanceStd = sumDistanceStd       #std of the sum of all distances
        self.backVolumeAria = backVolumeAria       #backcalculated volume
        self.backVolumeAriaStd = backVolumeAriaStd #std of the backcalculated volume
        self.allAssi = allAssi           #number of all assignments
        self.nta = nta                   #number of non-trivial assignments
        
        #contribution line:
        self.contribution = contribution #contribution to the whole peak volume
        self.fomContribution = fomContribution     #figure of merit
        self.distanceAve = distanceAve   #average distance
        self.distanceStd = distanceStd   #standard deviation of the sum of the average distance
        self.backVolume = backVolume     #back calculated volume
        self.backVolumeStd = backVolumeStd #standard deviation of the backcalculated volume
        self.segid1 = segid1             #segid 1
        self.residue1 = residue1         #residue number 1
        self.aa1 = aa1                   #three-letter code 1
        self.atomnameh1 = atomnameh1     #heteronucleus atomname 1
        self.assih1ppm = assih1ppm       #ppm of heteronucleus 1
        self.assidh1ppm = assidh1ppm     #delta of ppm of heteronucleus 1
        self.atomnamep1 = atomnamep1     #atomname of proton 1
        self.assip1ppm = assip1ppm       #ppm of proton 1
        self.assidp1ppm = assidp1ppm     #delta of ppm of proton 1
        self.segid2 = segid2             #segid 2
        self.residue2 = residue2         #residue number 2
        self.aa2 = aa2                   #three-letter code 2
        self.atomnameh2 = atomnameh2     #heteronucleus atomname 2
        self.assih2ppm = assih2ppm       #ppm of heteronucleus 2
        self.assidh2ppm = assidh2ppm     #delta of ppm of heteronucleus 2
        self.atomnamep2 = atomnamep2     #atomname of proton 2
        self.assip2ppm = assip2ppm       #ppm of proton 2
        self.assidp2ppm = assidp2ppm     #delta of ppm of proton 2
        
        #xeasy specific stuff:
        self.xeasyudst = xeasyudst       #user-defined spectrum type
        self.xeasyinte = xeasyinte       #integration method
        self.atomnumberh1 = atomnumberh1 #XEASY atomnumber hetero 1
        self.atomnumberp1 = atomnumberp1 #XEASY atomnumber proton 1
        self.atomnumberh2 = atomnumberh2 #XEASY atomnumber hetero 2
        self.atomnumberp2 = atomnumberp2 #XEASY atomnumber proton 2

        #comments:
        self.comment = comment           #each contribution may be commented!

###############################################################################
# Below there are some private classes and functions                          #
###############################################################################

def _DoesFileExist(fileName):
    if os.path.exists(fileName) == 0:
        print 'WARNING:', fileName, 'does not exist.'
        return 0
    return 1
