"""
NoeExclusion.py
does Libra/Balanced and Random exclusion of NOEs
Sean I. O'Donoghue and Jens Linge

USAGE:
    import NoeExclusion
    LibraExclusion(100, 20, 0.05, NoeExclusion.SetSeed())
    RandomExclusion(100, 20, 0.05, NoeExclusion.SetSeed())

Note:
    all the indices start from 0
    this is important e.g. for the structure numbers and the peak numbers

"""
__author__   = "$Author: linge $"
__revision__ = "$Revision: 1.1.1.1 $"
__date__     = "$Date: 1999/09/02 13:35:20 $"

import math, os, string, sys, time, whrandom

ExclusionError="Error calculating NOE exclusions"

def LibraExclusion(peakNo,Structures,Exclusion,RandNumGen):
    """
    peakNo=no. of noes
    Structures=structures per iteration
    Exclusion=fraction of structures in which each noe is excluded
    RandNumGen=whrandom instance

    returns a dictionary:
      key    (structureNumber, peakNumber)
      value  1 (=excluded) or 0 (=included)

    NOTE: structures and peak indices start with 0
    """
    if peakNo<1:
	raise ExclusionError, 'Number of NOEs, N, must be at least 1'
    if Structures<1:
	raise ExclusionError, 'Number of structures must be at least 1'
    if Exclusion<0:
	raise ExclusionError, 'Exclusion ratio must be greater than zero'
    if Exclusion>1:
	raise ExclusionError, 'Exclusion ratio must not be more than 1'
    try:
	RandNumGen.randint(0,peakNo)
    except:
	raise ExclusionError, "The supplied ratio random number generator instance, RandNumGen, didn't work."
        
    #create a dictionary (key: structureNumber, value: a list of peak numbers):
    totalPeakDic = {}
    for eachStructure in range(Structures):
        totalPeakDic[eachStructure] = range(peakNo)
    
    #the dictionary for the output:
    excludedDic = {}

    #loop over all the peaks for all the structures (except the last one):
    totalDev = 0
    for eachStructure in range(Structures - 1):
        totalDev = totalDev + peakNo*Exclusion - round(peakNo*Exclusion)
        upper = round(peakNo*Exclusion)
        #distribute the reminder equally, use totalDev for that:
        if totalDev >= 1:
            totalDev = totalDev - 1
            upper = upper + 1
        elif totalDev <= -1:
            totalDev = totalDev + 1
            upper = upper - 1
        for iii in range(upper):
            jjj = RandNumGen.randint(0, len(totalPeakDic[eachStructure]) - 1)
            #exclude this peak for the current structure:
            excludedDic[(eachStructure, totalPeakDic[eachStructure][jjj])] = 1
            #include this peak for all the other structures:
            for struc in range(Structures):
                if struc != eachStructure:
                    excludedDic[(struc, totalPeakDic[struc][jjj])] = 0
                #remove this peak from the totalPeakList:
                del totalPeakDic[struc][jjj]
    
    #the last structure gets the rest:
    for kkk in totalPeakDic[Structures-1]:
        excludedDic[(Structures-1, kkk)] = 1
                
    return excludedDic


def RandomExclusion(peakNo,Structures,Exclusion,RandNumGen):
    """
    peakNo=no. of noes
    Structures=structures per iteration
    Exclusion=fraction of structures in which each noe is excluded
    RandNumGen=whrandom instance

    returns a dictionary:
      key    (structureNumber, peakNumber)
      value  1 (=excluded) or 0 (=included)

    NOTE: structures and peak indices start with 0
    """
    if peakNo<1:
	raise ExclusionError, 'Number of NOEs, N, must be at least 1'
    if Structures<1:
	raise ExclusionError, 'Number of structures must be at least 1'
    if Exclusion<0:
	raise ExclusionError, 'Exclusion ratio must be greater than zero'
    if Exclusion>1:
	raise ExclusionError, 'Exclusion ratio must not be more than 1'
    try:
	RandNumGen.randint(0,peakNo)
    except:
	raise ExclusionError, "The supplied ratio random number generator instance, RandNumGen, didn't work."
    
    #create a dictionary (key: structureNumber, value: a list of peak numbers),
    #containing all the peaks for each structure:
    totalPeakDic = {}
    for eachStructure in range(Structures):
        totalPeakDic[eachStructure] = range(peakNo)
    
    #the dictionary for the output:
    excludedDic = {}
    
    #loop over all the peaks for all the structures:
    for eachStructure in range(Structures):
        for iii in range(round(peakNo*Exclusion)):
            jjj = RandNumGen.randint(0, len(totalPeakDic[eachStructure]) - 1)
            #exclude this peak for the current structure:
            excludedDic[(eachStructure, totalPeakDic[eachStructure][jjj])] = 1
            #remove this peak from the totalPeakList:
            del totalPeakDic[eachStructure][jjj]
        #include all the other peaks:
        for jjj in totalPeakDic[eachStructure]:
            excludedDic[(eachStructure, jjj)] = 0
    return excludedDic


def SetSeed():
    """
    sets the seed for the random number generator
    returns an whrandom.whrandom() instance

    uses the processID and the time for initializing the seed
    
    Note: This subroutine should be called only once
    """
    ProcessID = int(os.getpid())
    ProcessIDSize=len(str(ProcessID))
    ReverseProcessID = float(_ReverseString(str(ProcessID)))
    ProcessID = float(ProcessID)

    Time = int(time.time())
    TimeSize=len(str(Time))
    ReverseTime = string.atof(_ReverseString(str(Time)))
    Time = float(Time)

    Seed1 = int(math.fmod(1000*pow(10.,TimeSize-ProcessIDSize)*ProcessID/ReverseTime,256))
    Seed2 = int(math.fmod(1000*pow(10.,ProcessIDSize-TimeSize)*Time/ReverseProcessID,256))
    Seed3 = int(math.fmod(1000*(Time/ReverseTime)*(ProcessID/ReverseProcessID),256))

    RandNumGen = whrandom.whrandom()
    RandNumGen.seed(Seed1, Seed2, Seed3)
    i=0
    while i<100:
	Seed1 = RandNumGen.randint(0, 255)
	Seed2 = RandNumGen.randint(0, 255)
	Seed3 = RandNumGen.randint(0, 255)
	i=i+1
    RandNumGen.seed(Seed1, Seed2, Seed3)
    return RandNumGen


def _ReverseString(String):
    """
    This function returns a reverted string
    e.g.
      _ReverseString('12345')
      returns the string '54321'
    """
    n=len(String)
    RevString=""
    while n>0:
	n=n-1
	RevString=RevString+String[n]
    return(RevString)


#test code:
if __name__ == "__main__":
    print 'testing: LibraExclusion(1000, 20, 0.05, NoeExclusion.SetSeed())'
    print LibraExclusion(100, 20, 0.05, SetSeed())
    print 'testing: RandomExclusion(1000, 20, 0.05, NoeExclusion.SetSeed())'
    print RandomExclusion(100, 20, 0.05, SetSeed())
