"""
A module to split up an ANSIG crosspeaks export file in several files which
only contain the NOEs of one spectrum.

COMMANDLINE USAGE: python SplitAnsgigCpe filename

splits up the file

"""
__author__   = "$Author: anonymous $"
__revision__ = "$Revision: 2.1 $"
__date__     = "$Date: 2001/06/18 12:28:16 $"

import os, re, string, sys
from Aria.ThirdParty import FortranFormat 

def SplitAnsigCpeFile2OutRoot(fileName, outRoot):
    """
    reads in a ANSIG .cpe file
    writes all the files to outroot
    outroot must be an absolute path of a directory
    e.g. SplitAnsigCpeFile2OutRoot('/nmr/export.crp_170699','/nmr')
         produces some files in /nmr with the spectra names as filesnames
    """
    spectrumDic = SplitAnsigCpeFile(fileName)
    for eachSpectrum in spectrumDic.keys():
        if eachSpectrum == '':
            fileHandle = open(outRoot + '/' + 'withoutName.crp', 'w')
        else:
            fileHandle = open(outRoot + '/ansig_' + eachSpectrum + '.crp', 'w')
        fileHandle.write(spectrumDic[eachSpectrum])
        fileHandle.close()
    
def SplitAnsigCpeFile(fileName):
    """
    fileName must be the absolute path
    
    reads an ANSIG crosspeaks export file or an ANSIG storage crosspeaks file
    splits it up in strings (each string contains one crosspeaks export file)
    returns a list of strings containing the new crosspeaks export files
    """
    if _DoesFileExist(fileName) == 0:
        return
    print 'reading the ANSIG crosspeaks export file:\n ', fileName    
    cpHandle = open(fileName)
    return SplitAnsigCpeObject(cpHandle)

def SplitAnsigCpeObject(fileObject):
    """
    input: a fileObject containing a crosspeaks export file
    
    reads an ANSIG crosspeaks export file or an ANSIG storage crosspeaks file
    splits it up in strings (each string contains one crosspeaks export file)
    returns a list of strings containing the new crosspeaks export files
    """
    bang = re.compile('#')
    
    #get total number of crosspeaks and dimensionality from the first two lines:
    eachLine = fileObject.readline()
    firstTwoLines = eachLine
    eachLine = fileObject.readline()
    firstTwoLines = firstTwoLines + eachLine

    totAndDim = string.split(eachLine)
    totalNumber = totAndDim[-2]
    dimension = int(totAndDim[-1])
    
    #there are two possible cases:
    #1. ANSIG v3.3 export crosspeaks file
    #   second line contains 'totalNumber dimensionality'
    #   spectrumName appears in every line
    #2. ANSIG v3.3 storage crosspeaks file
    #   second line contains 'spectrumName totalNumber dimensionality'
    #   spectrumName appears only in the header

    if len(totAndDim) == 2:  #for the ANSIG crosspeaks files           
        format2 = FortranFormat.FortranFormat('3E13.6,A12,7I6,6A4')
        format3 = FortranFormat.FortranFormat('4E13.6,A12,9I6,9A4')
        format4 = FortranFormat.FortranFormat('5E13.6,A12,11I6,12A4')
        if dimension == 2:
            currentFormat = format2
        elif dimension == 3:
            currentFormat = format3
        elif dimension == 4:
            currentFormat = format4
        
        #dictionary of cp export files (key=spectrumName, value=String):
        cpFiles = {}
        
        for eachLine in fileObject.readlines():
            if bang.match(eachLine) or len(eachLine) < 40:
                continue

            lineList = FortranFormat.FortranLine(eachLine, currentFormat)
            spectrumName = string.strip(lineList[dimension + 1])

            #look if the spectrum exists internally:
            if spectrumName in cpFiles.keys():
                cpFiles[spectrumName] = cpFiles[spectrumName] + eachLine
                
            #otherwise create a new internal spectrum:
            else:
                print '  found spectrum', spectrumName
                cpFiles[spectrumName] = firstTwoLines + eachLine
        return cpFiles
    else:  #for the ANSIG storage files it's easy anyway!
        spectrumName = totAndDim[0]
        return {spectrumName: string.join(fileObject.readlines(), '')}
    
    
def _DoesFileExist(fileName):
    if os.path.exists(fileName) == 0:
        print 'WARNING:', fileName, 'does not exist.'
        return 0
    return 1


#test:
if __name__ == '__main__':
    testCpString = """ANSIG v3.3 export crosspeaks file
  8487     3
 5.739931E+01 1.178608E+02 8.546566E+00 6.400323E+08cbcanh           0     0     0   458  1085   458  1085     0     08   8   8   Ser Ser Ser CA  N   HN  
 1.197964E+02 7.995593E+00 0.000000E+00 3.826041E+07n-hsqc           0     0     0     0     0     0     0   570     03   3       Glu Glu     N   HN      
 1.717820E+02 8.554000E+00 0.000000E+00 7.009845E+07hnco             0     0     0     0     0     0     0     0     0
"""
    if len(sys.argv) == 3:
        fileName = sys.argv[1]
        outRoot = sys.argv[2]
        print 'working on file', fileName
        print 'writing files to directory', outRoot
        SplitAnsigCpeFile2OutRoot(fileName, outRoot)
    elif len(sys.argv) == 2:
        fileName = sys.argv[1]
        print 'working on file', fileName
        outRoot = os.getcwd()
        print 'writing files to directory', outRoot
        SplitAnsigCpeFile2OutRoot(fileName, outRoot)
    else:
        print 'testing mode'
        print 'the input crosspeaks export file is:'
        print testCpString
        import StringIO
        ST = StringIO.StringIO(testCpString)
        spectrumDic = SplitAnsigCpeObject(ST)
        for eachSpectrum in spectrumDic.keys():
            print '\nthe spectrum', eachSpectrum, 'has the crosspeaks export file:'
            print spectrumDic[eachSpectrum] #e.g. you can write it to another file
    print 'ciao'
