      program stereosearch
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          26-AUG-1988
c
c
c
c  This program, and all parts of it, are freeware.
c  No warranty is expressed or implied.
c
c
c
c  This program searches the database created with STEREOBASE to do the
c  stereospecific assignments and obtain torsion angle restraints.
c
c  The program asks for three files:
c
c    1) (OUTPUT)    the file containing the results of the search 
c    2) (INPUT)     the file containing the constraints derived from 
c                   NMR experiments
c    3) (INPUT)     the database file containing the three torsion 
c                   angles phi, psi and chi1, and corresponding coupling
c                   constants and interproton distances.
c
c
c  The constraints input file has the following form:
c  
c  residue ser 23
c    daN-   =  3.5      0.3
c    dNN+   >  dNN-     0.5
c    .....  .  .....    ...
c    .....  .  .....    ...
c    JNa    =  6.0      2.0
c    Jab2   =  ...      ...
c    Jab3   <  ...      ...
c    phi    =  -60.0    40.0   
c    psi    =  -60.0    40.0
c    chi1   =  60.0     20.0    3
c  end
c  
c  Each record has a maximum of five entries: 
c  1)  a connectivity, coupling constant or dihedral angle
c  2)  a relational operator (>, =, <)
c  3)  another connectivity etc. or an absolute value
c  4)  an error estimate (optional, default is 0).
c  5)  a multiplicity for the angle constraints (optional, default is 0).
c  The format within the records is free. The constraints can be entered
c  in any order; it may save some CPU time if one enters the most 
c  restrictive constraints first.
c
c  Connectivities and coupling constants have unique names generated as
c  follows: 
c    1) type of constraint:
c          d        connectivity
c          J        coupling constant
c    2) first atom type:
c          a, b2, b3, N for HCa, HCb2, HCb3, HN, resp.
c    3) second atom type:
c    4) a minus sign for a (i-1,i) connectivity, a plus sign for (i,i+1)
c  Thus, db2N+ or dbNb2+ is the noe connectivity from Hb2 of residue 
c  i to HN of i+1.
c  Cases are distinguished in the above examples for clarity only; the 
c  program is not case sensitive, everything can be entered uppercase
c  or lowercase.
c  
c
c  The database file is an unformatted file with one record for each confor-
c  mation. Each record contains the three torsion angles, the associated 
c  coupling constants and interproton distances in the following order:
c  
c    0  NOT USED
c    1  PHI
c    2  PSI
c    3  CHI1
c    4  JNA
c    5  JAB2
c    6  JAB3
c    7  DAN-    1)
c    8  DNN-    1)
c    9  DAB3
c   10  DAB2
c   11  DNB3
c   12  DNB2
c   13  DNB3+
c   14  DNB2+
c   15  DAN
c   16  DAN+
c   17  DNN+
c  
c  1) As these distances do not depend on phi, psi or chi1, the constraints
c     are not usually used by the search program. They may be useful, 
c     though, in connection with a database generated from crystal structures.
c
c
c
c
c
c
      implicit none
c  global variables
      include 'const.inc'
      logical NEXT_FILE_OPEN, READ_RESTRAINTS, READ_DB, SATISFIED
c  local variables
      integer i, n_cond
      integer i_data(2,MAXCOND), i_swap(2,MAXCOND)
      real base(-MAXREST:MAXREST), err(MAXCOND)
      integer mult(MAXCOND)
      character*1 rel(MAXCOND)
      character*10 errflag      
      integer phi(NANG), psi(NANG), chi1(NANG)
      integer n_res, n_ass, n_poss, n_inc, n_both, n_phi, n_psi, n_chi1
      integer n_base, n_input, n_swap, n_i, n_s, n_rgti, n_igtr
      integer n_rev_cr, n_inp_cr
      logical error
      integer u_out, u_constr, u_db
      character*80 outfile, constr, dbfile
      character*10 r_name
      integer r_num
      integer storeno(400), storeyes(400)
c
c
c  begin
c
c
c  get filenames from primary input and open files
c  output file
      if (.not. NEXT_FILE_OPEN('Output file: ', 
     2            outfile , u_out, 'WRITE', 'FORMATTED'))
     3  goto 9999
c  constraints input file
      if (.not. NEXT_FILE_OPEN('Constraints file: ', 
     2            constr, u_constr, 'READ', 'FORMATTED'))
     3  goto 9999
c  database file
      if (.not. NEXT_FILE_OPEN('Database file:     ',
     2            dbfile, u_db, 'READ', 'UNFORMATTED'))
     3  goto 9999
c
c
      write(u_out,'(a)')
     2 'STEREOSEARCH  (version 1.0, 21-MAR-1989)             M. Nilges'
      write(u_out,'(a)')
     2 'Stereospecific Assignments by Conformational Database Searches'
      write(u_out,'(a)')
     2 'No warranty expressed or implied'
      write(u_out,'(a)') ' '
      write(u_out,'(a,a60)') 'Constraints file: ', constr
      write(u_out,'(a,a60)') 'Database file   : ', dbfile
      write(u_out,'(a)')
      write(u_out,'(a)')
c
c
c
c  loop over all residues in input file
c
c
c
      do i=1,400
        storeno(i) = 0
        storeyes(i) = 0
      end do
      n_res = 0
      n_ass = 0
      n_both = 0
      n_inc = 0
      n_i = 0
      n_s = 0
      n_rgti = 0
      n_igtr = 0
      n_inp_cr = 0
      n_rev_cr = 0
      do while (READ_RESTRAINTS(u_constr, r_name, r_num, 
     2               n_cond, base, i_data, i_swap, 
     3               rel, err, mult, errflag))
c
c
c
      n_res = n_res + 1
c
c  initialize arrays
      do i=1, NANG
        phi(i) = 0
        psi(i) = 0
        chi1(i) = 0
      end do
c
c
c  search database
c
      n_base = 0
      n_input = 0
      n_swap = 0
      do while (READ_DB(u_db, base, errflag))
c
        n_base = n_base + 1
        i = 1
        do while (SATISFIED(i, n_cond,
     2               base(i_data(1,i)), base(i_data(2,i)), 
     3               err(i), rel(i), mult(i)) )
          i = i + 1
        end do
        if (i .gt. n_cond) then
          n_input = n_input + 1
          call STORE_ANGLES(base(1), phi, .false.)
          call STORE_ANGLES(base(2), psi, .false.)
          call STORE_ANGLES(base(3), chi1, .false.)
        end if
c
c
        i = 1
        do while (SATISFIED(i, n_cond,
     2               base(i_swap(1,i)), base(i_swap(2,i)), 
     3               err(i), rel(i), mult(i)) )
          i = i + 1
        end do
        if (i .gt. n_cond) then
          n_swap = n_swap + 1
          call STORE_ANGLES(base(1), phi, .true.)
          call STORE_ANGLES(base(2), psi, .true.)
          call STORE_ANGLES(base(3), chi1, .true.)
        end if
c
      end do   !!! database search
c
c
c  output the results
c
      write(u_out,'(a,a,i4,/)') 'Results for residue ',r_name(1:3),r_num
      write(u_out,'(a,i6)')   '# conf in database:          ', n_base
      write(u_out,'(a,i6)')   '# conf. input assignments:   ', n_input
      write(u_out,'(a,i6,/)') '# conf. swapped assignments: ', n_swap
      if     (n_input .gt. n_swap ) then
        n_igtr = n_igtr + 1
      end if
      if     (n_input .lt. n_swap ) then
        n_rgti = n_rgti + 1
      end if
      if     ((n_input .eq. 0) .and. (n_swap .eq. 0)) then
        write(u_out,'(a,/)') '==> Inconsistent data, no assignment'
        n_inc = n_inc + 1
        storeno(n_res) = -r_num
      elseif ((n_input .gt. 0) .and. (n_swap .gt. 0)) then
        write(u_out,'(a,/)') '==> Both assignments possible'
        n_both = n_both + 1
        storeno(n_res) = r_num
      else
        write(u_out,'(a,/)') '==> Unambiguously assigned'
        n_ass = n_ass + 1
        if (n_input. gt. 0) then
          n_i = n_i + 1
          storeyes(n_res) = r_num
        else
          n_s = n_s + 1
          storeyes(n_res) = -r_num
        end if
      end if
      if ((n_input + n_swap) .gt. 20) then
        if (n_input .gt. (10*n_swap)) then
          n_inp_cr = n_inp_cr + 1
        elseif (n_swap .gt. (10*n_input)) then
          n_rev_cr = n_rev_cr + 1
        end if
      end if
      write(u_out,'(a)') 'Allowed region for PHI:'
      call PRINT_ANGLES(u_out, phi)
      write(u_out,'(a)') 'Allowed region for PSI:'
      call PRINT_ANGLES(u_out, psi)
      write(u_out,'(a)') 'Allowed region for CHI1:'
      call PRINT_ANGLES(u_out, chi1)
c
      write(u_out, '(a,//)') '========================================'
      rewind(u_db)
c
c
      end do
c
c
c  end of loop over residues
c
      write(u_out, '(//)')
      if (n_inc .gt. 0) then
        write(u_out, '(a)') 'Inconsistent data at residues: '
        call print_res(u_out, storeno, n_res, -1)
      end if
      if (n_both .gt. 0) then
        write(u_out, '(a)') 'Both assignments possible at residues: '
        call print_res(u_out, storeno, n_res, 1)
      end if
      if (n_i .gt. 0) then
        write(u_out, '(a)') 'Assigned as input at residues: '
        call print_res(u_out, storeyes, n_res, 1)
      end if
      if (n_s .gt. 0) then
        write(u_out, '(a)') 'Reverse assignments at residues: '
        call print_res(u_out, storeyes, n_res, -1)
      end if
      write(u_out, '(//)')
      write(u_out, '(a,/)') 'Statistics:'
      write(u_out, '(a, i3)') 'Inconsistent data:         ', n_inc
      write(u_out, '(a, i3)') 'Both assignments possible: ', n_both
      write(u_out, '(a, i3)') 'Assigned as input:         ', n_i
      write(u_out, '(a, i3)') 'Reverse assignments:       ', n_s
      write(u_out, '(a, i3)') '#_rev > #_inp:             ', n_rgti
      write(u_out, '(a, i3)') '#_inp > #_rev:             ', n_igtr
      write(u_out, '(a, i3)') '#conf>20, #_rev > 10 #_inp:', n_rev_cr
      write(u_out, '(a, i3)') '#conf>20, #_inp > 10 #_rev:', n_inp_cr
      write(u_out, '(a, i3)') 'Total assigned:            ', n_ass
      write(u_out, '(a, i3)') 'Number of Residues:        ', n_res
c
c
9999  continue
      stop
      end      
      logical function READ_DB(dbunit, base, errflag)
c  gets the values of distances and angles etc. from the database.
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          01-SEP-1988
c
c
      implicit none
c  global variables
      include 'const.inc'
c  input / output
      integer dbunit
      real base(-MAXREST:MAXREST)
      character*10 errflag
c  local variables
      character*80 line
      integer i
c
c  begin
c
      READ_DB = .true.
      read(dbunit,  err=111, end=112) (base(i), i=0,MAXREST)
      goto 113
 111  errflag = 'READ_DB   '
 112  READ_DB = .false.
 113  continue
      end
      logical function READ_RESTRAINTS(unit, r_name, r_num, n_cond, 
     2  base, i_data, i_swap, rel, err, mult, errflag)
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          26-AUG-1988
c
c
      implicit none
c  global variables
      include 'const.inc'
c  input / output
      integer unit                 ! unit of input file
      character*10 r_name          ! residue name
      integer r_num                ! residue number
      integer n_cond               ! number of constraints read
      real base(-MAXREST:MAXREST)  ! database and absolut value constr.
      integer i_data(2,MAXCOND),
     2        i_swap(2,MAXCOND)    ! index arrays
      real err(MAXCOND)            ! error estimates
      character*1 rel(MAXCOND)     ! relational operator
      integer mult(MAXCOND)        ! multiplicity for angle constraints
      character*10 errflag         ! error report
c  local variables
      logical eof, endres, empty, ok
      character*10 word, c_1, c_2, cerr, cmult, cdummy
      integer i_1, s_1, i_2, s_2, imult
      real rerr, c2
      character crel
c
c  begin
c
c  search first line of set of constraints for residue
c
      call getline(unit,eof)
      call get_word(word)
      do while ((word(1:4) .ne. 'RESI') .and. (.not. eof))
        call getline(unit,eof)
        call get_word(word)
      end do
      call get_word(r_name)
      call get_int(r_num)
c
      n_cond = 0
      endres = .false.
      call getline(unit,eof)
c
c  loop to read in the constraints,skipping empty lines and comments
c
      do while ((.not. eof) .and. (.not. endres))
c
c       read in constraints
c
        call get_word(c_1)
        endres = (c_1(1:3) .eq. 'END') 
        empty = ((c_1(1:1) .eq. ' ') 
     2        .or. (c_1(1:4) .eq. 'COMM')
     3        .or. (c_1(1:1) .eq. '!'))
        if ((endres) .or. (empty)) goto 8888
c
        call get_word(cdummy)
        call get_word(c_2)
        call get_word(cerr)
        call get_word(cmult)
c
c       check and convert constraints.
c       first, the index of c_1 is found, then the same is done with
c       swapped assignments. if no index for c_2 can be found, c_2 is 
c       converted a real number, and a negative index is 
c       assigned to it.
c
        ok = .true.
        call get_index(c_1,i_1, .false.)
        ok = ((ok) .and. (i_1 .gt. 0))
        call get_index(c_1,s_1, .true.)
        call get_index(c_2,i_2, .false.)
        if (i_2 .eq. 0) then 
          read(c_2,*,err=111) c2
          i_2 = -i_1
          s_2 = -i_1
          goto 112
 111      ok = .false.
 112      continue
        else
          call get_index(c_2,s_2, .true.)
        end if
        crel = cdummy(1:1)
        ok = ((ok) .and. 
     2        ((crel.eq.'<').or.(crel.eq.'=').or.(crel.eq.'>')) )
        rerr = 0.0
        if (cerr(1:1) .eq. '!') goto 113
        read(cerr,*,err=113) rerr
 113    continue
        imult = 0
        if (cmult(1:1) .eq. '!') goto 114
        read(cmult,*,err=114) imult
 114    continue
c
c       if all is fine, store the indices and values in the appropriate
c       places. note that if c_2 is an absolute value (then i_2 < 0)
c       it is stored in the array BASE (in the part that is not overwritten
c       with values from the database).
c
        if (ok) then
          n_cond = n_cond + 1
          i_data(1,n_cond) = i_1
          i_data(2,n_cond) = i_2
          i_swap(1,n_cond) = s_1
          i_swap(2,n_cond) = s_2
          if (i_2 .lt. 0) base(i_2) = c2
          err(n_cond) = rerr
          mult(n_cond) = imult
          rel(n_cond) = crel
        else 
          write(6,*) 'unable to process following line:'
          call putline(6)
        end if
c
 8888   call getline(unit,eof)
      end do                
c
      READ_RESTRAINTS = (n_cond .gt. 0)
c
      end
      subroutine get_index(const,index,swap)
c  this subroutine gets the number of the constraint const.
c  if const is not found in the list, index is set to 0.
c  if swap = true, the assignments for the beta preotons 
c  are swapped
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          27-AUG-1988
c    modified (GLY included)                          -OKT-1988
c    modified (arbitrary order in names)            03-NOV-1988
c
c
      implicit none
c  global variables
c  input / output
      character*10 const
      integer index
      logical swap
c  local variables
c
c  begin
c
      index = 0
c
      if     ((const .eq. 'DAN-      ')
     2 .or.   (const .eq. 'DNA-      ')) then
        index = 7
      elseif  (const .eq. 'DNN-      ')  then
        index = 8
      elseif ((const .eq. 'DAN       ')
     2 .or.   (const .eq. 'DNA       ')) then 
        index = 15
      elseif ((const .eq. 'DAN+      ') 
     2 .or.   (const .eq. 'DNA+      ')) then 
        index = 16
      elseif  (const .eq. 'DNN+      ')  then 
        index = 17
      elseif ((const .eq. 'JNA       ') 
     2 .or.   (const .eq. 'JAN       ')) then 
        index = 4
      elseif  (const .eq. 'PHI       ')  then 
        index = 1
      elseif  (const .eq. 'PSI       ')  then 
        index = 2
      elseif  (const .eq. 'CHI1      ')  then 
        index = 3
      endif
c
      if (.not. swap) then
        if     ((const .eq. 'DAB2      ') 
     2   .or.   (const .eq. 'DB2A      ')) then
          index = 10
        elseif ((const .eq. 'DAB3      ') 
     2   .or.   (const .eq. 'DB3A      ')) then
          index = 9
        elseif ((const .eq. 'DNB2      ') 
     2   .or.   (const .eq. 'DB2N      ')) then
          index = 12
        elseif ((const .eq. 'DNB3      ') 
     2   .or.   (const .eq. 'DB3N      ')) then
          index = 11
        elseif ((const .eq. 'DB2N+     ') 
     2   .or.   (const .eq. 'DNB2+     ')) then 
          index = 14
        elseif ((const .eq. 'DB3N+     ') 
     2   .or.   (const .eq. 'DNB3+     ')) then 
          index = 13
        elseif ((const .eq. 'JAB2      ') 
     2   .or.   (const .eq. 'JB2A      ')) then 
          index = 5
        elseif ((const .eq. 'JAB3      ') 
     2   .or.   (const .eq. 'JB3A      ')) then 
          index = 6
c  special case GLY
        elseif ((const .eq. 'DA2N      ') 
     2   .or.   (const .eq. 'DNA2      ')) then
          index = 10
        elseif ((const .eq. 'DA3N      ') 
     2   .or.   (const .eq. 'DNA3      ')) then
          index = 9
        elseif ((const .eq. 'DA2N+     ') 
     2   .or.   (const .eq. 'DNA2+     ')) then 
          index = 12
        elseif ((const .eq. 'DA3N+     ') 
     2   .or.   (const .eq. 'DNA3+     ')) then 
          index = 11
        elseif ((const .eq. 'JNA2      ') 
     2   .or.   (const .eq. 'JA2N      ')) then 
          index = 4
        elseif ((const .eq. 'JNA3      ') 
     2   .or.   (const .eq. 'JA3N      ')) then 
          index = 5
        endif
      else
        if     ((const .eq. 'DAB2      ') 
     2   .or.   (const .eq. 'DB2A      ')) then
          index = 9
        elseif ((const .eq. 'DAB3      ') 
     2   .or.   (const .eq. 'DB3A      ')) then
          index = 10
        elseif ((const .eq. 'DNB2      ') 
     2   .or.   (const .eq. 'DB2N      ')) then
          index = 11
        elseif ((const .eq. 'DNB3      ') 
     2   .or.   (const .eq. 'DB3N      ')) then
          index = 12
        elseif ((const .eq. 'DB2N+     ') 
     2   .or.   (const .eq. 'DNB2+     ')) then 
          index = 13
        elseif ((const .eq. 'DB3N+     ') 
     2   .or.   (const .eq. 'DNB3+     ')) then 
          index = 14
        elseif ((const .eq. 'JAB2      ') 
     2   .or.   (const .eq. 'JB2A      ')) then 
          index = 6
        elseif ((const .eq. 'JAB3      ') 
     2   .or.   (const .eq. 'JB3A      ')) then 
          index = 5
c  special case GLY
        elseif ((const .eq. 'DA2N      ') 
     2   .or.   (const .eq. 'DNA2      ')) then
          index = 9
        elseif ((const .eq. 'DA3N      ') 
     2   .or.   (const .eq. 'DNA3      ')) then
          index = 10
        elseif ((const .eq. 'DA2N+     ') 
     2   .or.   (const .eq. 'DNA2+     ')) then 
          index = 11
        elseif ((const .eq. 'DA3N+     ') 
     2   .or.   (const .eq. 'DNA3+     ')) then 
          index = 12
        elseif ((const .eq. 'JNA2      ') 
     2   .or.   (const .eq. 'JA2N      ')) then 
          index = 5
        elseif ((const .eq. 'JNA3      ') 
     2   .or.   (const .eq. 'JA3N      ')) then 
          index = 4
        endif
      endif
c
      return
      end
      logical function SATISFIED(i, n_cond, c_1, c_2, err, rel, mult)
c  this function tests if constraint c_1 is equal to, larger or
c  smaller (depending on character rel) within the error given 
c  by err. 
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          26-AUG-1988
c
c
      implicit none
c  global variables
c  input / output
      integer i, n_cond         ! actual and max. no. of conditions
      real c_1, c_2             ! constraints on noes etc
      real err                  ! error estimate
      character*1 rel           ! relation operator
      integer mult              ! multiplicity for angle constraints
c  local variables
      logical larger, smaller, equal, ok
      real c1, c2, d1, d2, angdist, angdist2
c
c  begin
c
      SATISFIED = .false.
      if (i .gt. n_cond) goto 9999
c
      c1 = c_1
      c2 = c_2
c
      if ((c1 .gt. 900) .or. (c2 .gt. 900)) goto 9999
c
      d1 = c1 - c2
c
      if (mult .ge. 1) then
        angdist = 360.0 / mult
        angdist2 = 180.0 / mult
        if (d1 .gt. 0.0) then
          do while (d1 .ge. angdist2)
            c1 = c1 - angdist
            d1 = c1 - c2
          end do
        else
          do while (-d1 .ge. angdist2)
            c1 = c1 + angdist
            d1 = c1 - c2
          end do
        end if
      end if        
      larger  = (d1 .gt. -err)
      smaller = (d1 .lt. +err)
      equal   = (larger .and. smaller)
c
      ok = (  (larger  .and. (rel .eq. '>'))  
     2   .or. (equal   .and. (rel .eq. '='))
     3   .or. (smaller .and. (rel .eq. '<')) )
      SATISFIED = ok
c
 9999 continue
      end
      subroutine store_angles(phi, aphi, swap)
c  stores the angles for which all the constraints are satisfied
c  in the character array aphi.
c  1:   satisfied for assignment given in input file
c  2:   satisfied for swapped assignment
c  3:   satisfied for both possible assignments, i. e. assignment
c           is not possible.
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          26-AUG-1988
c
c
      implicit none
c  global variables
      include 'const.inc'
c  input / output
      real phi
      integer aphi(NANG)
      logical swap
c  local variables
      integer i_phi, i_psi, i_chi1
c
c  begin
c
      if (phi .lt. 0) phi = phi + 360.0
      i_phi = nint(phi/STEP) + 1
      if (i_phi .gt. NANG) i_phi = i_phi - NANG
      if (i_phi .le. 0)    i_phi = i_phi + NANG
      if (swap) then
        if (aphi(i_phi) .eq. 1) then
          aphi(i_phi) = 3
        elseif (aphi(i_phi) .eq. 3) then
          aphi(i_phi) = 3
        else
          aphi(i_phi) = 2
        end if
      else
        if (aphi(i_phi) .eq. 2) then
          aphi(i_phi) = 3
        elseif (aphi(i_phi) .eq. 3) then
          aphi(i_phi) = 3
        else
          aphi(i_phi) = 1
        end if
      end if
c
      return
      end

      subroutine print_angles(unit, ang)
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          26-AUG-1988
c
c
      implicit none
c  global variables
      include 'const.inc'
c  input / output
      integer unit
      integer ang(NANG)
c  local variables
      character*80 result
      integer i,k,l, r_i, r_j, a_i, a_j
      logical found
      character*13 str(3)
      real firstphi, lastphi
      integer l_ang(NANG), nang2
c
c  begin
c
      str(1) = '   input:    '
      str(2) = '   swapped:  '
      str(3) = '   both:     '
c
c      nang2 = nint(NANG / 2.0)
c      do k=1, nang2
c        l_ang(k) = ang(nang2+k)
c      end do
c      do k=nang2+1, NANG
c        l_ang(k) = ang(k-nang2)
c      end do
c
      do k=1, NANG
        l_ang(k) = ang(k)
      end do
c
      do k=1,3
        do l=1,80
          result(l:l) = ' '
        end do
        r_i = 1
        r_j = r_i + 13 - 1
        write(result(r_i:r_j),'(a)') str(k)
        r_i = r_j + 1
c
        a_i = 0
        a_j = 0
        i = 0
        do while (i .lt. NANG)
          i = a_j + 1
          found = .false.
          do while ((i .le. NANG) .and. (.not. found))
            found = ((l_ang(i) .eq. k) .or. (l_ang(i) .eq. 3))
            i = i + 1
          end do
          a_i = i - 1
          if (.not. found) then
            a_i = NANG + 1
            a_j = a_i
          else
            found = .false.
            i = a_i + 1
            do while((i .le. NANG) .and. (.not. found))
              found = ((l_ang(i) .ne. k) .and. (l_ang(i) .ne. 3))
              i = i + 1
            end do
            if (found) then
              a_j = i - 2
            else
              a_j = i - 1
            end if
          end if
          if (a_i .le. NANG) then
            if (r_i .gt. 58) then
              write(unit, '(a)') result
              do l=1,80
                result(l:l) = ' '
              end do
              r_i = 14
            end if
            firstphi = (a_i - 1)*step
            lastphi  = (a_j - 1)*step
            if (firstphi .gt. 180.0) firstphi = firstphi - 360.0
            if (lastphi .gt. 180.0) lastphi = lastphi - 360.0
            r_j = r_i + 22 - 1
            write(result(r_i:r_j),'(f6.1,a5,f6.1,a5)') 
     2        firstphi - 0.5*step , ' ... ',
     3        lastphi  + 0.5*step , '     '
            r_i = r_j + 1
          end if
        end do
        write(unit, '(a)') result
c
      end do
c
      return
      end 





      subroutine print_res(unit, vector, n, flag)
c
c  Author: Michael Nilges, LCP, NIDDK, NIH          26-AUG-1988
c
c
      implicit none
c  global variables
      include 'const.inc'
c  input / output
      integer unit
      integer vector(*)
      integer n
      integer flag
c  local variables
      character*80 result
      integer i,k,l, r_i, r_j, a_i, a_j
c
c  begin
c
c
      do l=1,80
        result(l:l) = ' '
      end do
      r_i = 1
      r_j = 2
c
      if (flag .gt. 0) then
        do k=1, n
          if (vector(k) .gt. 0) then
            r_j = r_i + 5
            if (r_j .gt. 80) then
              write(unit, '(a)') result
              do l=1,80
                result(l:l) = ' '
              end do
              r_i = 1
              r_j = r_i + 5
            end if
            write(result(r_i:r_j),'(i3,a2)') abs(vector(k)), ', '
            r_i = r_j + 1
          end if
        end do
        result(r_j-2:r_j-1) = '  '
        write(unit, '(a)') result
      elseif (flag .lt. 0) then
        do k=1, n
          if (vector(k) .lt. 0) then
            r_j = r_i + 5
            if (r_j .gt. 80) then
              write(unit, '(a)') result
              do l=1,80
                result(l:l) = ' '
              end do
              r_i = 1
              r_j = r_i + 5
            end if
            write(result(r_i:r_j),'(i3,a2)') abs(vector(k)), ', '
            r_i = r_j + 1
          end if
        end do
        result(r_j-2:r_j-1) = '  '
        write(unit, '(a)') result
      end if
c
      return
      end 
