/* fastq.c - FASTQ sequence file functions */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <err.h>
#include <stdlib.h>
#include <string.h>

#include "fastq.h"


int fastq_init(fqseq_t *seq) {

  seq->lnam = 0; seq->snam = NULL;
  seq->lseq = 0; seq->sseq = NULL;
  seq->lqua = 0; seq->squa = NULL;

  return 0; }


int fastq_fini(fqseq_t *seq) {

  free(seq->snam);
  free(seq->sseq);
  free(seq->squa);

  return 0; }


void fastq_null(fqseq_t *seq) {

  if (seq == NULL) { return; }
  if (seq->lnam > 0) { *seq->snam = '\0'; }
  if (seq->lseq > 0) { *seq->sseq = '\0'; }
  if (seq->lqua > 0) { *seq->squa = '\0'; }

  return; }


int fastq_read(FILE *f, fqseq_t *seq) {
  char *p, buf[1024];
  int i, n = 4, s;
  size_t len, nlen, slen, qlen;

  fastq_null(seq);

  s = 0; nlen = slen = qlen = 0;
  while (fgets(buf, (int)sizeof(buf), f) != NULL) {

    if (*buf == '\n' && s == 0) continue;
    len = strlen(buf); p = buf + len;
    if (*(p-1) == '\n') len--;

    switch (n) {
    case 4:        /* sequence name */
      if (s == 0 && *buf != '@')
	errx(EXIT_FAILURE, "sequence: bad header line");
      i = (s == 0 && *buf == '@') ? 1 : 0;
      if (seq->lnam < nlen + len) {
	seq->lnam += len;
	seq->snam = realloc(seq->snam, seq->lnam + 1);
	if (seq->snam == NULL)
	  err(EXIT_FAILURE, "malloc failed"); }
      p = seq->snam + nlen;
      (void)memcpy(p, buf + i, len - i);
      nlen += len - i; p += len - i; *p = '\0';
      break;
    case 3:        /* sequence string */
      if (seq->lseq < slen + len) {
	seq->lseq += len;
	seq->sseq = realloc(seq->sseq, seq->lseq + 1);
	if (seq->sseq == NULL)
	  err(EXIT_FAILURE, "malloc failed"); }
      p = seq->sseq + slen;
      (void)memcpy(p, buf, len);
      slen += len; p += len; *p = '\0';
      break;
    case 2:        /* quality name */
      if (s == 0 && *buf != '+')
	errx(EXIT_FAILURE, "quality: bad header line");
      break;
    case 1:        /* quality string */
      if (seq->lqua < qlen + len) {
	seq->lqua += len;
	seq->squa = realloc(seq->squa, seq->lqua + 1);
	if (seq->squa == NULL)
	  err(EXIT_FAILURE, "malloc failed"); }
      p = seq->squa + qlen;
      (void)memcpy(p, buf, len);
      qlen += len; p += len; *p = '\0';
      break;
    default:
      errx(EXIT_FAILURE, "fastq: invalid format"); }

    p = buf + len;
    if (*p != '\n') { s++; continue; }
    n--; s = 0;
    if (n == 0) break; }
  if (ferror(f) != 0) return -1;

  if (seq->sseq == NULL || seq->squa == NULL || slen != qlen)
    errx(EXIT_FAILURE, "fastq: bad sequence/quality lines");

  return 0; }


int fastq_write(FILE *f, fqseq_t *seq) {
  int res;

  if (seq == NULL || seq->snam == NULL || *seq->snam == '\0') return 0;

  res = fprintf(f, "@%s\n%s\n+\n%s\n", seq->snam, seq->sseq, seq->squa);

  return res; }
