
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <err.h>
#include <libgen.h>
#include <limits.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "compat.h"
#include "fastq.h"
#include "xfile.h"

static void usage(char *);
static void fastq_scoring(const char *, uint64_t, int *, int *, int *);


int main(int argc, char **argv) {
  FILE *ou;
  char *p, *prg, *ofile;
  int i, qmin, qmax;
  int qfmt, dsco, off;
  size_t n;
  int64_t tmp;
  uint64_t obj;
  fqfmt_t sfmt, dfmt;

  char *q, *s, *xbuf;
  ptrdiff_t dif;
  size_t xfer, xinc, xmax;
  ssize_t ssiz, rsiz;
  uint64_t lig;
  xfile_t *xf;

  /* Inits */
  prg = basename(*argv);
  obj = 0;
  sfmt = fqfmts[0];
  dfmt = fqfmts[1];
  dsco = dfmt.fmt;
  ofile = NULL;
  ou = stdout;

  while ((i = getopt(argc, argv, "dhln:o:")) != -1) {
    switch (i) {
    case 'd':
      /* Detect only, disable conversion */
      dsco = FASTQ_NONE; break;
    case 'h':
      usage(prg); return EXIT_SUCCESS;
    case 'l':
      /* List known scoring systems */
      for (n = 1; n < __arraycount(fqfmts); n++) {
	printf("%s (Phred+%d, values %d-%d)\n", fqfmts[n].dsc,
	       fqfmts[n].off, fqfmts[n].min, fqfmts[n].max); }
      return EXIT_SUCCESS;
    case 'n':
      /* Restrict detection to max object number */
      tmp = strto64(optarg, &p, 10);
      if (*p != '\0' || tmp < 0)
        errx(1, "-%c: not a positive integer", i);
      obj = (uint64_t)tmp; break;
    case 'o':
      ofile = optarg; break;
    default:
      usage(prg); return EXIT_FAILURE; }
  }
  if (argc - optind < 1) {
    usage(prg); return EXIT_FAILURE; }

  if (ofile != NULL && (ou = fopen(ofile, "w")) == NULL)
    err(EXIT_FAILURE, "%s: open failed", ofile);

  /* Proceed all sequence files */
  for (i = optind; i < argc; i++) {
    const char *ifile = *(argv+i);
    char c;

    /* Detect qualities scoring system ... */
    fastq_scoring(ifile, obj, &qfmt, &qmin, &qmax);
    (void)fprintf(stderr, "%s: Found", ifile ); c = ' ';
    for (n = 0; n < __arraycount(fqfmts); n++) {
      if ((qfmt & fqfmts[n].fmt) == 0) continue;
      (void)fprintf(stderr, "%c%s", c, fqfmts[n].dsc); c = ',';
      sfmt = fqfmts[n]; }
    if (qfmt == 0 && n == __arraycount(fqfmts))
      (void)fprintf(stderr, "%c%s", c, fqfmts[0].dsc);
    (void)fprintf(stderr, " qualities (%d-%d)\n", qmin, qmax);

    if (qfmt == FASTQ_NONE || dsco == FASTQ_NONE) { continue; }
    off = dfmt.off - sfmt.off;

    /* Convert ... */
    if ((xf = xfile_open(ifile, "r")) == NULL)
      err(EXIT_FAILURE, "%s: open failed", ifile);
    xfile_xfer(&xfer, xf); xinc = xfer; xmax = 10 * xinc;
    if ((xbuf = malloc(xfer+1)) == NULL)
      err(EXIT_FAILURE, "memory allocation failed");
    ssiz = rsiz = 0; lig = 0;
    while (/*CONSTCOND*/1) {
      ssiz = xfile_read(xf, xbuf + rsiz, xfer - (size_t)rsiz);
      if (ssiz < 1) break;

      rsiz += ssiz; p = xbuf;
      while (/*CONSTCOND*/ 1) {

	q = memchr(p, '\n', (size_t)rsiz);
	if (q == NULL) { break; }
	dif = q - p + 1; s = p; p = q + 1; lig++;
	rsiz -= dif;

	if (lig % 4 == 0) { /* Quality string */
	  q = s; while (q < p) {
	    *q += off;
	    if (*q < dfmt.min) { *q = dfmt.min; }
	    if (*q > dfmt.max) { *q = dfmt.max; }
	    q++;
	  }
	}
	(void)fprintf(ou, "%.*s\n", (int)dif - 1, s);
      }

      if (rsiz == 0) continue;
      (void)memmove(xbuf, p, (size_t)rsiz);
      if (p != xbuf) continue;
      if (xfer == xmax)
	errx(EXIT_FAILURE, "%s: line too long", ifile);
      xfer += xinc;
      if ((xbuf = realloc(xbuf, xfer+1)) == NULL)
	err(EXIT_FAILURE, "realloc failed");
    }
    if (xfile_error(xf) != 0)
      err(EXIT_FAILURE, "%s: read failed", ifile);
    if (lig % 4 != 0)
      errx(EXIT_FAILURE, "%s: truncated file", ifile);

    free(xbuf);
    if (xfile_close(xf) == EOF)
      err(EXIT_FAILURE, "%s: close failed", ifile);

  }

  if (ofile != NULL && fclose(ou) == EOF)
    err(EXIT_FAILURE, "%s: close failed", ofile);

  return EXIT_SUCCESS; }


static void usage(char *prg) {
  FILE *f = stderr;

  (void)fprintf(f, "usage: %s [options] <file> [...]\n", prg);
  (void)fprintf(f, "\noptions:\n");
  (void)fprintf(f, "  -d        ... Do not convert, detect only.\n");
  (void)fprintf(f, "  -h        ... Print this message and exit.\n");
  (void)fprintf(f, "  -l        ... List known scoring systems.\n");
  (void)fprintf(f, "  -n <num>  ... Restrict detection to <num> objects.\n");
  (void)fprintf(f, "  -o <file> ... Save output to <file>.\n");

  return; }


/* Detect FastQ scoring format */
static void fastq_scoring(const char *file, uint64_t obj, int *fmt, int *min,
		   int *max) {
  int qmin, qmax, qfmt;
  char *p, *q, *s, *xbuf;
  ptrdiff_t dif;
  size_t n, xfer, xinc, xmax;
  ssize_t ssiz, rsiz;
  uint64_t lig, cnt;
  xfile_t *xf;

  qmin = INT_MAX; qmax = INT_MIN;
  qfmt = FASTQ_UNKNOWN;

  if ((xf = xfile_open(file, "r")) == NULL)
    err(EXIT_FAILURE, "%s: open failed", file);
  xfile_xfer(&xfer, xf); xinc = xfer; xmax = 10 * xinc;

  if ((xbuf = malloc(xfer+1)) == NULL)
    err(EXIT_FAILURE, "memory allocation failed");
  ssiz = rsiz = 0; lig = 0; cnt = 0; n = 0;
  while (/*CONSTCOND*/1) {
    ssiz = xfile_read(xf, xbuf + rsiz, xfer - (size_t)rsiz);
    if (ssiz < 1) break;

    rsiz += ssiz; p = xbuf;
    while (/*CONSTCOND*/ 1) {

      q = memchr(p, '\n', (size_t)rsiz);
      if (q == NULL) { break; }
      dif = q - p + 1; s = p; p = q + 1; lig++;
      rsiz -= dif;

      if (lig % 4 != 0) { continue; } /* Not a quality string */
      while (--dif > 0) {
	if (*s < qmin) { qmin = *s; }
	if (*s > qmax) { qmax = *s; }
	s++; }
      if (obj != 0 && ++cnt == obj) { break; }
    }
    if (obj != 0 && cnt == obj) { break; }

    if (rsiz == 0) continue;
    (void)memmove(xbuf, p, (size_t)rsiz);
    if (p != xbuf) continue;
    if (xfer == xmax)
      errx(EXIT_FAILURE, "%s: line too long", file);
    xfer += xinc;
    if ((xbuf = realloc(xbuf, xfer+1)) == NULL)
      err(EXIT_FAILURE, "realloc failed");
  }

  if (xfile_error(xf) != 0)
    err(EXIT_FAILURE, "%s: read failed", file);
  if (lig % 4 != 0)
    errx(EXIT_FAILURE, "%s: truncated file", file);

  free(xbuf);
  if (xfile_close(xf) == EOF)
    err(EXIT_FAILURE, "%s: close failed", file);

  for (n = 0; n < __arraycount(fqfmts); n++) {
    if ((qfmt & fqfmts[n].fmt) == 0) continue;
    if (qmin < fqfmts[n].min || qmax > fqfmts[n].max) {
      qfmt &= ~(fqfmts[n].fmt); continue; }
    qfmt &= ~(fqfmts[n].dis); }

  *fmt = qfmt; *min = qmin; *max = qmax;

  return; }
