/* xfile.c - XFILE file functions */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/statvfs.h>

#include <fcntl.h>
#include <limits.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <zlib.h>

#include "xfile.h"

#define GZIP_MAGIC "\037\213"
#define GZIP_OFFSET 0x00
#define GZIP_LENGTH 0x02


static compress_t xfile_detect(const char *file) {
  int i, fd;
  uint8_t buf[2];

  /* Check gzip compression ... */
  fd = open(file, O_RDONLY, 0666);
  if (fd == -1) { return COMPRESS_UNKNOWN; }
  (void)lseek(fd, GZIP_OFFSET, SEEK_SET);
  (void)read(fd, buf, GZIP_LENGTH);
  (void)close(fd);
  i = memcmp(buf, GZIP_MAGIC, GZIP_LENGTH);
  if (i == 0) { return COMPRESS_GZIP; }

  /* No compression ... */
  return COMPRESS_NONE; }


xfile_t *xfile_open(const char *file, const char *mode) {
  int fd;
  xfile_t *fq = NULL;
  struct statvfs stv;

  fq = malloc(sizeof(*fq));
  if (fq == NULL) { return NULL; }
  fq->typ = xfile_detect(file);
  fq->xfer = 512;

  fd = open(file, O_RDONLY, 0666);
  if (fd == -1) { free(fq); return NULL; }
  if (fstatvfs(fd, &stv) != -1)
    fq->xfer = stv.f_iosize;

  switch (fq->typ) {
  case COMPRESS_GZIP:
    fq->des.gd = gzdopen(dup(fd), mode); break;
  case COMPRESS_NONE:
    fq->des.fd = fdopen(dup(fd), mode); break;
  default:
    free(fq); fq = NULL; break; }

  (void)close(fd);

  return fq; }


int xfile_close(xfile_t *fq) {
  int res;

  switch (fq->typ) {
  case COMPRESS_GZIP:
    res = gzclose(fq->des.gd); break;
  case COMPRESS_NONE:
    res = fclose(fq->des.fd); break;
  default:
    res = -1; break; }

  free(fq);

  return res; }


void xfile_xfer(size_t *xfer, xfile_t *fq) {

  *xfer = fq->xfer;

  return; }


int xfile_size(off_t *off, xfile_t *fq) {
  char buf[1024];
  int res;
  off_t siz = 0;
  z_off_t zoff;

  struct stat st;

  switch (fq->typ) {
  case COMPRESS_GZIP:
    zoff = gzseek(fq->des.gd, 0, SEEK_CUR);
    gzrewind(fq->des.gd);
    while (/*CONSTCOND*/1) {
      res = gzread(fq->des.gd, buf, (unsigned int)sizeof(buf));
      if (res < 1) { break; }
      siz += res; }
    gzseek(fq->des.gd, zoff, SEEK_SET);
    break;
  case COMPRESS_NONE:
    res = fstat(fileno(fq->des.fd), &st);
    siz = st.st_size;
    break;
  default:
    res = -1; break; }

  if (res == 0) { *off = siz; }

  return res; }


ssize_t xfile_read(xfile_t *fq, void *buf, size_t len) {
  ssize_t siz;

  switch (fq->typ) {
  case COMPRESS_GZIP:
    len = (len > UINT_MAX) ? UINT_MAX : len;
    siz = gzread(fq->des.gd, buf, (unsigned int)len); break;
  case COMPRESS_NONE:
    siz = read(fileno(fq->des.fd), buf, len); break;
  default:
    siz = 0; break; }

  return siz; }


off_t xfile_seek(xfile_t *fq, off_t off, int whe) {
  off_t res;

  switch (fq->typ) {
  case COMPRESS_GZIP:
    res = gzseek(fq->des.gd, off, whe); break;
  case COMPRESS_NONE:
    res = lseek(fileno(fq->des.fd), off, whe); break;
  default:
    res = -1; break; }

  return res; }


char *xfile_gets(char *buf, int siz, xfile_t *fq) {
  char *str;

  switch (fq->typ) {
  case COMPRESS_GZIP:
    str = gzgets(fq->des.gd, buf, siz); break;
  case COMPRESS_NONE:
    str = fgets(buf, siz, fq->des.fd); break;
  default:
    str = NULL; break; }

  return str; }


int xfile_eof(xfile_t *xf) {
  int res;

  switch (xf->typ) {
  case COMPRESS_GZIP:
    res = gzeof(xf->des.gd); break;
  case COMPRESS_NONE:
    res = feof(xf->des.fd); break;
  default:
    res = -1; break; }

  return res; }


int xfile_error(xfile_t *xf) {
  int res, num;

  switch (xf->typ) {
  case COMPRESS_GZIP:
    (void)gzerror(xf->des.gd, &num);
    res = (num < 0); break;
  case COMPRESS_NONE:
    res = ferror(xf->des.fd); break;
  default:
    res = -1; break; }

  return res; }
