/* index.c - Golden indexes functions */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "error.h"
#include "index.h"

#ifndef DATADIR
#define DATADIR "/usr/local/share"PACKAGE
#endif

/* Functions prototypes */
static int index_compare(const void *a, const void *b);


/* Get golden indexes directory */
char *index_dir(void) {
  char *p;

  if ((p = getenv("GOLDENDATA")) == NULL) {
    p = (char *)DATADIR; }

  return p; }


/* Get golden index file name */
char *index_file(const char *dir, const char *dbase, const char *suf) {
  const char *p;
  char *file;
  size_t len;

  if ((p = dir) == NULL) { p = index_dir(); }

  len = strlen(p) + 1 + strlen(dbase) + 1 + strlen(suf);
  if ((file = (char *)malloc(len+1)) == NULL) {
    error_fatal("memory", NULL); }
  (void)sprintf(file, "%s/%s.%s", p, dbase, suf);

  return file; }


/* Search database indexes */
result_t *index_search(char *file, char *name) {
  FILE *f;
  int i;
  long min, cur, max, pos, indnb;
  size_t len;
  indix_t inx;
  result_t *res;

  /* Checks for valid query name */
  if (strlen(name) > NAMLEN) {
    error_fatal(name, "query name too long"); }

  if ((f = fopen(file, "r")) == NULL) {
    error_fatal(file, NULL); }
  if (fread(&indnb, sizeof(long), 1, f) != 1) {
    error_fatal(file, NULL); }

  min = 0; max = indnb - 1;
  while(min <= max) {

    /* Set current position */
    cur = (min + max) / 2;
    pos = sizeof(long) + cur * sizeof(indix_t);
    if (fseek(f, pos, SEEK_SET) == -1) {
      error_fatal(file, NULL); }

    /* Check current name */
    if (fread(&inx, sizeof(indix_t), 1, f) != 1) {
      error_fatal(file, NULL); }
    if ((i = strcmp(name, inx.name)) == 0) break;

    /* Set new limits */
    if (i < 0) { max = cur - 1; }
    if (i > 0) { min = cur + 1; }

  }

  if (fclose(f)) {
    error_fatal(file, NULL); }

  /* Not found */
  if (i != 0) { return NULL; }

  len = sizeof(result_t);
  if ((res = (result_t *)malloc(len)) == NULL) {
    error_fatal("memory", NULL); }
  res->name = strdup(inx.name);
  res->dbase = NULL;
  res->filenb = inx.filenb;
  res->offset = inx.offset;

  return res; }


/* Merge indexes */
int index_merge(char *file, long nb, indix_t *ind) {
  FILE *f, *g;
  int i;
  char *new;
  size_t len;
  long newnb, oldnb, totnb;
  indix_t *cur, old;

  /* Sort indexes */
  qsort(ind, (size_t)nb, sizeof(indix_t), index_compare);
  newnb = nb; totnb = 0; cur = ind;

  /* Create missing empty index file */
  if (access(file, F_OK) != 0) {
    if ((g = fopen(file, "w")) == NULL) {
      error_fatal(file, NULL); }
    if (fwrite(&totnb, sizeof(long), 1, g) != 1) {
      error_fatal(file, NULL); }
    if (fclose(g)) {
      error_fatal(file, NULL); }
  }

  if (nb == 0) { return 0; }

  /* Make new file */
  len = strlen(file) + 4;
  if ((new = (char *)malloc(len+1)) == NULL) {
    error_fatal("memory", NULL); }
  (void)sprintf(new, "%s.new", file);
  if ((f = fopen(new, "w")) == NULL) {
    error_fatal(new, NULL); }
  if (fwrite(&totnb, sizeof(long), 1, f) != 1) {
    error_fatal(new, NULL); }

  /* Process old index file & merge with new */
  if ((g = fopen(file, "r")) == NULL) {
    error_fatal(file, NULL); }
  if (fread(&oldnb, sizeof(long), 1, g) != 1) {
    error_fatal(file, NULL); }
  while(oldnb) {
    if (fread(&old, sizeof(indix_t), 1, g) != 1) {
      error_fatal(file, NULL); }

    /* Insert new entries */
    while(newnb && (i = index_compare(cur, &old)) < 0) {
      if (fwrite(cur, sizeof(indix_t), 1, f) != 1) {
	error_fatal(new, NULL); }
      newnb--; cur++; totnb++; }

    /* Update existing entries */
    if (newnb && i == 0) {
      if (fwrite(cur, sizeof(indix_t), 1, f) != 1) {
	error_fatal(new, NULL); }
      newnb--; oldnb--; cur++; totnb++; continue; }

    /* Write old entry */
    if (fwrite(&old, sizeof(indix_t), 1, f) != 1) {
      error_fatal(new, NULL); }
    oldnb--; totnb++; }

  if (fclose(g)) {
    error_fatal(file, NULL); }

  /* Add new remaining indexes */
  while(newnb) {
    if (fwrite(cur, sizeof(indix_t), 1, f) != 1) {
      error_fatal(new, NULL); }
    newnb--; cur++; totnb++; }

  /* Write entries number */
  if (fseek(f, (long)0, SEEK_SET) == -1) {
    error_fatal(new, NULL); }
  if (fwrite(&totnb, sizeof(long), 1, f) != 1) {
    error_fatal(new, NULL); }

  if (fclose(f)) {
    error_fatal(new, NULL); }

  /* Rename file */
  if (rename(new, file) == -1) {
    error_fatal(new, NULL); }

  free(new);

  return 0; }


/* Compare indexes by names */
static int index_compare(const void *a, const void *b) {
  const indix_t *p, *q;

  p = (const indix_t *)a; q = (const indix_t *)b;

  return strncasecmp(p->name, q->name, NAMLEN); }
