/* golden.c - Golden retriever main functions */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>

#include <stdio.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <ctype.h>
#include <libgen.h>

#include "access.h"
#include "entry.h"
#include "error.h"
#include "list.h"
#include "locus.h"


#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1
#endif

#ifndef HAVE_FSEEKO
#define fseeko fseek
#define off_t long
#endif


/* Functions prototypes */
static void usage(int);

/* Global variables */
static char *prog;


/* Main function */
int main(int argc, char **argv) {
  FILE *f, *g;
  int i, loc, acc, chk;
  char *p, *q, *dbase, *name, *file;
  size_t len;
  result_t *res;

  /* Inits */
  prog = basename(*argv);

  /* Checks command line options & arguments */
  i = loc = acc = chk = 0; file = NULL;
  while((i = getopt(argc, argv, "achilo:")) != -1) {
    switch(i) {
    case 'a':
      acc = 1; break;
    case 'c':
      chk = 1; break;
    case 'h':
      usage(EXIT_SUCCESS); break;
    case 'i':
      loc = 1; break;
    case 'l':
      if (list_check()) {
	error_fatal("databases", "cannot retrieve list"); }
      return EXIT_SUCCESS;
    case 'o':
      file = optarg; break;
    default:
      usage(EXIT_FAILURE); break; }
  }
  if ((loc + acc) == 0) { loc = acc = 1; }
  if (argc - optind != 1) {
    usage(EXIT_FAILURE); }

  /* Set output stream */
  g = stdout;
  if (file != NULL && (g = fopen(file, "w")) == NULL) {
    error_fatal(file, NULL); }

  /* Parse database & query names */
  p = argv[optind]; len = strlen(p);
  if (strchr(p, ':') == NULL)
    error_fatal(p, "invalid query value");
  if ((dbase = (char *)malloc(len+1)) == NULL ||
      (name = (char *)malloc(len+1)) == NULL) {
    error_fatal("memory", NULL); }
  q = dbase; while(*p && *p != ':') *q++ = *p++; *q = '\0'; p++;
  q = name; while(*p) *q++ = toupper((unsigned char)*p++); *q = '\0';

  /* Search name in databases indexes */
  if (acc) {
    res = access_search(dbase, name);
    if (res != NULL) loc = 0; }
  if (loc) {
    res = locus_search(dbase, name); }
  if (res == NULL) {
    fprintf(stderr, "%s: no entry found.\n", argv[optind]);
    free(name); free(dbase);
    return EXIT_SUCCESS; }

  /* Get database flat file name */
  p = list_name(res->dbase, res->filenb);

  /* Display database entry */
  if (chk == 0) {
    if ((f = fopen(p, "r")) == NULL) {
      error_fatal(p, NULL); }
    if (fseeko(f, res->offset, SEEK_SET) != 0) {
      error_fatal(p, NULL); }
    if (entry_display(f, g)) {
      error_fatal(p, NULL); }
    if (fclose(f) == EOF) {
      error_fatal(p, NULL); }
    free(p); }

  free(res->name); free(res->dbase);
  free(res);

  free(name); free(dbase);

  if (file != NULL && fclose(g) == EOF) {
    error_fatal(file, NULL); }

  return EXIT_SUCCESS; }


/* Usage display */
static void usage(int status) {
  FILE *f = stdout;

  (void)fprintf(f, "usage: %s [options] <dbase:name>\n\n", prog);
  (void)fprintf(f, "options:\n");
  (void)fprintf(f, "  -a        ... Search query by accession number.\n");
  (void)fprintf(f, "  -c        ... Check query.\n");
  (void)fprintf(f, "  -h        ... Prints this message and exit.\n");
  (void)fprintf(f, "  -i        ... Search query by entry name.\n");
  (void)fprintf(f, "  -l        ... List available databases.\n");
  (void)fprintf(f, "  -o <file> ... Place output into <file>.\n");

  exit(status); }
