"""
Content all the Mobyle Parameter types 
"""
 

import os , os.path , re
import shutil
import types
import logging

import Mobyle.SequenceConverter
import Mobyle.AlignmentConverter

from Mobyle.MobyleError import MobyleError , UserValueError , UnDefAttrError

c_log = logging.getLogger( 'mobyle.core' )
b_log = logging.getLogger('mobyle.builder')



def safeMask(  mask ):
    import string
    for car in mask :
        if car not in string.printable :
           mask = mask.replace( car , '_')
    
    #don't use re.UNICODE because safeFileName don't permit to use unicode char.
    #we must work whit the same char set    
    mask = re.sub( "(;|`|\{|\}|\s)+" , '_' , mask )
    
    mask = re.sub( "^.*[\\\:/]", "" ,  mask  )
    return mask



class DataType( object ):

    def __init__( self , name = None ):
        if name:
            self.name = name
        else:
            self.name = self.__class__.__name__[0:-8]
        
        self.ancestors = [ k.__name__[0:-8] for k in self.__class__.mro() ][0:-2]

        if self.name not in self.ancestors :
            self.ancestors.insert( 0 , self.name )
        
            
    def isPipableToDataType( self , targetDataType ):
                
        return targetDataType.name in self.ancestors

    def getName( self ):
        return self.name
    
    @staticmethod
    def isFile():
        return False
        
    
    def toDom( self ):
        """
        @return: a dom representation of this datatype
        @rtype: Ft.Xml.Domlette.Element
        """
        import Ft.Xml , Ft.Xml.Domlette
        from Ft.Xml import EMPTY_NAMESPACE
        
        
        if self.name == self.__class__.__name__[0:-8] :
            klass = self.name
            superKlass = None 
        else:
            klass = self.name
            superKlass = self.__class__.__name__[0:-8]
            
        doc = Ft.Xml.Domlette.implementation.createDocument( Ft.Xml.EMPTY_NAMESPACE , "datatype" , None )
        
        klassNode = doc.createElementNS( EMPTY_NAMESPACE , "class" )
        klassNodeText = doc.createTextNode( klass )
        klassNode.appendChild( klassNodeText )
        doc.documentElement.appendChild( klassNode )
        
        if superKlass :
            superKlassNode = doc.createElementNS( EMPTY_NAMESPACE , "superclass" )
            superKlassNodeText = doc.createTextNode( superKlass )
            superKlassNode.appendChild( superKlassNodeText )
            doc.documentElement.appendChild( superKlassNode )
        
        return doc.documentElement 

            
        
class DataTypeFactory( object ):
    
    _ref = None
    definedDataTypes = {}
    
    
    def __new__( cls ):
        if cls._ref is None:
            cls._ref = super( DataTypeFactory , cls ).__new__( cls )
        return cls._ref
    
    
    def newDataType( self , name , xmlName = None ):
        """
        @param name: the value of element superclass or class if there is no superclass
        @type name: string
        @param xmlName: the value of element class when the element superclass is specify
        @type xmlName: string
        @return: an instance of datatype according to the name and xmlName
        @rtype: a Datatype instance   
        """
        if xmlName:
            realName = xmlName + "DataType"
        else:
            try:
                realName = name + "DataType"
            except TypeError:
                raise MobyleError , "the argument \"name\" must be a string ( %s received )" %str( type( name )) 
        if self.definedDataTypes.has_key( realName ):
            dt = self.definedDataTypes[ realName ]
            
            if( dt.__class__.__name__[0:-8] != name ):
                #c_log.error("consistency error:")
                raise MobyleError , "consistency error: a \"%s\" is already defined with python type \"%s\" instead of \"%s\"" %( dt.getName() ,
                                                                                                                                 dt.__class__.__name__[0:-8] ,
                                                                                                                                 name 
                                                                                                                                 )
            
            return dt
        
        else:
            if not xmlName :
                s = name + 'DataType()'
            else:
                s = name + 'DataType( name = "' + str( xmlName ) + '")'
            try:
                self.definedDataTypes[ realName ] = eval( s )
            except NameError:
                #the import is not at top level to avoid cyclic import
                #between Core.py and Local.CustomClasses
                import Local.CustomClasses  
                s = "Local.CustomClasses." + s
                self.definedDataTypes[ realName ] = eval( s )

            return self.definedDataTypes[ realName ]



    def issubclass(self , dataType1 , name2 , xmlName2= None ):
        """
        @param dataType1: the dataType to test
        @type dataType1: instance of a Datatype
        @param name2: the value of element superclass or class if there is no superclass
        @type name2:  string
        @param xmlName2: the value of element class when the element superclass is specify
        @type xmlName2:  string
        @return: True if dataType1 is an instance of the datatype represente by name2 , xmlName2. False otherwise  
        @rtype: boolean
        """
        dataType2 = self.newDataType( name2 , xmlName= xmlName2 )
        try:
            return  issubclass( dataType1.__class__ , dataType2 .__class__ )
        except AttributeError , err :
            raise TypeError , "there is no DataType named "+ str( name2 )
    


class DataTypeTemplate( DataType ):

    @staticmethod
    def convert(value):
        return "DataTypeTemplate convert: " + str( value )
    
    @staticmethod    
    def validate( param ):
        return "DataTypeTemplate validate "
        
        
        
class BooleanDataType( DataType ):
    """
    """
    @staticmethod
    def convert( value , param ):
        """
        cast the value in a Boolean.
        The values: "off", "false" ,"0" or '' (case insensitive) are false,
        all others values are True.  
        @param value: the value provide by the User for this parameter
        @type value: String
        @return: the value converted in Boolean
        """
        #type controls
        value_type = type( value )
        if value is None:
            # in html form boolean appear as checkbox
            # if the chexkbox is not selected
            # the parameter is not send in the request
            return False
        if value_type == type( True ):
            return value
        
        elif value_type == types.StringType or value_type == types.UnicodeType :
            value = value.lower()
            if value == "off" or value == "false" or value == "0" or value == '' or value == "''" or value == '""':
                return False
            elif value == "on" or value == "true" or value == "1" :
                return True
        else:
            msg = "invalid value: " + str( value ) + "is not a boolean (allowed value: off, false, 0, on, true 1"
            raise UserValueError( parameter = param , msg = msg)



    @staticmethod
    def validate( param ):
        """
        do type controls. if the value is True or False or None return True,
        otherwise return False.        
        """
        value = param.getValue()
        
        if value == True or value == False:
            return True
    
        else:
            return False


        
class IntegerDataType( DataType ):


    @staticmethod
    def convert( value , param ):
        """
        Try to cast the value in integer. the allowed values are digit
        and strings.
        
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in Integer.
        @rtype: int
        @raise UserValueError: if the cast fail a UserValueError is raised.
        Unlike python, this method convert "8.0" in 8 and
        raised a UserValueError if you try to convert 8.2 .
        """
        if value is None:
            return None
        
        #type controls
            # int("8.0") failed and a  ValueError is raised
            # int(8.1) return 8
        
        try:
            f= float( value )
            i = int( f  )
            if ( (f - i) == 0):
                 return i 
            else:
                msg = "\"%s\" : this parameter must be an integer" %value
                raise  UserValueError( parameter = param , msg = msg)
        except OverflowError ,err:
            raise UserValueError( parameter = param , msg = "this value is too big" )
        except ( ValueError , TypeError ):
            msg = "\"%s\" : this parameter must be an integer" %value
            raise UserValueError( parameter = param , msg = msg)


    @staticmethod
    def validate( param ):
        """
        @return True if the value is an integer, False othewise.
        """   
        value = param.getValue()

        if value is None:
            return True                       
        try:
            int( value )
        except ( TypeError , ValueError ):
            return False



class FloatDataType( DataType ):


    @staticmethod
    def convert( value , param ):
        """
        Try to cast the value in float.
        
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in Float
        @rtype: float
        """
        if value is None:
            return None
                
        try:
            return float( value )
        except ( ValueError, TypeError ):
            msg = str( value ) + " this parameter must be a Float"
            raise UserValueError( parameter = param , msg = msg )
        

    @staticmethod
    def validate( param ):
        value = param.getValue()
        
        if value is None:
            return True     
        try:
            float( value )
        except ( ValueError , TypeError ):
            return False




class StringDataType( DataType ):


    @staticmethod
    def convert( value , param ):
        """
        Try to cast the value in String. 
        
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in string.
        @rtype: String
        @raise UserValueError: if the cast fail a UserValueError is raised.
        """
        if value is None:
            return None
            #msg = str( value )+": the parameter \"%s\" should be a String" %( param.getName() )
            #raise UserValueError( parameter = param , msg = msg )

        #type controls
        try:
            value = str( value )
            
            #the string with space are alowed
            #but if the string will appear as shell instruction it must be quoted
            if value.find(' ') != -1 :
                if not param.hasParamfile():
                    value = "'%s'" %value
            
            return value
            
        except ValueError :
            msg = " the parameter \"%s\" should be a String" %( param.getName() )
            raise UserValueError( parameter = param , msg = msg )
        
        return value                        
        
        
    @staticmethod
    def validate( param  ):

        value = param.getValue()
        if value is None:
            return True
        #try:
        #    str( value )
        #except ( ValueError, TypeError ):
        #    return False
        
        #allowed characters:
        #the words , space, ' , - , + , and dot if is not followed by another dot 
        #and eventually surrounded by commas
        reg = "(\w|\ |-|\+|,|\.(?!\.))+"
        if re.search( "^(%s|'%s')$" % (reg, reg) ,  value ) :         
            return True
        else:
            msg = "this value : \"" + str( value ) + "\" , is not allowed"
            raise UserValueError( parameter = param , msg = msg )
        
       


class ChoiceDataType( StringDataType ):
    #the values of ChoiceDataType are literals thus they are String

    @staticmethod        
    def convert( value , param ):
        """
        The values of ChoiceDataType are literals thus this method try to cast
        the value in string.
                
        @param value: the value provide by the User for this parameter
        @type value: 
        @return: the value converted in String
        @rtype: string
        """
        if value is None:
            return None
                
        try:
            value  = str( value )
            if value == param.getListUndefValue():
                return None
            else:
                return value
        except ValueError:
            
            msg = str( value )+" this parameter is a Choice its value should be a String" 
            raise UserValueError( parameter = param , msg = msg )


    @staticmethod
    def validate( param ):
        """
        @return: True if the value is valid. that 's mean the value
        should be a string among the list defined in the xml.
        otherwise a MobyleError is raised.
        @rtype: boolean
        @param value: a value from a Choice parameter
        @type value: Choice value
        @raise UserValueError: if the value is not a string or is not among the
        list defined in the xml vlist.
        """
        value = param.getValue()
        if param.hasVlist() :
            authorizedValues = param.getVlistValues()
        elif param.hasFlist() :
            authorizedValues = param.getFlistValues()
        else:
            msg = "%s a choice must have a flist or vlist" %( param.getName() )
            c_log.error( msg )
            raise MobyleError , msg     
        
        if value is None or value in authorizedValues:
            return True
        else:
            paramName = param.getName()
            logMsg = "Unauthorized value for the parameter : %s : authorized values = %s : provide value = %s" %( paramName , 
                                                                                                                  authorizedValues ,
                                                                                                                  value 
                                                                                                                  )
            c_log.error( logMsg )
            
            msg = "Unauthorized value for the parameter : %s" %( param.getName() )
            raise UserValueError( parameter = param , msg = msg )
                
               
           




class MultipleChoiceDataType( StringDataType ):

        

    
    @staticmethod
    def convert( value , param ):
        """
        The MutipleChoiceDataType value are literals thus this method try to cast
        the value in a list of string.
                
        @param value: the values provide by the User for this parameter
        @type value: list
        @return: a string based on each selected value and join by the separator.
        @rtype: String .
        @raise UserValueError: if the value can't be convert in a string.
        """       
        if value is None:
            return None
        valueType = type( value )
       
        sep = param.getSeparator()
        if sep is None:
            sep = ''
        elif sep.find( ' ' ) != -1 :
            raise MobyleError , "MultipleChoiceDataType doesn't accept space in separator"
        
        if valueType == types.UnicodeType :
            if value.find( sep ) == -1 :
                raise MobyleError ,"MultipleChoiceDataType doesn't separator in value"
            
            if sep == '':
                values = [ char for char in str( value ) ]
            else:
                values = str( value ).split( sep )[ 0 : -1 ]
                                
        elif valueType == types.StringType :
            if value.find( sep ) == -1 :
                raise MobyleError , "MultipleChoiceDataType can't find separator in value"
            if sep == '':
                values = [ char for char in value ]
            else:
                values = value.split( sep )[ 0 : -1 ]
                
        elif valueType == types.ListType or valueType == types.TupleType :
            try:
                values = [ str( elem ) for elem in value ]
            except ValueError:
                msg  = "this parameter is a MultipleChoice its all values must be Strings" %value 
                raise UserValueError( parameter = param , msg = msg )
        else:
            raise MobyleError , "MultipleChoiceDataType accept only strings or list of strings as value"           
        
        for oneValue in values:
                if sep == '' and len( oneValue ) > 1:
                    raise MobyleError , "in MultipleChoiceDataType if seperator is the empty string, each value length must be 1 (povided :%s)"%oneValue

        return sep.join( values )
    

    @staticmethod
    def validate( param ):
        
        userValues = param.getValue() #it's a string
        sep = param.getSeparator()
        if sep == '':
            userValues = [ i for i in userValues ]
        else:
            userValues = userValues.split( sep )

        authorizedValues =  param.getVlistValues()
        
        for value in userValues:
            if value not in authorizedValues :
                msg = "the value %s is not allowed (allowed values: %s " % (
                str( value ) ,
                str( param.getVlistValues() )
                )
                raise UserValueError( parameter = param , msg = msg )

        return True



    

class AbstractTextDataType( DataType ):

    
    @staticmethod
    def isFile( ):
        return True  

    
    @staticmethod    
    def head( data ):
        return data[ 0 : 50 ]
    
    @staticmethod
    def cleanData( data ):
      # trying to guess the encoding, before to convert the data to ascii
      try:
        # trying ascii
        data = unicode(data.decode('ascii','strict'))
      except:
        try:
          # utf8 codec with BOM support
          data = unicode(data,'utf_8_sig')
        except:
          try:
            # utf16 (default Windows Unicode encoding)
            data = unicode(data,'utf_16')
          except:
            # latin1
            data = unicode(data,'latin1')
      # converting the unicode data to ascii
      data = data.encode('ascii','replace')
      
      return  re.sub( "\r\n|\r|\n" , '\n' , data )


    @staticmethod    
    def _toFile(  param , data  , dest , destFileName , src , srcFileName ):
        """
        Write file (of user data) in dest directory .
        @param fileName:
        @type fileName: string
        @param data: the content of the file
        @type data: string
        @param dest:
        @type dest:
        @param src:
        @type src:
        @return: the name ( asbolute path )of the created file
        @rtype: string
        @raise: L{UserValueError} when filename is not allowed (for security reason)
        @raise: L{MobyleError} if an error occured during the file creation
        """
            
        try:
            destSafeFileName = Mobyle.Utils.safeFileName( destFileName )            
        except UserValueError, err:
            raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % destFileName )

      
        abs_DestFileName= os.path.join( dest.getDir() , destSafeFileName )

        # if the user upload 2 files with the same basename Mobyle.Utils.safeFileName
        # return the same safeFileName
        # I add an extension to avoid _toFile to erase the existing file.

        ext = 1
        completeName = abs_DestFileName.split( '.' )
        base = completeName[0]
        suffixe = '.'.join( completeName[1:] )

        while os.path.exists( abs_DestFileName ):
            abs_DestFileName = base + '.' + str( ext ) + '.' + suffixe
            ext = ext + 1
        
        if src:

            try:
                srcSafeFileName = Mobyle.Utils.safeFileName( srcFileName )
            except UserValueError, err:
                raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % fileName )

            #the realpath is because if the abs_SrcFileName is a soft link ( some results are ) the
            # hardlink point to softlink and it causse ane error : no such file 
            abs_SrcFileName = os.path.realpath( os.path.join( src.getDir() , srcSafeFileName ) )
            
            try:
                os.link(  abs_SrcFileName , abs_DestFileName )
            except OSError :
                #if the src and dest are not on the same device
                #an OSError: [Errno 18] Invalid cross-device link , is raised
                try:
                    shutil.copy( abs_SrcFileName , abs_DestFileName )
                except IOError ,err:
                    # I don't know  - neither the service ( if it exists )
                    #               - nor the job or session ID 
                    # I keep the Job or the Session to log this error 

                    msg = "can't copy data from %s to %s : %s" %( abs_SrcFileName ,
                                                                  abs_DestFileName ,
                                                                  err )
                                                    
                    raise MobyleError , "can't copy data : "+ str(err)
                    
  
        else:
        
            clean_content = AbstractTextDataType.cleanData( data )

            try:
                fh = open( abs_DestFileName , "w" )
                fh.write( clean_content )
                fh.close()
            except IOError , err:
                # I don't know  - neither the service ( if it exists )
                #               - nor the job or session ID 
                # I keep the Job or the Session to log this error 

                msg = "error occur when creating file : " + abs_DestFileName + str( err )
                raise MobyleError , msg


        size = os.path.getsize( abs_DestFileName )
        return os.path.basename( abs_DestFileName ) , size
    
          
    @staticmethod    
    def convert( value , param ):
        """
        @param value: is a tuple ( destFileName , data , dest , src , srcFileName) 
        @type value: tuple ( string filename , string content , L{Job} or L{Session} instnace )
          - filename is mandatory
          - data is a string reprensenting the data. it could be None if src is specify
          - dest is L{Job} or L{Session} instance is where the data will be store
          - src is L{Job} or L{Session} instance is where the data come from (it must be specify only if data is None).
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest ,L{Job} or L{Session} instance, src)
        @return: the fileName ( basename ) of the text file
        @rtype: string
        """

        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )
            #if value is None :
            #    raise UserValueError( parameter = param , msg = msg )
            #else:
            #    safeFileName = Mobyle.Utils.safeFileName( value )
            #    return safeFileName
        
        if value is None:
            return None        
        if len( value ) == 5 :
            data , dest , destFileName ,  src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: ( destFileName , data , Job/MobyleJob/JobState or Session instance , Job/MobyleJob/JobState or Session instance , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"
        
        if param.isInfile():
             #_toFile return a basename
             fileName  , size = AbstractTextDataType._toFile( param , data  , dest , destFileName , src , srcFileName )
            #fileName = os.path.basename( fileName ) 
        else:
            #safeFileName return a basename
            fileName = Mobyle.Utils.safeFileName( destFileName )

        try:
            jobstate = dest.jobState
        except AttributeError :
            jobstate = None
   
        if jobstate is not None:
            lang = param.cfg.lang()
            if param.promptHas_lang( lang ):
                prompt = param.getPrompt( lang = lang )
            else:
                prompt = None
                lang   = None
                
            jobstate.setInputDataFile( param.getName() ,
                                       ( prompt , lang ) ,
                                       param.getType() , # et pourquoi pas self
                                       ( fileName , size ,None )
                                       )
            jobstate.commit()

        return fileName


    @staticmethod
    def validate( param ):
        """
        @todo: faut t'il le passer dans une regexp? apriori Non il est deja passer par toFile et donc par safeFilename
        """
        value = param.getValue()
        if param.isout():
            if value is not None : #un parametre isout ne doit pas etre modifier par l'utilisateur 
                return False
            else:
                
                
                #####################################################
                #                                                   #
                #  check if the Parameter have a secure filenames   #
                #                                                   #
                #####################################################
                
                try:
                    debug = param.getDebug()
                    if debug > 1:
                        b_log.debug( "check if the Parameter have a secure filename" )
    
                    #getFilenames return list of strings representing a unix file mask which is the result of a code evaluation
                    #getFilenames return None if there is no mask for a parameter.
                    filenames = param.getFilenames( ) 
    
                    for filename in filenames :
                        if filename is None:
                            continue
                        mask = safeMask( filename )
    
                        b_log.debug( "filename= %s    safeMask = %s"%(filename, mask))
                        if  not mask or mask != filename :
                            # comment logger ce genre d'erreur 
                            # la faire remonter a MobyleJob validate ???
                            #msg = "The Parameter:%s, have an unsecure filenames value: %s " %( param.getNAme() ,
                            #                                                                    filename )
                            #self._logError( admMsg = "MobyleJob._validateParameters : " + msg ,
                            #                logMsg = None ,
                            #                userMsg = "Mobyle Internal Server Error"
                            #                )
                            #self.m_log.critical( "%s : %s : %s" %( self._service.getName(),
                            #                              self._job.getKey() ,
                            #                              msg
                            #                              )
                            #                )
                            raise MobyleError , "Mobyle Internal Server Error"
                        else:
                            if debug > 1:
                                b_log.debug( "filename = %s ...........OK" % filename )
    
                                               
                except UnDefAttrError :
                    b_log.debug("no filenames")
                
        else:
            if value is None:
                return True #an infile Text ne peut pas avoir de vdef mais peut il etre a None" => oui s'il n'est pas obligatoire
            else:
                return os.path.exists( param.getValue() )
        


class TextDataType( AbstractTextDataType ):
    # this trick is to avoid that SequenceDataType is a subclass of TextDataType
    pass



class SequenceDataType( AbstractTextDataType ):


    @staticmethod
    def convert(  value , param ):
        """
        convert the sequence contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is not supported
        or a MobyleError if something goes wrong during the conversion.

        @param value: is a tuple ( destFileName , data , dest , src , srcFileName) 
          - filename is mandatory
          - data is a string reprensenting a sequence. it could be None if src is specify
          - dest is L{Job} or L{Session} instance is where the data will be store
          - src is L{Job} or L{Session} instance is where the data come from (it must be specify only if data is None).
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest ,L{Job} or L{Session} instance, src)
        @return: the fileName ( basename ) of the  sequence file
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )
        #if param.isout():
        #    if value is None :
        #        raise UserValueError( parameter = param , msg = msg )
        #    else:
        #        safeFileName = Mobyle.Utils.safeFileName( value )
        #        return safeFileName

        if value is None:
            return None
         
        elif len( value ) == 5 :
             data , dest , destFileName , src , srcFileName  = value

        else:
            raise MobyleError , "value must be a tuple of 5 elements: ( data , dest , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"

        try:
            jobstate = dest.jobState
        except AttributeError :
            jobstate = None
        
        if param.isInfile():

            fileName  , sizeIn = SequenceDataType._toFile( param , data  , dest , destFileName , src , srcFileName )
            absFileName = os.path.join( dest.getDir() , fileName )
            prg , code , format , seq_nb = Mobyle.SequenceConverter.detect( absFileName )
            
            codeList = param.getAcceptedDataFormats()
            if not codeList:
                service = param.getFather()
                
                if service is not None:
                    while not service.isService():
                        service = service.getFather()

                    c_log.info( " %s.%s is a SequenceDataType with no AcceptedDataFormat tag"
                                %( service.getName() , param.getName() )
                                )
                              
                if format is not None: 
                        if jobstate is not None:
                            lang = param.cfg.lang()
                            if param.promptHas_lang( lang ):
                                prompt = param.getPrompt( lang = lang )
                            else:
                                prompt = None
                                lang   = None
    
                            jobstate.setInputDataFile( param.getName() ,
                                                       ( prompt , lang ) ,
                                                       param.getType() , 
                                                        ( fileName  ,  sizeIn ,  format ) ,
                                                       fmtProgram = prg 
                                                       ) 
                            jobstate.commit()
                        param.setDataFormat( ( format , seq_nb , prg ) )  #permit at the session to have format information  
                        return os.path.basename( fileName )
                
                else:
                    msg = " invalid Sequence format"
                    raise UserValueError( parameter = param , msg = msg )

            try:
                forceSeqfmt = param.forceReformating()
            except UnDefAttrError :
                forceSeqfmt = False

            # I must work with absolute path because the portal use this
            # and it not work in the same directory as the sequence files

            # the decision to convert the sequence if the format detected is in codeList or not
            # is take by Mobyle.SequenceConverter.convert
            fmtPrg , fmtIn , inFileName ,fmtOut , outFileName , sq_nb = Mobyle.SequenceConverter.convert( absFileName , codeList , force = forceSeqfmt )

            param.setDataFormat( ( fmtOut , seq_nb , fmtPrg ) ) #permit at the session to have format information
            
            if fmtIn is None :
                msg ="Cannot detect the sequence format. The supported sequence formats are: " + str( Mobyle.SequenceConverter.supportedFormat() )
                                       
                if jobstate is not None:
                    lang = param.cfg.lang()
                    if param.promptHas_lang( lang ):
                        prompt = param.getPrompt( lang = lang )
                    else:
                        prompt = None
                        lang   = None

                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               (  fileName  , sizeIn , "UNKNOWN" ) ,
                                               fmtProgram = prg 
                                               )                
                    jobstate.commit()

                raise UserValueError( parameter = param , msg = msg )    
                        
            elif jobstate is not None:
                lang = param.cfg.lang()
                if param.promptHas_lang( lang ):
                    prompt = param.getPrompt( lang = lang )
                else:
                    prompt = None
                    lang = None
                
                if fmtOut is None: # the data has not been converted
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg
                                               )
                    jobstate.commit()
                else:# the data has been converted
                    sizeOut = os.path.getsize( outFileName )
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg  ,
                                               formattedFile = ( outFileName , sizeOut , fmtOut )
                                               )                    
                jobstate.commit()
                
            outFileName = os.path.basename( outFileName )
            return outFileName


        else: #it's a file in output
            fileName = Mobyle.Utils.safeFileName( destFileName )
        
        return fileName 
       


    @staticmethod
    def detect(  param  ):
        """
        detect the sequence contain in the file fileName .
        throws an UnsupportedFormatError if the output format is not supported
       
        @param value: is a tuple ( filename , data , dest ) 
          - filename is mandatory
          - data is a string reprensenting a sequence. 
          - dest is L{Job} or L{Session} instance is where the data will be store
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest)
        @return: the program which detect the format
                 the sequence format
                 the number of sequences
                 the filename
        @rtype: ( string prg , int code , string format , int seq_nb , string fileName ) 
        """

          
        if value is None :
            return None

        elif len( value ) == 4:
            data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 4 elements:( data , dest , destFileName , src , srcFileName ) "

        if fileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if param.isInfile() :
            fileName , size =   SequenceDataType._toFile( param , data , dest , destFileName , src , srcFileName )
            absInFileName = os.path.join( dest.getDir() , fileName )
            fmtPrg , code , fmtOut , seq_nb = Mobyle.SequenceConverter.detect( absInFileName )
            
            param.setDataFormat( ( fmtOut , seq_nb , fmtPrg ) ) #permit at the session to have format information
            
            return ( fmtPrg , code , fmtOut , seq_nb , fileName )

        else:
            fileName = Mobyle.Utils.safeFileName( fileName )
       
        return fileName 
       



    @staticmethod
    def validate( param ):
        """
        """

        value = param.getValue()

        if param.isout():
            if value is not None : #un parametre isout ne doit pas etre modifier par l'utilisateur 
                return False
            else:
                
                
                #####################################################
                #                                                   #
                #  check if the Parameter have a secure filenames   #
                #                                                   #
                #####################################################
                
                try:
                    debug = param.getDebug()
                    if debug > 1:
                        b_log.debug( "check if the Parameter have a secure filename" )
    
                    #getFilenames return list of strings representing a unix file mask which is the result of a code evaluation
                    #getFilenames return None if there is no mask for a parameter.
                    filenames = param.getFilenames( ) 
    
                    for filename in filenames :
                        if filename is None:
                            continue
                        mask = safeMask( filename )
    
                        b_log.debug( "filename= %s    safeMask = %s"%(filename, mask))
                        if  not mask or mask != filename :
            
                            raise MobyleError , "have an unsecure filenames value before safeMask: %s , after safeMask: %s"%( filename , mask )
                        else:
                            if debug > 1:
                                b_log.debug( "filename = %s ...........OK" % filename )
    
                                               
                except UnDefAttrError :
                    self.build_log.debug("no filenames")

        else:
            if value is None:
                return True
            else:
                prg , code , format , seq_nb = Mobyle.SequenceConverter.detect( value )
                
                try:
                    #codeList = self.getSeqfmt()
                    codeList = param.getAcceptedDataFormats()
                except UnDefAttrError:
                    return True
        
                if code in codeList:
                    return True
                else:
                    return False

        
        
    
class AlignmentDataType( AbstractTextDataType ):

    @staticmethod
    def convert( value , param  ):
        """
        convert the alignment contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is not supported
        or a MobyleError if something goes wrong during the conversion.

        @param value: is a tuple the first element
          - filename is mandatory
          - data is a string reprensenting an alignment. it could be None if src is specify
          - dest is L{Job} or L{Session} instance is where the data will be store
          - src is L{Job} or L{Session} instance is where the data come from (it must be specify only if data is None).
        @type value: ( string filename, string data , L{Job} or L{Session} instance dest ,L{Job} or L{Session} instance, src)
        @return: the fileName ( basename ) of the alignment file
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )
        #if param.isout():
        #    if value is None :
        #        raise UserValueError( parameter = param , msg = msg )
        #    else:
        #        safeFileName = Mobyle.Utils.safeFileName( value )
        #        return safeFileName

        if value is None:
            return None

        elif len( value ) == 5 :
            data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: ( data , dest , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        try:
            jobstate = dest.jobState
        except AttributeError:
            jobstate = None

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"
        
        if param.isInfile():
            fileName , sizeIn = AlignmentDataType._toFile( param , data , dest , destFileName , src , srcFileName )
            
            absInFileName = os.path.join( dest.getDir() , fileName )
            sizeIn = os.path.getsize( absInFileName )
            prg , code , format , al_nb = Mobyle.AlignmentConverter.detect( absInFileName )
            
            codeList = param.getAcceptedDataFormats()
            if not codeList:
                service = param.getFather()
                
                if service is not None:
                    while not service.isService():
                        service = service.getFather()

                    c_log.info( " %s.%s is an AlignmentDataType with no AcceptedDataFormat tag"
                                    %( service.getName() , param.getName() )
                                    )

                if format is not None:#codelist =None , format != None => it's a session no conversion
                    if jobstate is not None:
                        lang = param.cfg.lang()
                        if param.promptHas_lang( lang ):
                            prompt = param.getPrompt( lang = lang )
                        else:
                            prompt = None
                            lang   = None

                        jobstate.setInputDataFile( param.getName() ,
                                                   ( prompt , lang ) ,
                                                   param.getType() ,
                                                    ( fileName  , sizeIn , format ) ,
                                                   fmtProgram = prg 
                                                   )
                        jobstate.commit()
                    param.setDataFormat( ( format , al_nb , prg ) )    #permit at the session to have format information
                    return  fileName 
                
                else:
                    msg = " invalid Alignment format"
                    raise UserValueError( parameter = param , msg = msg )

            try:
                forceSeqfmt = param.forceReformating()
            except UnDefAttrError :
                forceSeqfmt = False

            # I must work with absolute path because the portal use this
            # and it not work in the same directory as the sequence files

            # the decision to convert the Alignment if the format detected is in codeList or not
            # is take by Mobyle.AlignmentConverter.convert
            fmtPrg , fmtIn , inFileName , fmtOut , outFileName , al_nb = Mobyle.AlignmentConverter.convert( absInFileName ,
                                                                                                     codeList , 
                                                                                                     force = forceSeqfmt )

            param.setDataFormat( ( fmtOut , al_nb , fmtPrg ) ) #permit at the session to have format information

            if fmtIn is None :
                msg ="Cannot detect the alignment format. The supported alignment formats are: " + str( Mobyle.AlignmentConverter.supportedFormat() )
                if jobstate is not None:
                    lang = param.cfg.lang()
                    if param.promptHas_lang( lang ):
                        prompt = param.getPrompt( lang = lang )
                    else:
                        prompt = None
                        lang   = None
    
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               (  fileName ,  sizeIn , "UNKNOWN" ) ,
                                               fmtProgram = prg 
                                               )                
                    jobstate.commit()
                    
    
                raise UserValueError( parameter = param , msg = msg )    
                        
            elif jobstate is not None:
                lang = param.cfg.lang()
                if param.promptHas_lang( lang ):
                    prompt = param.getPrompt( lang = lang )
                else:
                    prompt = None
                    lang   = None

                
                if fmtOut is None: # the data was not convert
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg
                                               )
                    jobstate.commit()
                else:# the was conveerted
                    sizeOut = os.path.getsize( os.path.join( dest.getDir() , outFileName) )
                    jobstate.setInputDataFile( param.getName() ,
                                               ( prompt , lang ) ,
                                               param.getType() ,
                                               ( inFileName , sizeIn , fmtIn ) ,
                                               fmtProgram = fmtPrg  ,
                                               formattedFile = ( outFileName , sizeOut , fmtOut )
                                               )
                jobstate.commit()
                
            outFileName = os.path.basename( outFileName )
            return outFileName


        else:
            fileName = Mobyle.Utils.safeFileName( fileName )
        
        return fileName 
       

    @staticmethod
    def detect(  value  ):
        """
        convert the sequence contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is nos supported
        or a MobyleError if something goes wrong during the conversion.

        @param value: the fileName of the sequence file
        @type value: String
        @return: the fileName of the  sequence file
        """
          
        if value is None :
            return None

        elif len( value ) == 5 :
             data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: (  data , destination , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"

        if param.isInfile() :
            try:
                fileName , size = AlignmentDataType._toFile( param , data , dest , destFileName , src , srcFileName )
            except UserValueError , err :
                pass  #a gerer idem convert TextDataType

            fmtPrg , code , fmtOut , al_nb = Mobyle.AlignmentConverter.detect( os.path.join( dest.getDir() , fileName ) )
            param.setDataFormat( ( fmtOut , al_nb , fmtPrg ) )

            return ( fmtPrg , code , fmtOut ,  al_nb , fileName )
        

        else:
            fileName = Mobyle.Utils.safeFileName( fileName )
        
        return fileName 
       


    @staticmethod
    def validate( param ):
        """
        filename = nom de fichier local tout est en local!
        cette methode ne doit etre apple que si l'objet est ratache a un service , job ...
        
        convert the sequence contain in the file fileName in the rigth format
        throws an UnsupportedFormatError if the output format is nos supported
        or a MobyleError if something goes wrong during the conversion.
        """
        value = param.getValue()


        if param.isout():
            if value is not None : #un parametre isout ne doit pas etre modifier par l'utilisateur 
                return False
            else:
                
                
                #####################################################
                #                                                   #
                #  check if the Parameter have a secure filenames   #
                #                                                   #
                #####################################################
                
                try:
                    debug = param.getDebug()
                    if debug > 1:
                        b_log.debug( "check if the Parameter have a secure filename" )
    
                    #getFilenames return list of strings representing a unix file mask which is the result of a code evaluation
                    #getFilenames return None if there is no mask for a parameter.
                    masks = param.getFilenames( ) 
    
                    for mask in masks :
                        if mask is None:
                            continue
                        mySafeMask = safeMask( mask )
    
                        b_log.debug( "filename= %s    safeMask = %s"%(mySafeMask, mask))
                        if  not mySafeMask or mySafeMask != mask :
                            # comment logger ce genre d'erreur 
                            # la faire remonter a MobyleJob validate ???
                            #msg = "The Parameter:%s, have an unsecure filenames value: %s " %( param.getNAme() ,
                            #                                                                    filename )
                            #self._logError( admMsg = "MobyleJob._validateParameters : " + msg ,
                            #                logMsg = None ,
                            #                userMsg = "Mobyle Internal Server Error"
                            #                )
                            #self.m_log.critical( "%s : %s : %s" %( self._service.getName(),
                            #                              self._job.getKey() ,
                            #                              msg
                            #                              )
                            #                )
                            raise MobyleError , "Mobyle Internal Server Error"
                        else:
                            if debug > 1:
                                b_log.debug( "filename = %s ...........OK" % mask )
    
                                               
                except UnDefAttrError :
                    self.build_log.debug("no filenames")

        else:
            if value is None:
                return True
            else:
                prg , code , format , al_nb = Mobyle.AlignmentConverter.detect( value )
                
                try:
                    #codeList = self.getSeqfmt()
                    codeList = param.getAcceptedDataFormats()
                except UnDefAttrError:
                    return True
        
                if code in codeList:
                    return True
                else:
                    return False
            


class BinaryDataType( DataType ):

    @staticmethod   
    def isFile():
        return True
    
    @staticmethod
    def head( data ):
        return 'Binary data'
            
    @staticmethod          
    def cleanData( data ):
        """
        prepare data prior to write it on a disk
        @param data: 
        @type data:a buffer
        """
        return data


    @staticmethod   
    def _toFile( param , data , dest , destFileName , src , srcFileName ):
        """
        Write file (of user data) in the working directory .
        @param fileName:
        @type fileName: string
        @param content: the content of the file
        @type content: string
        @return: the name ( absolute path ) of the created file created ( could be different than the arg fileName )
        @rtype: string
        @call: L{MobyleJob._fillEvaluator}
        """
        

        try:
            destSafeFileName = Mobyle.Utils.safeFileName( destFileName )
        except UserValueError, err:
            raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % fileName )

        abs_DestFileName = os.path.join( dest.getDir() , destSafeFileName )

        # if the user upload 2 files with the same basename Mobyle.Utils.safeFileName
        # return the same safeFileName
        # I add an extension to avoid _toFile to erase the existing file.

        ext = 1
        completeName = abs_DestFileName.split( '.' )
        base = completeName[0]
        suffixe = '.'.join( completeName[1:] )

        while os.path.exists( abs_DestFileName ):
            abs_DestFileName = base + '.' + str( ext ) + '.' + suffixe
            ext = ext + 1

        if src:
            try:
                srcSafeFileName = Mobyle.Utils.safeFileName( srcFileName )
            except UserValueError, err:
                raise UserValueError( parameter = param , msg = "this value : %s is not allowed for a file name, please change it" % srcFileName  )
            
            #the realpath is because if the abs_SrcFileName is a soft link ( some results are ) the
            # hardlink point to softlink and it causse ane error : no such file  
            abs_SrcFileName = os.path.realpath( os.path.join( src.getDir() , srcSafeFileName ) )
            try:
                os.link(  abs_SrcFileName , abs_DestFileName )
            except OSError :
                #if the src and dest are not on the same device
                #an OSError: [Errno 18] Invalid cross-device link , is raised
                try:
                    shutil.copy( abs_SrcFileName , abs_DestFileName )
                except IOError ,err:
                    # je ne connais - ni le service (s'il existe)
                    #               - ni le l' ID du job ou de la session
                    # donc je laisse le soin au Job ou la session a logger l'erreur

                    msg = "can't copy data from %s to %s : %s" %( abs_SrcFileName ,
                                                                  abs_DestFileName ,
                                                                  err )
                    
                    raise MobyleError , "can't copy data : "+ str(err)
 
  
        else:
            try:
                fh = open( abs_DestFileName , "wb" )
                fh.write( data )
                fh.close()
            except IOError , err:
                # je ne connais - ni le service (s'il existe)
                #               - ni le l' ID du job ou de la session
                # donc je laisse le soin au Job ou la session a logger l'erreur

                msg = "error occur when creating file %s: %s" %( os.path.basename( abs_DestFileName ) ,  err )

                raise MobyleError , msg
        size  = os.path.getsize( abs_DestFileName )
        return os.path.basename( abs_DestFileName ) , size
    

    @staticmethod
    def convert( value , param ):
        """
        Do the generals control and cast the value in the right type.
        if a control or the casting fail a MobyleError is raised

        @param value: the value provide by the User for this parameter
        @type value: String
        @return: the fileName ( basename ) of the binary file
        """
        if param.isout():
            raise UserValueError( parameter = param , msg = "out parameter can't be modify by users" )       
        #if param.isout():
        #    if value is None :
        #        raise UserValueError( parameter = param , msg = msg )
        #    else:
        #        safeFileName = Mobyle.Utils.safeFileName( value )
        #        return safeFileName

        if value is None:
            return None
 
        elif len( value ) == 5 :
            data , dest , destFileName , src , srcFileName = value
        else:
            raise MobyleError ,"value must be a tuple of 5 elements: ( data , dest , destFileName , src , srcFileName )"

        if destFileName is None:
            return None
        
        if dest is None:
            raise MobyleError, "the destination is mandatory"

        if  data and src :
            raise MobyleError, "you cannot specify data and src in the same times"

        if not data and ( not dest or not src ) :
            raise MobyleError , "if data is not specify, dest and src must be defined"

        if src and not srcFileName :
            raise MobyleError , "if src is specify , srcFileName must be also specify"

        if param.isInfile():
            #param._toFile return an absolute path
            fileName , size = BinaryDataType._toFile( param , data , dest , destFileName , src , srcFileName )

        else:
            #safeFileName return a basename
            fileName = Mobyle.Utils.safeFileName( destFileName )

        try:
            jobstate = dest.jobState
        except AttributeError :
            jobstate = None

        if jobstate is not None:
            lang = param.cfg.lang()
            if param.promptHas_lang( lang ):
                prompt = param.getPrompt( lang = lang )
            else:
                prompt = None
                lang   = None
            
            jobstate.setInputDataFile( param.getName() ,
                                       ( prompt , lang ) ,
                                       param.getType() ,
                                       ( fileName , str( size ) , None )
                                       )
            jobstate.commit()

        return fileName



    @staticmethod
    def validate( param ):
        """
        @todo: il faudrait avoir value = None et verifier file names
        """
        if param.isout():
            value = param.getValue()
            if value is None :
                return True
            else:
                return False
        else:        
            return os.path.exists( param.getValue() )
        

#===============================================================================
# 
#    
# class FilenameDataType( DataType ):
# 
#    @staticmethod
#    def convert( value ,param ):
#        if value is None:
#            return None
#            #raise UserValueError( parameter = param , msg= " this parameter must be a String" )
# 
#        fileName = Mobyle.Utils.safeFileName( value )
#        return fileName
#    
#    @staticmethod
#    def validate( param ):
#        return True
#===============================================================================

class FilenameDataType( DataType ):
 
   @staticmethod
   def convert( value ,param ):
       if value is None:
           return None
           #raise UserValueError( parameter = param , msg= " this parameter must be a String" )
 
       fileName = Mobyle.Utils.safeFileName( value )
       return fileName
   
   @staticmethod
   def validate( param ):
       value = param.getValue()
       if value is None :
           return True
       safefileName = Mobyle.Utils.safeFileName( value )
       if safefileName != value:
           msg = "invalid value: %s :the followings characters \:/ ;` {} are not allowed" %( value )
           raise UserValueError( parameter = param , msg = msg )
       else:
           return True





class StructureDataType(DataType):
    
    @staticmethod
    def convert( value , param ):
        """
        Do the general control and cast the value in the right type.
        if a control or the casting fail a MobyleError is raised.
        
        @param value: the value provide by the User for this parameter
        @type value: String
        @return: 
        """
        raise NotImplementedError ,"ToDo"
    
    @staticmethod
    def validate( param ):
        raise NotImplementedError ,"ToDo"
        

class PropertiesDataType(DataType):
    
    @staticmethod
    def convert( value , param ):
        """
        Do the general control and cast the value in the right type.
        if a control or the casting fail a MobyleError is raised.
        
        @param value: the value provide by the User for this parameter
        @type value: String
        @return: 
        """
        raise NotImplementedError ,"ToDo"
    
    
    @staticmethod    
    def validate( param ):
        raise NotImplementedError ,"ToDo"



    
        
if __name__ == "__main__":
     import sys
     sys.path.append('/home/bneron/Mobyle')
     sys.path.append('/home/bneron/Mobyle/Src')
     
     dtf = DataTypeFactory()

     s = dtf.newDataType('Sequence')

     ss = dtf.newDataType('Sequence','SpecificSequence')

     fileName = '/home/bneron/Developement/Python/MobyleTest2/Results/toppred2/A00000000000000/index.xml'
     
     from Mobyle.JobState import JobState   
     js = JobState( fileName )

     
     from Mobyle.Service import MobyleType , Parameter
     smt = MobyleType( s , format=[ 'FASTA' ] )
     seqParam = Parameter( smt , name = "sequenceParam")
     
     ssmt = MobyleType( ss , format=[ 'FASTA' ] )
     seqSpeParam = Parameter( ssmt , name = "sequenceSpecificParam" )
     
     print s.convert( ('GAATTC' , js , "testCoreSeq" , None , None) , seqParam )
     print ss.convert( ('AAAAUUUUUUUUAA', js , "testCoreSpeSeq" , None , None) , seqSpeParam  )
     print s.name
     print ss.name
     print " s  -> ss " , s.isPipableToDataType(ss)
     print " ss -> s  " , ss.isPipableToDataType(s)
     print " ss -> ss " , ss.isPipableToDataType(ss)



