
"""
factory = ClientFactory()
clustalw = factory.createService('clustalw', phylip_alig=True)
clustalw_job = clustalw.createJob()
clustalw_job.quicktree("1")
clustalw_job.run(infile="toto")

clustalw_alignment = clustalw_job.getSomeResult('aligfile')

dnapars = factory.createService('dnapars')
dnapars_job = dnapars.createJob()
dnapars_job.run(infile=clustalw_alignment)

print dnapars_job.getSomeResult('treefile')
"""

import os ,os.path
from httplib import HTTP
import sys
import re
import time
from Bio.Seq import Seq
from Bio.SeqRecord import SeqRecord
from Bio.Clustalw import ClustalAlignment

try:
    _MOBYLEHOME = os.environ['MOBYLEHOME']
except KeyError :
    #os.environ['MOBYLEHOME']= "/Users/letondal/Packages/Mobyle"
    os.environ['MOBYLEHOME']= "/data/aw3/Mobyle"
    _MOBYLEHOME = os.environ['MOBYLEHOME']
    #print >> sys.stderr , "the environement variable MOBYLEHOME is not defined"

#append Mobyle Home to the search modules path

if (os.path.join(_MOBYLEHOME,'Src')) not in sys.path:    
    sys.path.append(os.path.join(_MOBYLEHOME,'Src'))

import Ft.Xml.Domlette

import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
import Mobyle.Net
from  Mobyle.Classes.Core import DataTypeFactory
from Mobyle.MobyleError import *
from Mobyle.Utils import ServiceLocator


_cfg = Mobyle.ConfigManager.Config()

__extra_epydoc_fields__ = [('call', 'Called by','Called by')]



class ClientFactory:

    #location = 'http://kerka.sis.pasteur.fr:81/cgi-bin/MobylePortal/jobcgi.py'
    #email = 'bneron@pasteur.fr'
    location = 'http://localhost/cgi-bin/MobylePortal/jobcgi.py'
    email = 'letondal@pasteur.fr'

    def __init__(self, email=None, location=None, verbose=False):
    	if email is not None:
    	    self.email = email
            if location is not None:
                self.location = location
            self._verbose = verbose


    def createService(self, service, **args):
    	"""   
    	@param service: name of a service
    	@type service: string
    
    	    - get the XML definition from the remote Mobyle server
                e.g: http:mobyle.pasteur.fr/cgi-bin/MobylePortal/getservice.py?service=clustalw
            - check the response (might be unavailable)
    
    	@param args: parameters and their value
    	@type args: a dicitonary
        @return: a ClientService
    	"""


        # fileName = os.path.join( os.environ['MOBYLEHOME'] , 'Local' , 'Programs' , service + ".xml" )

        serviceLocator = ServiceLocator()
        try:
            fileName = serviceLocator[ service ]
        except MobyleError, e:
            msg = "cannot find the \"" + service + "\" xml description file in Programs directory"
            print >> sys.stderr,  msg 
            print >> sys.stderr, "ClientFactory.ClientFactory : " + msg + " " + service + " is temporary unvailable"
            raise MobyleError , msg

        try:    
            #serviceUri = 'file://' + os.path.abspath( fileName )
            serviceUri = 'file://' + fileName 
            parser = ClientServiceParser()
            parser.parse( serviceUri )
            service = parser.data
        except MobyleError ,err:
            print >> sys.stderr, "ClientFactory._makeService : " + str( err ) + "Mobyle internal server error"
            raise MobyleError , str( err )
             
        
        for param, value in args.items():
            if param == "email":
                service.email = value
            elif param == "location":
                service.location = value
            elif param == "verbose":
                service.verbose = value

        for paramName in service.parameters.keys():
           if args.has_key(paramName):
	       service.setParam(paramName, args[paramName])

	if service.email is None:
	    service.email = self.email

	if service.location is None:
	    service.location = self.location

        return service


    def getAvailableAnalyses(self):
        pass

    def getAvailableCategories(self):
        pass

    def getAvailableAnalysesInCategory(self):
        pass

    def getServiceLocation(self, service):
        pass




class ClientService:

    """
     - creates jobs
     - provide information about a service (default values, prompts, ...)
    """

    def __init__(self, name, verbose=0, location=None, **args):
        """
        """
    	self.name = name
    	self.parameters = {}
    	self._value = {}
    	self.email = None
    	self.location = location
        if args.has_key("email"):
            self.email = args["email"]
        self._verbose = verbose
    	self._datainput = [] #il peut y avoir pusieurs maininput

    def createJob(self, location=None, interval=10, **args):
        """
          - instantiates ClientJob

        @param args: a list of parameters' values
        @type args: dictionary
    	@return: a ClientJob
        @raise L{MobyleError}: 
        @todo: 

        """
    	if location is None:
    	    location = self.location
    	self.interval = interval
        return ClientJob(self, location, interval, **args)


    def createAndRun(self, location= None, interval= 10, **args):
        clientjob = self.createJob(self)
        return clientjob.run(self.name, location, interval, **args)


    def getParamsOfClass(self, klass, inOrOut):
    	"""
    	@param klass: name of the parameter class
    	@type klass: string/python class
    
    	@param inOrOut: whether one wants in (input) or out (output) parameters
    	@type inOrOut: string
    
    	@return: a list of parameter names (only inOrOut parameters)
    	"""
    
    	result = []
    	for paramName in self.parameters.keys():
    	    if self.parameters[param]['class'] == klass:
    		result.append(paramName)
    	return result

    
    def getDefaultValue(self, param):
        return self.parameters[param]['vdef']


    def getParamClass(self, param):
        #return self.parameters[param]['class']
        return self.parameters[param]['type']


    def setParam(self, name, value=None):
        if ParamMethod.debug:
            print >>sys.stderr, "ClientService setParam: name: ", name
            
        if not self.parameters.has_key(name):
            raise MobyleError, "ClientService setParam: parameter " + name + " does not exist for " + self.name
        if ParamMethod.debug:
	        print >>sys.stderr, "ClientService setParam: param: setting ", name
                
        self._value[name] = value


    def getParam(self, name):
	       return self._value[name]

    def getDataInputParam(self):
        """
        return the main inputs parameter name
        @rtype: list of string
	    """
        return self._datainput

 
class ClientJob:

    debug = 1

    def __init__(self, service, location, interval, **args):
    	"""
    	"""
    	self.service = service
    	self.location = location
    	self.interval = interval
    	self.args = args
        self._value = {}
        self._param_method = {}
    	self.jobstate = None
        for param in self.service.parameters.keys():
            if args.has_key(param):
                self.setParam(param,args[param])
	    else:
		try:
		    self.setParam(param, self.service.getParam(param))
		except KeyError:
		    pass

    def jobState(self, url):
        """
        To setup jobstate from url
        """
        print "jobState URL:", url
        self.jobstate = Mobyle.JobState.JobState(url)

    def run(self, location=None, **args):
    	"""
    	Launches the CGI and waits for completion
    	http://mobyle.pasteur.fr/cgi-bin/MobylePortal/jobcgi.py ...
    
    	Needs to know the url!
    	"""
        if location is None:
            location = self.location
            
        try:               
            self.submit(location, **args)
            if self.debug:
                print >>sys.stderr, "ClientService run: job submitted: ", self.jobid()
                
        except MobyleError, e:
            print >>sys.stderr, e
            return
    
    	self.waitFor()

 
    def setParam(self, name, value=None):
        if ParamMethod.debug:
            print >>sys.stderr, "ClientJob param: name: ", name, " value: ", value
            
    	if not self.service.parameters.has_key(name):
                raise MobyleError, "parameter " + name + " does not exist for " + self.service.name
        if value is not None:
            if ParamMethod.debug:
                print >>sys.stderr, "ClientJob param: setting ", name, " to: ", value
                
            self._value[name] = value
        else:
            if self._value.has_key(name):
                return self._value[name]
            else:
                return None

    def makePipe(self, outjob, outParamName, inParamName):
    	"""
    	This method feeds the parameter of name inParamName with the data available from 
    	a previous job's output parameter outParamName.
    
    	- 1st get the file name corresponding to the output parameter outParamName
    	- [future optimisation] if servers of the outjob and the current one are the same, 
    	     ask the server for a link
            - feed the content of this file to  parameter inParamName of the current job.
    
    	Example:
    
    	dnapars_job = dnapars.createJob()
    	dnapars_job.run(infile=sys.argv[1])
    	drawtree_job = drawtree.createJob()
    	drawtree_job.makePipe(dnapars_job, outParamName='treefile', inParamName='treefile')
    	plotfile = drawtree_job.saveSomeResult('plotfile')
    
    
    	@param outjob: previous step's job which output is requested for feeding new job
    	@type outjob: ClientJob
    
    	@param outParamName: an ouput parameter name of job outjob
    	@type outParamName: string
     
    	@param inParamName: an input parameter name of current job
    	@type inParamName: string
    	"""
    
    	file = None
        #######################ATTENTION
        results = outjob.jobstate.getOutputs()
        
        #print >>sys.stderr, "ClientJob.makePipe: Results: ", results
    	for param, typed_files in results:
            #param is an instance of class Parameter with a mobyleType etc....
            #files is a list of tuples [( filename , format or None ) , ... ] 
            #print >>sys.stderr, "ClientJob.makePipe: param=", param
    	    if param.getName() == outParamName:
                #print >>sys.stderr, "ClientJob.makePipe: ", outParamName, " found"
                files = [ f[0] for f in typed_files ]
    		#file = results[paramName][1][0]
    		break
    
    	# optimisation when both jobs run on the same server:
    	# if outjob.location == self.location:
    	#     self.makeLink(outjob, outParamName, inParamName)
    	# --> makeLink produces a special parameter in http request, that will be handled
    	# in the server CGI, for instance by a soft or hard link on the filesystem.
    	if files:
            # we arbitrarily choose the 1sr one
    	    result = outjob.getSomeResult(files[0])
    	    self.setParam(inParamName, result)
    	else:
    	    raise MobyleError, "ClientJob.makePipe() " + outParamName + ": no such output parameter"


    def __getattr__(self, name):
        
        # 1. calling self.<parameter name>()
        #  -> define it on the fly if needed
        #print >> sys.stderr, "ClientJob __getattr__: ", name
        #s = raw_input()
        try:
            p = self.service.parameters[name]
            if self.debug:
                print >>sys.stderr, "ClientJob __getattr__: ", name, " : this is a parameter for ", self.service.name
            if not self._param_method.has_key(name):
                self._param_method[name] = ParamMethod(ClientJob.__dict__['setParam'], self, name)
            return self._param_method[name]

        # 2. other statically defined attributes
        except KeyError:
            try:
                value = self.__dict__[name]
            except KeyError:
                raise AttributeError, name
            if type(value) is not types.FunctionType:
                return value
            return ParamMethod(value, self)

    
    def submit(self, location=None, **args):
        """
        Asynchrone
        """
        if location is None:
            location = self.location

    	if self.service.email is not None:
    	    self.email = self.service.email

        for param in self.service.parameters.keys():
            if args.has_key(param):
                self.setParam(param,args[param])

        self._error = None
        self._error_message = ""
        self._finished = False
        self._args = []
        self._results = None
        self._tmp_dir = ""
        self._ua = None
    
        self._submit()
	    
        return self.jobid()

    def jobid(self):
    	if self.jobstate is None:
    	    raise MobyleError, "ClientJob.jobid(): no jobstate"
    	return self.jobstate.getID()

    def finished(self):
    	if self.jobstate is None:
    	    raise MobyleError, "ClientJob.finished(): no jobstate"
    	return self.jobstate.isFinished()

    def error(self):
        return self._error

    def error_message(self):
        return self._error_message

    def waitFor(self):
    	"""
    	Check for job completion
    	by using Jobstate with the job url (isFinished or getStatus)
    	better: http://mobyle.pasteur.fr/cgi-bin/MobylePortal/jobcgi.py + method name
    	"""
    
    	while not self.error() and not self.finished():
                time.sleep(self.interval)


    def getResults(self):
    	"""
    	with Jobstate (job url) (getResults -> list of filenames)
    	needs the url path to construct the complete url of a result (getID)
    	"""
    	if self.jobstate is None:
    	    raise MobyleError, "ClientJob.getResults(): no jobstate"
    	results = []
    	#for Type, files in self.jobstate.getResults().values():
        for param, typed_files in self.jobstate.getOutputs():
            files = [ f[0] for f in typed_files ]
    	    results.extend(files)
    	return results



    def getSomeResult(self, filename):
    	"""
    	@param filename: filename of the result -- e.g result.ps
    	@type filename: string
    	"""
    	if self.jobstate is None:
    	    raise MobyleError, "ClientJob.getSomeResult(): no jobstate"
    	try:
    	    return self.jobstate.getOutputFile(filename)
    	except MobyleError, e:
    	    for resultfile in self.getResults():
                if resultfile.find(filename) >= 0:
                    return self.jobstate.getOutputFile(resultfile)
    	    raise MobyleError, e


    def saveSomeResult(self, filename, outputfile=None):
    	"""
    	@param filename: filename of the result --e.g result.ps
    	@type filename: string
    
    	@param outputfile: name of the file where to save result; defaults to filename
    	@type outputfile: string
    
    	@return result file name
    	"""
    	if self.jobstate is None:
    	    raise MobyleError, "ClientJob.getSomeResult(): no jobstate"
    	try:
    	    result = self.jobstate.getOutputFile(filename)
    	    resultfile = filename
    	except MobyleError, e:
    	    result = None
    	    for resultfile in self.getResults():
    		if resultfile.find(filename) >= 0:
    		    result = self.jobstate.getOutputFile(resultfile)
    		    break
    	    if result is None:
    		raise MobyleError, e
    	if outputfile is None:
    	    outputfile = resultfile
    	fh = open(outputfile, 'w')
    	fh.write(result)
    	fh.close()
    	return resultfile
    	

    def runAndWaitFor(self, inputs):
        pass

    def getAnalysisType(self):
    	"""
    	See ClientService
    	"""
    	pass

    def getInputSpec(sef):
    	"""
    	See ClientService
    	"""
    	pass

    def getResultSpec(self, result):
    	"""
    	@param result: an url
    
    	@type result: a string
    
    	@return: type/class of the output file (for piping purpose)
    
    	See ClientService
    
    	"""
    	pass


    def describe(self):
        pass

    def destroy(self):
        pass

 
    # --------------------------------------------------------
    # Internal core methods
    
    def _submit(self):
        if self.jobstate is not None:
            self._error = 1
            self._error_message = "Job already lauchend"
            raise MobyleError, self._error_message

        service = self.service
        datatype_factory = DataTypeFactory()
        
        content = {}
        
        for param in service.parameters.keys():
            vdef = service.getDefaultValue(param)
	    try:
		value = self._value[param]
	    except KeyError:
                if vdef is not None:
                    value = vdef
                continue

            type = service.getParamClass(param)
            datatype = type.getDataType()

            if ClientJob.debug:
                print >>sys.stderr, "ClientJob _submit: Filling content with ", param, " type: ", type
            if type.isFile():
                if isinstance(value,Seq):
                    if self.debug:
                        print >>sys.stderr, "input sequence is an instance of Bio.Seq Seq class"
                    content[param] = value.data + "\n"
                    
                elif isinstance(value,SeqRecord):
                    if self.debug:
                        print >>sys.stderr, "input sequence is an instance of Bio.SeqRecord SeqRecord class"
                    if value.name == '<unknown name>':
                        content[param] = ">" + value.description + "\n"
                    else:
                        content[param] = ">" + value.name + "\n"
                    content[param] += value.seq.data + "\n"
                    
                elif isinstance(value,ClustalAlignment):
                    if self.debug:
                        print >>sys.stderr, "input sequence is an instance of Bio.Clustalw ClustalAlignment class"
                    content[param] = ""
                    for seq in value.get_all_seqs():
                        content[param] += ">" + seq.description + "\n"
                        content[param] += seq.seq.data + "\n"
                    
                else:
		    print >> sys.stderr, " path, fh or string ...?"
		    if os.path.exists(value):
			print >> sys.stderr, " path...?"
			if self.debug:
			    print >>sys.stderr, "input sequence is a file"
			content[param] = value
		    else:
			try:
			    res = value.isatty()
			    if self.debug:
				print >>sys.stderr, "input sequence is a filehandle"
			    content[param] = "".join(value.readlines())

			except Exception, e:
			    if self.debug:
				print >>sys.stderr, "input sequence is a string (", e, ")"
			    content[param] = value

            elif datatype_factory.issubclass(datatype, "Boolean"):
                if value:
                    content[param] = "on"
            else:
                 content[param] = value

        if ClientJob.debug:
            print >>sys.stderr, "ClientJob _submit: content: ", content
            
        boundary = "3350843711048987223263301088"
        postdata = "--" + boundary

        postdata += "\r\nContent-Disposition: form-data; name=\"Cmd\""
        postdata += "\r\n\r\n" + self.service.name + "\r\n"
        postdata += "--" + boundary

        postdata += "\r\nContent-Disposition: form-data; name=\"_action\""
        postdata += "\r\n\r\n" + 'getId' + "\r\n"
        postdata += "--" + boundary

        postdata += "\r\nContent-Disposition: form-data; name=\"programName\""
        postdata += "\r\n\r\n" + self.service.name + "\r\n"
        postdata += "--" + boundary

        postdata += "\r\nContent-Disposition: form-data; name=\"email\""
        postdata += "\r\n\r\n" + self.service.email + "\r\n"
        postdata += "--" + boundary

        datatype_factory = DataTypeFactory()

        for param in content.keys():
            type = service.getParamClass(param)
            datatype = type.getDataType()
            value = content[param]
            if self.debug:
                print >>sys.stderr, "Filling postdata with ", param, " type: ", type, " value: ", value

	    if self.service.getDataInputParam() == param:
		http_param = param + "_data"
	    else:
		http_param = param

            #if type == "Sequence" or type == "InFile" or type == "Alignment":
            if type.isFile():
                 if os.path.exists(value):
                    filename = os.path.basename(value)
		    postdata += "\r\nContent-Disposition: form-data; name=\"" + http_param + "\""
                    postdata += "; filename=\"" + filename + "\"\r\n"
                    postdata += "Content-Type: text/plain\r\n" + "\r\n"
                    f=open(value)
                    l=f.readline()
                    while l:
                        l=l[0:-1]
                        postdata += l + "\r\n"
                        l=f.readline()
                        
                    f.close()
                    postdata += l + "\r\n"
                    postdata += "--" + boundary
    
                 else: 
		    postdata += "\r\nContent-Disposition: form-data; name=\"" + http_param + "\""
                    postdata += "\r\n\r\n" + value + "\r\n"
                    postdata += "--" + boundary
            elif datatype_factory.issubclass(datatype, "Boolean"):
		postdata += "\r\nContent-Disposition: form-data; name=\"" + http_param + "\""
		if value:
		    postdata += "\r\n\r\n" + "1" + "\r\n"
		else:
		    postdata += "\r\n\r\n" + "0" + "\r\n"
                postdata += "--" + boundary
                
            elif datatype_factory.issubclass(datatype, "Integer") or datatype_factory.issubclass(datatype,"Float"):
                postdata += "\r\nContent-Disposition: form-data; name=\"" + http_param + "\""
                postdata += "\r\n\r\n" + str(value) + "\r\n"
                postdata += "--" + boundary
                
            else:
                postdata += "\r\nContent-Disposition: form-data; name=\"" + http_param + "\""
                postdata += "\r\n\r\n" + value + "\r\n"
                postdata += "--" + boundary


        postdata += "--\r\n"
        content_length = str(len(postdata))

        host = self._get_host(self.location)
        cgi = self._get_cgi(self.location)

        if self.debug:
            print >>sys.stderr, "Submitting to host: ", host, " cgi: ", cgi, " ..."
            
        ua = self._get_ua(host)
        ua.putrequest('POST', cgi)
        ua.putheader("Accept", "text/html")
        ua.putheader("User-Agent", "Python/Pise")
        ua.putheader("Content-Length", content_length)
        ua.putheader("Content-type", "multipart/form-data; boundary=" + boundary)
        ua.endheaders()

        if self.debug:
            print >>sys.stderr, "\npostdata:\n", postdata
            print >>sys.stderr, "\ncontent_length: ", content_length

        ua.send(postdata)

        ec, em, h = ua.getreply()
        if self.debug or ClientJob.debug:
            print >>sys.stderr, "ClientJob _submit: Request status: ", ec, " ", em
            
        if ec == 200:
            if self.debug:
                print >>sys.stderr, "HTTP POST ", cgi, " request successful"
                
            self._results = ua.getfile()
            #t = self._results.read()
            #print t
            #return
	    self._results = self._results.read()
	    print >> sys.stderr, "ClientJob _submit: self._results = " , self._results[:-1]
	    try:
		self.jobstate = Mobyle.JobState.JobState( self._results[:-1] )
                self.job_status, self.job_message = self.jobstate.getStatus()
                if self.job_status == "error":
                    self._error = True
                    self._error_message = self.job_message
                    raise MobyleError, self.job_message
	    except MobyleError, msg:
		self._error = True
		self._error_message = msg
		raise MobyleError, msg
	    #print >> sys.stderr, "ClientJob _submit: jobstate = " , self.jobstate , dir(self.jobstate)
	    return 

        else:
            self._error = True
            self._error_message = "Problem in HTTP request:" + em
            if self.debug: 
                print >>sys.stderr, "Problem in HTTP request:" + em
            raise MobyleError, "Problem in HTTP request: " + em



    # ------------------------------------------
    # Utilities for internal use.
    
    def _get_ua(self, host):
        if not self._ua:
            self._ua = HTTP(host)
        return self._ua

    def _get_host(self,url):
        location_re = re.compile('http://([^/]+)/(.*)')
        m = location_re.match(url)
    	if m:
    	    host = m.group(1)
    	    return host
    	else:
    	    raise JobError, "url: " + url + " not valid"

    def _get_uri(self,url):
        location_re = re.compile('http://([^/]+)(/.*)')
        m = location_re.match(url)
        uri = m.group(2)
        return uri
    
    def _get_filename(self,url):
        location_re = re.compile('http://[^/]+/.*/([^/]+)')
        m = location_re.match(url)
        if m:
            filename = m.group(1)
        else:
            filename = url
        return filename

    
    def _get_cgi(self,url):
        return self.location



class ClientServiceParser:
    """
    parse a mobyle service uri and build a L{CientService} instance
    """

    def __init__( self ):
    	self._service = None
        self.data = self._service

    def parse(self , serviceUri ):
        """
        parse a mobyle service uri and build a L{CientService} instance
        @param serviceUri: the uri of a mobyle service
        @type serviceUri: string
        """
        doc = Ft.Xml.Domlette.NonvalidatingReader.parseUri( serviceUri )
        mobyleNode = doc.xpath( './mobyle')
        if mobyleNode:
            try:
                serviceNode = doc.xpath( './mobyle/program')[0]
                self.parseProgram( serviceNode )
            except IndexError:
                try:
                    pipeNode = doc.xpath( './mobyle/pipeline')[0]
                    self.parsePipeline( pipeNode )
                except IndexError:
                    raise MobyleError , "a service must be a program or a pipeline"
             
        else:            
            try:
                serviceNode = doc.xpath( './program')[0]
                self.parseProgram( serviceNode )
            except IndexError:
                try:
                    pipeNode = doc.xpath( './program')[0]
                    self.parsePipeline( pipeNode )
                except IndexError:
                    raise MobyleError , "a service must be a program or a pipeline"
        self.data =  self._service    
                      
                    
    def parsePipeline( self , pipelineNode ):
        raise NotImplementedError , "todo"
    
    
    def parseProgram( self , programNode ):
        """
        parse a mobyle service and construct a L{ClientService}
        @param programNode: a node program in mobyle dtd
        @type programNode: a node object
        """
        name = programNode.xpath('./head/name/text()')[0].data
        self._service = ClientService( name = name )

        allParameterNodes = programNode.xpath( '//parameter' )

        for parameterNode in allParameterNodes:
            parameter = self.parseParameter(  parameterNode )
            
            self._service.parameters[ parameter[ 'name' ] ] = parameter
        
        
    def parseParameter(self , parameterNode ):
        """
        @param parameterNode: the none of a parameter
        @type parameterNode: node object
        @return : a dictionary containg the 'name ', the 'vdef', the 'value' and type of this parameter
        @rtype: dictionary
        """
        current_parameter = {}
        
        try:
            current_parameter[ 'name' ] = parameterNode.xpath( './name/text()')[0].data
        except IndexError:
            raise MobyleError , "a parameter must have a name "
          
        try:
          datainput =  str( parameterNode.xpath( './@ismaininput' )[0].value )
          if datainput == "1":
              self._service._datainput.append( current_parameter[ 'name' ] )
        except IndexError:
            pass
        
                     
        try:
            #attention vdef peut etre une liste de valeur ( clustalw )
            current_parameter[ 'vdef' ] = parameterNode.xpath( './vdef/value/text()')[0].data
        except IndexError:
            current_parameter[ 'vdef' ] = None
            
        current_parameter[ 'value' ]  = current_parameter[ 'vdef' ]
        
        
        try:
            bioTypes = parameterNode.xpath( "./type/biotype/text()" )
            bioTypes = [ bt.data for bt in bioTypes ]
        except IndexError :
            bioTypes = None
        
        
        formatList = parameterNode.xpath( "./type/acceptedDataFormats/dataFormat/text()" )
        
        if formatList:
            formats = [ fmt.data for fmt in formatList ]
            try:
                forceformat = parameterNode.xpath( "./type/acceptedDataFormats/@force" )[0].value 
            except IndexError:
                forceFormat = False
        else:
            formats = []
            forceFormat = False
        try:    
            card = str( parameterNode.xpath( "./type/card/text()" )[0].data )
            try:
                min , max = card.split(",")
            except ValueError:
                if len( card ) == 2:
                    min = max = card 
                else:
                    raise ParserError , "invalid card: %s .the card element must be a string of 2 integer or \"n\" separate by a comma : "% card 
            try:
                min = int( min )
            except ValueError :
                if min != "n":
                    raise ParserError , "invalid card: %s .the card element must be a string of 2 integer or \"n\" separate by a comma : "% card 
            try:
                max = int( max )
            except ValueError :
                if max != "n":
                    raise ParserError , "invalid card: %s .the card element must be a string of 2 integer or \"n\" separate by a comma : "% card 
            card = ( min , max )
        except IndexError :
            card = ( 1 , 1 )


        try:
            superKlass = parameterNode.xpath( "./type/datatype/superclass/text()" )[0].data
        except IndexError:
            superKlass = None
            
        try:
            klass = parameterNode.xpath( "./type/datatype/class/text()" )[0].data
        except IndexError:
            if superKlass is None :
                raise MobyleError , parameterNode.xpath( "./name/text()")[0].data + " must have either a \"class\" element either a \"class\" and \"superclass\" element"
            else:
                raise MobyleError , parameterNode.xpath( "./name/text()")[0].data + " if the \"superclass\" is specified the the \"class\" element must be also specified"
        
        dataTypeFactory =  DataTypeFactory()   
        
        try:
            if superKlass:
                dataType = dataTypeFactory.newDataType( superKlass , xmlName = klass )
            else:
                dataType = dataTypeFactory.newDataType( klass )
        except MobyleError , err :
            self.log.error( "an error occured during a parameter parsing: "+ str( err ) )
            raise MobyleError , err

        mobyleType = Mobyle.Service.MobyleType( dataType , bioTypes = bioTypes , formats = formats , forceFormat = forceFormat , card = card )
        current_parameter['type'] = mobyleType      
        return current_parameter
        
        
        
#===============================================================================
#        
#    def start_parameter(self, attributes=None):
#	self._current_parameter = {}
#	self._current_parameter['value'] = None
#	self._current_parameter['vdef'] = None
#	self._current_parameter['class'] = attributes['class']
# 
#    def end_parameter(self, *args):
#	if self._current_parameter['vdef'] is not None:
#	    self._current_parameter['value'] = self._current_parameter['vdef'] 
# 
#	self._service.parameters[self._current_parameter['name']] = self._current_parameter
#    def vdef(self, values):
#	self._current_parameter['vdef'] = values
# 
#    def param(self, value):
#	if value[2].find("/head/datainput/param") != -1: 
#	    self._service._datainput = value[0]
#===============================================================================




class ParamMethod: 
    """
    A way to trap AttributeError by redefining __getattr__ in
    ClientJob, which returns a ParamMethod for callable
    attributes (http://www.python.org/doc/essays/metaclasses/).
    """
    
    debug = 0
    
    def __init__(self, function, instance, param=None): 
        if ParamMethod.debug: 
            print >>sys.stderr, "ParamMethod __init__: function: ", function, " parameter: ", param  
        self.function = function 
        self.instance = instance 
        self.param = param 
        
    def __call__(self, *args): 
        if ParamMethod.debug: 
            print >>sys.stderr, "ParamMethod __call__: calling", self.function, "for", self.instance, "with parameter: ", self.param, " and: args: ", args 
        if self.param: 
            return apply(self.function, (self.instance, self.param,) + args) 
        else: 
            return apply(self.function, (self.instance,) + args) 

