"""
Module to manage central registry and local services directory
(prototype)
"""

import os
import os.path
import sys
from httplib import HTTP

try:
    MOBYLEHOME = os.environ[ 'MOBYLEHOME']
    if ( os.path.join( os.environ[ 'MOBYLEHOME' ] , 'Src' ) ) not in sys.path:
        sys.path.append(  os.path.join(os.environ[ 'MOBYLEHOME' ], 'Src' )  )
except KeyError:
    print >> sys.stderr , "the env vars MOBYLEHOME must be defined "
    raise KeyError, "the env vars MOBYLEHOME must be defined "

from Mobyle.MobyleError import *
from Mobyle.Utils import ServiceLocator

# which server(s) maintains the main registry 
registry_maintainers = ['bioserv.rpbs.jussieu.fr/cgi-bin/MobylePortal/registry.py', 'mobyle.pasteur.fr/cgi-bin/MobylePortal/registry.py']

# general registry of Mobyle portals
# the site maintains this file only if it it the registry maintainer or backup
# this file contains a list of portals url to query the registry
portals_file = os.path.join(MOBYLEHOME, 'Tools' , 'public_portals')


# services that are exported
exported_services_file = os.path.join(MOBYLEHOME, 'Local' , 'Config', 'exported_services')

# remote services that are allowed and available locally
imported_services_file = os.path.join(MOBYLEHOME, 'Local' , 'Config', 'imported_services')

# cache file
cache_file = os.path.join(MOBYLEHOME, 'Local' , 'Config', 'cached_public_portals')

# how to compose a service url?
root_service_url = ''


class AbstractRegistry:
    """
    Defines the interface for all the classes that query registry or directories
    """
    pass

class ClientPortal(AbstractRegistry):
    """
    This class manages imported remote services allowed for the local portal users

    - imported services are listed in MOBYLEHOME/Local/Config/imported_services
    syntax of this file:
    portal_name\tservice_name

    """

    def __init__(self, **args):
        self._load()

    def _load(self):
        try:
            fh = open(imported_services_file)
        except IOError, e:
            print e

        self.imported_services = {}
        for line in fh:
            if line[0] != '#' and len(line) > 1:
                #print "Registry/clientPortal/_load: line: ", line
                portal, service = line[:-1].split('\t')
                if self.imported_services.has_key(portal):
                    self.imported_services[portal].append(service)
                else:
                    self.imported_services[portal] = [ service ]
        fh.close()

    def get_portals(self):
        """
        Returns the list of portals which services may be imported by the local portal
        """
        return self.imported_services.keys()

    def get_portal_services(self, portal):
        """
        Returns the list of services run by the remote portal 
        """
        return self.imported_services[portal]

    def get_services(self):
        """
        Returns the total list of imported services (for all the portals) as a list of (portal, service) pairs
        """
        return self.imported_services.items()

    def get_service(self, portal, service):
        """
        Returns the service as an url/xml (url = the url of jobcgi.py)
        """
        
        service_url = ''
        service_xml = ''
        return service_url, service_xml

    def run_service(self, portal, service):
        """
        Launches the service remotely
        """
        pass


class Directory(AbstractRegistry):

    """
    For a given portal, manages the list of exported public services

    - must be available on each portal
    - public services are listed in MOBYLEHOME/Local/Config/public_services

    """

    def __init__(self, **args):
        self._load()

    def _load(self):
        try:
            fh = open(exported_services_file)
        except IOError, e:
            print e

        self.name = ""
        self.exported_services = []
        for service in fh:
            if service.find('portal:') == 0:
                label, name = service.split('portal:') 
                self.name = name.strip()
            elif service[0] != '#':
                self.exported_services.append(service[:-1])
        fh.close()


    def get_services(self):
        """
        Returns the services run by the local portal 
        """
        return self.exported_services

    def get_service(self, service):
        """
        Returns the service as an url/xml (url = the url of jobcgi.py)
        """
        
        serviceLocator = ServiceLocator()
        try:
            fileName = serviceLocator[ service ]
        except MobyleError, e:
            msg = "cannot find the \"" + service + "\" xml description file in Programs directory"
            raise MobyleError, msg

        service_url = self.name + 'jobcgi.py'
        fh = open(fileName)
        service_xml = fh.read()
        fh.close()
        return service_url, service_xml



class Registry(AbstractRegistry):

    """
    Manages the list of Mobyle portals

    - must be run on the central registry (which can be duplicated in fact)
    """


    def __init__(self, **args):
        self._load()

    def _load(self):
        try:
            fh = open(portals_file)
        except IOError, e:
            print e

        self.portals = []
        for portal in fh:
            if portal[0] != '#':
                self.portals.append(portal[:-1])
        fh.close()


    def get_portals(self):
        """
        Returns the list of portals
        """
        return self.portals


    def get_portal_services(self, portal):
        """
        Returns the list of services offered by portal
        """
        services = mobyle_requests.fetch_portal_services(portal)
        return services


    def get_services(self):
        """
        Returns the total list of services (for all the portals) as a list of (portal, service) pairs
        """
        services = {}
        for portal in self.portals:
            services[portal] = self.get_portal_services(portal)
        return services.items()

    def get_service(self, portal, service):
        """
        Returns the service of portal as an url/xml
        """
        service_url = ''
        service_xml = ''
        return service_url, service_xml

    def search_service(self, key):
        """
        Search the service in all portals
        Returns a list of (portal, service) pairs
        """
        for portal, service in self.get_services():
            if service == key:
                return portal, service
        return None, None

    def add_portal(self, portal):
        """
        Add a line in the registry file
        (simplistic for now isnt'it ?)
        """
        try:
            i = self.portal.index(portal)
            raise MobyleError, "portal " + portal + " already exist"
        except ValueError, e:
            pass

        try:
            fh = open(portals_file, 'a')
        except IOError, e:
            print e

        fh.write(portal + '\n')
        fh.close()



class RegistryProxy(AbstractRegistry):
    """
    Manages local treatment of remote registry requests
    Manages the cache and periodic updates of the cache
    """

    def __init__(self):
        self._load()

    def _load(self):
        try:
            fh = open(cache_file)
            self.portals = []
            for portal in fh:
                if portal[0] != '#':
                    self.portals.append(portal[:-1])
            fh.close()
        except IOError, e:
            print e


    def get_portals(self):
        """
        Returns the list of portals
        """
        return self.portals


    def get_portal_services(self, portal):
        """
        Returns the list of services offered by portal
        """
        services = mobyle_requests.fetch_portal_services(portal)
        return services

    def get_services(self):
        """
        Returns the total list of services (for all the portals) as a list of (portal, service) pairs
        """
        services = []
        for portal in self.portals:
            services.extend(self.get_portal_services(portal))
        return services


    def get_service(self, portal, service):
        """
        Returns the service of portal as an url/xml
        """
        pass

    def search_service(self, service):
        """
        Search the service in all portals
        Returns a list of (portal, service) pairs
        """
        pass


    def update_cache(self):
        """
        Updates the local portals file
        """
        self.portals = mobyle_requests.fetch_portals()
        try:
            fh = open(cache_file, 'w')
        except IOError, e:
            print e
        fh.write("\n".join(self.portals))
        fh.close()
        


class DirectoryRequestsAgent:

    
    def __init__(self):
        self._directory_agent = {}
        registry_maintainer = self.get_registry_maintainer()
        registry_host = registry_maintainer[:registry_maintainer.find('/')]
        self._registry_agent = HTTP(registry_host)
        self._registry_path = registry_maintainer[registry_maintainer.find('/'):]

    def get_registry_maintainer(self):
        """
        Simplistic (for now)
        """
        return registry_maintainers[0]

    def fetch_portals(self):
        """
        Query the registry
        """
        #print "Registry/MobyleDirectoryRequests/fetch_portals: ", registry_maintainer
        self._registry_agent.putrequest('GET', self._registry_path)
        self._registry_agent.putheader("Accept", "text/plain")
        self._registry_agent.putheader("User-Agent", "Python/Mobyle")
        self._registry_agent.endheaders()
        ec, em, h = self._registry_agent.getreply()
        if ec == 200:
            results_file = self._registry_agent.getfile()
            results = results_file.read()
            portals = []
            for portal in results.split("\n"):
                if portal != '':
                    portals.append(portal)
            return portals
        raise MobyleError, "Registry/MobyleDirectoryRequests/fetch_portals: an error has occured: " + em

    def fetch_portal_services(self, portal):
        """
        Query the remote portal directory
        """
        if not self._directory_agent.has_key(portal):
            portal_host = portal[:portal.find('/')]
            self._directory_agent[portal] = HTTP(portal_host)

        directory_agent = self._directory_agent[portal]
        portal_path = portal[portal.find('/'):]
            
        directory_agent.putrequest('GET', portal_path + '/directory.py')
        directory_agent.putheader("Accept", "text/plain")
        directory_agent.putheader("User-Agent", "Python/Mobyle")
        directory_agent.endheaders()
        ec, em, h = directory_agent.getreply()
        if ec == 200:
            results_file = directory_agent.getfile()
            results = results_file.read()
            services = []
            for service in results.split("\n"):
                if service != '':
                    services.append(service)
            return services

        raise MobyleError, "Registry/MobyleDirectoryRequests/fetch_portal_services: an error has occured: " + em



mobyle_requests = DirectoryRequestsAgent()
