#! /usr/bin/env python 

"""
Classes ecxuting the command and managing the results  
"""
import os ,os.path
from subprocess import Popen, PIPE
import signal , siginterrupt
import sys
import glob
import time
import cPickle
import zipfile , StringIO
#import Ft.Xml 
import Ft.Xml.Domlette

import logging ,logging.config



#"the environement variable MOBYLEHOME must be defined in the environemnt"
#append Mobyle Home to the search modules path
try:
    if ( os.path.join( os.environ[ 'MOBYLEHOME' ] , 'Src' ) ) not in sys.path:
        sys.path.append(  os.path.join( os.environ[ 'MOBYLEHOME' ] , 'Src' )  )
except KeyError:
    print >> sys.stderr , "the env vars MOBYLEHOME must be defined "


from Mobyle import MobyleLogger
MobyleLogger.MLogger( child = True )

rc_log = logging.getLogger('mobyle.runnerchild')


import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
import Mobyle.Net
import Mobyle.Session

import Local.Policy
from Mobyle.MobyleError import *


_cfg = Mobyle.ConfigManager.Config()
__extra_epydoc_fields__ = [('call', 'Called by','Called by')]


class AsynchronJob:
    """
    is instantiated in child process
      - should listen the father's signals
      - instantiate a Script
    when the script is finished
      - manage the results
    """

    
    def __init__(self, commandLine, dirPath, serviceName, resultsMask , session , email = None, jobState = None , xmlEnv = None):
        """
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (nomarly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param resultsMask: the unix mask to retrieve the results of this job
        @type resultsMask: a dictionary { paramName : [ string prompt , ( string class , string or None superclass ) , string mask ] }
        @param session: the session of this job
        @type session: L{Mobyle.Session} instance
        @param jobState: the jobState link to this job
        @type jobState: a L{JobState} instance
        @call: by the main of this module which is call by L{AsynchronRunner}
        """
        self._command = commandLine
        self._dirPath = dirPath
        self.serviceName = serviceName
        self.father_pid = os.getppid()
        self.father_done = False

        signal.signal( signal.SIGUSR2, self.sigFromFather )
 
        # encapsulated module to siginterrupt system.
        # see Mobyle/Utils/Siginterrupt/siginterrupt.c
 
        # this cause that the signalSIGUSR1 send to the child 
        # will not break the system call ( pipe.wait )
        # the child process is notified a sigusr2 has been received
        # but the handler will be executed at the end of the system call ( pipe wait )
        
        siginterrupt.siginterrupt( signal.SIGUSR2 , False )
        
        if jobState is None:
            self.jobState = Mobyle.JobState.JobState( os.getcwd() )
        
        self.email = email
                
        if self._dirPath[-1] == '/':
            self._dirPath = self._dirPath[:-1]

        self.jobKey = os.path.split( self._dirPath )[ 1 ]

        self._adm = Mobyle.Utils.Admin( self._dirPath )
        script = Script( self._command , self._dirPath , serviceName , session , jobState , xmlEnv )
        self.results = {}

        for paramName in resultsMask.keys():
            resultsFiles = []
            
            #type is a tuple ( klass , superKlass )
            prompt , Type , masks = resultsMask[ paramName ]

            for mask in masks :
                for File in  glob.glob( mask ):
                    size = os.path.getsize( File )
                    if size != 0:
                        resultsFiles.append(  ( str( File ) , size , None ) ) #we have not information about the output format 
            if resultsFiles: 
                self.results[ paramName ] = resultsFiles  #a list of tuple (string file name , int size ,  string format or None )
                self.jobState.setOutputDataFile( paramName , prompt , Type , resultsFiles )

        self.jobState.commit()
        try:
            zipFileName = self.zipResults()
        except Exception , err:
            import traceback, StringIO
            tmp_stderr = sys.stderr
            sys.stderr = StringIO.StringIO()
            traceback.print_exc()
            msg = "an error occured during the ziping results :\n\n" + sys.stderr.getvalue()
            rc_log.critical( "%s/%s : %s" %( self.serviceName , self.jobKey , msg ) )

            sys.stderr = tmp_stderr
            zipFileName = None
            
        if self.father_done:
            self.emailResults(  FileName = zipFileName )
        else:
            #print >> sys.stderr, "AsynchronJob stderr: mon pere est encore la, je le previens"
            try:
                os.kill( self.father_pid , signal.SIGUSR1 )
            except OSError , err :
                msg = "Can't send a signal to my father process:" + str( err )
                
                self._adm.setStatus( 5 , msg ) #error
                self._adm.commit

                self.jobState.setStatus( 5 , "Mobyle Internal Server Error" )
                self.jobState.commit()
                

                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.jobKey ,
                                                   msg
                                                   )
                                 )


                session.updateJobStatus( jobID = jobID  , status = 5 )
                raise MobyleError , msg


    def sigFromFather(self,signum ,frame):
        """
        @call: when the father is terminated and send a SIGUSR2 to the child process
        """
        self.father_done = True
        signal.signal(signal.SIGUSR2 , signal.SIG_IGN)


    def emailResults(self , FileName = None ):
        """
        """

        dont_email_result , maxmailsize =  _cfg.mailResults()
        if dont_email_result :
            return
        else:
            if self.email :
                subject = 'your job %s is finished. Access your results ' % ( self.serviceName )
                
                if FileName is not None:
                    zipSize = os.path.getsize( FileName )
    
                    if zipSize > maxmailsize - 2048 :
                        #2048 octet is an estimated size of email headers

                        msg = """
Your results are too big to be sent by email.
They are accessible at the following address:
%s

Your results will be kept on our server for %s days from now.""" % ( self.jobState.getID() ,  _cfg.remainResults() )
                        try:
                            self.email.sendMsg( subject , msg )
                            return
                        except EmailError ,err :
                            msg = str(err)
                            self._adm.setMessage( msg )
                            rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                                          self.jobKey ,
                                                          msg
                                                          )
                            )

                            return
                    else:
                        msg="""
Your results are accessible at the following address:
%s

Your results will be kept on our server for %s days from now.


An archive of your job is attached to this email. 
It includes the index.xml file which is a summary of your job submission 
(you can view it in your favorite web browser).
""" % ( self.jobState.getID() , _cfg.remainResults() )
                        try:   
                            self.email.sendFiles( subject , [ FileName ] , msg = msg )
                            return
                        except TooBig ,err :
                            msg = """
Your results are too big to be sent by email.
They are accessible at the following address:
%s

Your results will be kept on our server for %s days from now.""" % ( self.jobState.getID() ,  _cfg.remainResults() )
                                    
                        try:
                            self.email.sendMsg( subject , msg )
                        except EmailError ,err :
                            msg = str(err)
                            self._adm.setMessage( msg )
                            rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                                          self.jobKey ,
                                                          msg
                                                          )
                            )
                            
                            return
                else: #if there is a problem on zip creation
                    msg = """
Your results are accessible at the following address:
%s

Your results will be kept on our server for %s days from now.""" % ( self.jobState.getID() ,  _cfg.remainResults() )
                    self.email.sendMsg( subject , msg )
            else:
                return


    def zipResults(self ):

        resultsFiles = []

        for File in self.results.values():
            resultsFiles += File  #File is tuple (string file name , int size , string format or None )
       
        xsl_path = os.path.join( os.environ[ 'DOCUMENT_ROOT' ] , _cfg.htdocs_url() , "xsl" ,)
        xslSize = os.path.getsize( xsl_path )
        resultsFiles.append( ( "index.xml" , xslSize , None ) )
        
        jobXslPath = os.path.join( xsl_path , "job.xsl" ) 
        jobXslSize = os.path.getsize( jobXslPath )
        resultsFiles.append( ( jobXslPath , jobXslSize , None ) )
                
        mobyleXslPath = os.path.join( xsl_path , "mobyle.xsl" )
        mobyleXslSize = os.path.getsize( mobyleXslPath )
        resultsFiles.append( ( mobyleXslPath , mobyleXslSize , None ) )
        
        paramsfiles = self.jobState.getParamfiles()
        if paramsfiles: 
            for paramsfile in paramsfiles:
	    	resultsFiles.append(  ( paramsfile[0] , paramsfile[1] , None ) )
        
        inputFiles = self.jobState.getInputFiles() #inputFiles = [ ( parameter , [ (fileName , format or None ) , ...) , ... ]
        if inputFiles is not None:
            for files in inputFiles:
                for item in files[1]: #item = ( filename , size , fmt ) 
                    resultsFiles.append( item )
            
	zipFileName = "%s_%s.zip" %(self.serviceName , self.jobKey )
        myZipFile = zipfile.ZipFile( zipFileName  , "w" )
        
	for fileName , size , fmt in resultsFiles :
            #for python up to 2.5 
            #If allowZip64 is True zipfile will create ZIP files that use the ZIP64 extensions when the zipfile is larger than 2 GB.
            #If it is false (the default) zipfile will raise an exception when the ZIP file would require ZIP64 extensions. 
            #ZIP64 extensions are disabled by default because the default zip and unzip v5.52 commands on Unix  don't support these extensions.
            
            
            if size > 0 and size < 2147483648 : #2Go 
                if fileName  == 'index.xml':
                    tree = self._indexPostProcess()
                    myZipInfo = zipfile.ZipInfo( 'index.xml', time.localtime()[:6] )
                    myZipInfo.external_attr = 2175008768   # set perms to 644
                    myZipFile.writestr(  myZipInfo  , tree )
                elif size < 10 :
                    myZipFile.write(   fileName   , os.path.basename( fileName ) , zipfile.ZIP_STORED )
                else:
                    myZipFile.write(   fileName   , os.path.basename( fileName ) , zipfile.ZIP_DEFLATED )
                    
            elif size >= 2147483648 :
                myZipInfo = zipfile.ZipInfo( os.path.basename( fileName ) , time.localtime()[:6] )
                myZipInfo.external_attr = 2175008768   # set perms to 644
                myZipFile.writestr(  myZipInfo  , "Sorry this file result is too large ( > 2GiB ) to be included in this archive." )    
            else:
                #the file is empty we don't add it to this archive
                pass
            
        myZipFile.close()
        return  zipFileName


    def _indexPostProcess( self ):
        fh_tree = open( 'index.xml' , 'r')
        tree = ''.join( fh_tree.readlines() )
        doc = Ft.Xml.Domlette.NoExtDtdReader.parseString( tree )
        
        old_pi = doc.childNodes[0]
        new_pi = doc.createProcessingInstruction('xml-stylesheet' ,
                                                 'href="job.xsl" type="text/xsl"')
        doc.replaceChild( new_pi , old_pi)
        
        fh = StringIO.StringIO()
        Ft.Xml.Domlette.PrettyPrint( doc ,fh )
        tree = fh.getvalue()
        return tree 


class Script:
    """
    Choose de right class of Batch (in Config.Config.py) , instantiate it and do run()
    """

    def __init__(self,commandLine, dirPath, serviceName, session , jobState = None , xmlEnv = None ):
        """
        Choose de right class of Batch (in Config.Config.py) , instantiate it and do run()
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (nomarly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param session: the session link to this job
        @type session:  L{Mobyle.Session.Session} instance
        @param jobState:
        @type jobState: a L{JobState} instance
        @call: L{AsynchronJob.__init__}
        """
        batch = _cfg.batch( serviceName )
                
        if batch == "SGE":
            self.batch = SGE( commandLine , dirPath , serviceName , session , jobState = jobState , xmlEnv = xmlEnv)
        elif batch == "PBS":
            rc_log.debug("Considering a PBS submission for %s: %s" % (serviceName , commandLine) )
            self.batch = PBS( commandLine , dirPath , serviceName , session , jobState = jobState , xmlEnv = xmlEnv)
            # rc_log.debug("PBS submission batch instanced")
            
            # raise MobyleError,"try to instanciate PBS but PBS not yet implemented"
        elif batch == "LSF":
            raise MobyleError,"try to instanciate LSF but LSF not yet implemented"
        elif batch == "SYS":            
            self.batch = Sys( commandLine , dirPath , serviceName , session , jobState = jobState , xmlEnv = xmlEnv)
        else:
            msg = "unknown batch system : " + str(batch)
            rc_log.critical("%s : %s" %( serviceName ,
                                         msg
                                         )
            )

            if jobState is None:
                jobState = Mobyle.JobState.JobState( uri = dirPath )
            jobState.setStatus( 5 , msg = 'Mobyle internal server error' )
            jobState.commit()

            self._adm = Mobyle.Utils.Admin( dirPath )
            self._adm.setStatus( 5 , msg )
            self._adm.commit()

            session.updateJobStatus( jobID = jobID  , status = 5 )  
            raise MobyleError, msg


class _Batch:
    """
    abstract class
    manage the status by updating the file index.xml
    """

    def __init__( self , commandLine , dirPath , serviceName , session , jobState = None , xmlEnv = None ):
        """
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (nomarly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param session: the session link to this job
        @type session:  L{Mobyle.Session.Session} instance
        @param jobState:
        @type jobState: a L{JobState} instance
        @call: L{Script.__init__}

        """

        self.commandLine = commandLine
        
        self._protectedCommandLine = ''
        for c in self.commandLine:
            self._protectedCommandLine += '\\'+ c
            
        self.dirPath = dirPath
        self.serviceName = serviceName

        if not jobState :
            self.jobState = Mobyle.JobState.JobState( self.dirPath )

        jobID = self.jobState.getID()
        self.jobState.setStatus(3) #running
        self.jobState.commit()

        self._adm = Mobyle.Utils.Admin( self.dirPath )
        self._adm.setStatus( 3 )
        self._adm.commit()
        
        if xmlEnv:
            self.xmlEnv = xmlEnv
        else:
            self.xmlEnv = {}

        self.session = session
        self.session.updateJobStatus( jobID = jobID  , status = 3 )   
        queue = _cfg.queue( self.serviceName )
        
        self.queue = Local.Policy.queue( queue )

        new_path = ''
        binary_path = _cfg.binary_path()
        if binary_path :
                new_path = ":".join( binary_path )        

        if self.xmlEnv.has_key( 'PATH' ) :      
                new_path = "%s:%s" %( self.xmlEnv[ 'PATH' ] , new_path )

        if new_path :
            self.xmlEnv[ 'PATH' ] = "%s:%s" %( new_path , os.environ[ 'PATH' ] ) 
        else:
            self.xmlEnv[ 'PATH' ] = os.environ[ 'PATH' ] 
            
        for var in os.environ.keys():
            if var != 'PATH':
                self.xmlEnv[ var ] = os.environ[ var ]

        self._returncode = None
        accounting = _cfg.accounting()

        if accounting:
            beg_time = time.time()

        ###################################
        mobyleStatus  , statusMsg = self.run()
        ###################################
        
        if accounting:
            end_time = time.time()
            elapsed_time = end_time - beg_time
            
            a_log = logging.getLogger( 'mobyle.account' )
            #%d trunc time to second
            #%f for milisecond
            a_log.info("%s/%s %d-%d %d %s" %( serviceName ,
                                           _cfg.batch( serviceName ) ,
                                           beg_time ,
                                           end_time ,
                                           elapsed_time ,
                                           self._returncode 
                                           )
                       )
        
        self.jobState.setStatus( mobyleStatus  , msg = statusMsg )

        self.jobState.commit()
        self._adm.setStatus( mobyleStatus  )
        if statusMsg :
            self._adm.setMessage( statusMsg )
        self._adm.commit()


        if self.session:
            try:
                self.session.updateJobStatus( jobID = jobID  , status = mobyleStatus )
            except ( KeyError , IOError) , err :
                #this job doesn't exist in session
                #it was remove by user "remove this job" button
                if mobyleStatus != 6: #killed
                    rc_log.error( "%s/%s : try to update jobStatus but job does not exist in session %s" %(serviceName ,
                                                                                                           self.getKey(),
                                                                                                           self.session.getKey()    
                                                                                                           )
                    )
                    
                else:
                    #this job was killed 
                    #it could be the result of "remove this job" in web interface
                    #and thus the job was already remove from the session by session.removeJob ( 2 independant process )
                    rc_log.warning( "%s/%s : try to update jobStatus in session %s to %s have error %s" %(serviceName ,
                                                                                                          self.getKey(),
                                                                                                          self.session.getKey(),
                                                                                                          mobyleStatus,
                                                                                                          err )
                    )

                    pass
                
                

    def getKey( self ):
        """
        extract the Job key from the dirPath
        """
        if self.dirPath[-1] == '/':
            self.dirPath = self.dirPath[:-1]
        return os.path.split( self.dirPath )[1]
    
    def run( self ):
        """
        run the job
        abstract method. this method must be implemented in child classes
        """
        raise NotImplementedError, "Must be Implemented in child classes"

    @staticmethod
    def getStatus( number ):
        """
        @param number:
        @type number:
        @return the status of the job
        @rtype:
        abstract method. this method must be implemented in child classes
        """
        raise NotImplementedError, "Must be Implemented in child classes"

    @staticmethod
    def kill( self , number ):
        """
        kill the Job
        @param number:
        @type number:
        abstract method. this method must be implemented in child classe
        """
        raise NotImplementedError, "Must be Implemented in child classes"
            



    def _logError( self , admMsg = None , userMsg = None , logMsg = None ):

        if  admMsg :
            self._adm.setStatus( 5 , admMsg )
            self._adm.commit()

        if userMsg :
            self.jobState.setStatus( 5 , userMsg )
            self.jobState.commit()

        if logMsg :
            rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                          self.getKey() ,
                                          logMsg
                                          )
                        )

        if self.session:
            self.session.updateJobStatus( jobID = self.jobState.getID()  , status = 5 )



          
          
          
          
          
          ###########################################################
          #                                                         #
          #       to run with queuing system Sun Grid Engine        #
          #                                                         #
          ###########################################################
                
class SGE (_Batch):
    """
    
    """
        
    @staticmethod
    def _sgeInit( ):

        SGE_ROOT , SGE_CELL = _cfg.sge()
                    
        arch_path= os.path.join( SGE_ROOT , 'util' , 'arch' )

        try:
            arch_pipe = Popen(arch_path         ,
                              shell     = False ,
                              stdout    = PIPE  ,
                              stdin     = None  ,
                              stderr    = None  ,
                              close_fds = True  )

            arch_pipe.wait() 
            arch_rc = arch_pipe.returncode
            

        except OSError , err:
            #this error is log by calling method because I can't acces to jobKey , adm status ... from static method
            msg = "SGE: I can't determined the system arch:"+str(err)
            raise MobyleError , msg
     
        if arch_rc != 0 :
            msg = "I can't determined the system arch (return code = " + str( rc ) + " )"
            raise MobyleError , msg
        
        arch = ''.join( arch_pipe.stdout.readlines() ).strip()

        sge_prefix = os.path.join( SGE_ROOT , 'bin' , arch )
        sge_env = {'SGE_CELL': SGE_CELL , 'SGE_ROOT': SGE_ROOT }   
        return ( sge_prefix , sge_env )



    def run(self):
        """
        """
        if os.getcwd() != os.path.abspath( self.dirPath ):
            msg = "the child process execute itself in a wrong directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg 

        else:
            jobKey = self.getKey()
            
            options = { '-cwd': ''           ,  #set the working dir to current dir
                        '-now': 'n'          ,  #the job is not executed imediatly
                        '-N' : jobKey        ,  #the id of the job
                        '-q' : self.queue    ,  #the queue 
                        '-V' : ''               #job inherits of the whole environment
                        }

            sge_opts = ''

            for opt in options.keys():
                sge_opts += opt + ' ' + options[opt]+' '
            
            try:
                sge_prefix , sge_env = SGE._sgeInit()
            except MobyleError , err:
                self._logError( admMsg = str( err ) ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( str( err ) ) 
                    
                raise MobyleError , err
            
                      
            sge_cmd = os.path.join( sge_prefix , 'qrsh' ) + ' ' + sge_opts

            cmd = sge_cmd + " " + self._protectedCommandLine

            try:
                fout = open( self.serviceName + ".out" , 'w' )
                ferr = open( self.serviceName + ".err" , 'w' )
            except IOError , err:
                msg= "SGE: can't open file for standard job output: "+ str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = msg )

                raise MobyleError , msg

            self.xmlEnv.update( sge_env )
            
            try:
                pipe = Popen( cmd ,
                              shell  = True ,
                              stdout = fout  ,
                              stdin  = None  ,
                              stderr = ferr  ,
                              close_fds = True ,
                              env       = self.xmlEnv
                              )

            except OSError, err:
                msg= "SGE execution failed: "+ str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = msg )

                raise MobyleError , msg

            self._adm.setBatch( 'SGE')
            self._adm.setNumber( jobKey )
            self._adm.commit()
            
            linkName = ( "%s/ADMINDIR/%s.%s" %( _cfg.results_path() ,
                                                self.serviceName ,
                                                jobKey
                                                )
                         )

            try:
                os.link(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                msg = "can't create link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )

                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey  ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            pipe.wait()

            try:
                os.unlink( linkName )
            except OSError , err:
                msg = "can't remove link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )

                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            fout.close()
            ferr.close()

            ####################################
            #
            # verifier comment se comport sge vis vis des code de retours !!!
            #
            #####################################

            self._returncode = pipe.returncode
            
            if pipe.returncode == 0 :
                mobyleStatus = 4    # finished
                statusMsg = None
            elif pipe.returncode == 137 or pipe.returncode == 153: 
                ## 137 =  9 ( SIGKILL ) + 128 ( python add 128 )
                ## 153 = 25 ( SIGXFSZ ) + 128
                ## be careful if a job exit with a 137 or 153 exit status it will be labelled as killed 
                mobyleStatus = 6    # killed
                statusMsg = "Your job has been cancelled"
            else:
                mobyleStatus = 4    # finished
                statusMsg = "Your job finished with an unusual status code ( %s ), check your results carefully." % pipe.returncode 


            return ( mobyleStatus , statusMsg )


            


    @staticmethod
    def getStatus( jobNum ):
        """
        @param jobNum:
        @type jobNum:
        @return: the status of job with number jobNum
        @rtype: string
        @todo: for best performance, restrict the sge querying to one queue
        """
        sge_prefix , sge_env = SGE._sgeInit( )
        sge_cmd = [ os.path.join( sge_prefix , 'qstat' ) , '-r' ]
        
        try:
            pipe = Popen( sge_cmd,
                          executable = sge_cmd[0] ,
                          shell = False ,
                          stdout = PIPE ,
                          stdin = None  ,
                          stderr= None  ,
                          close_fds = True ,
                          env= sge_env
                          )
        except OSError , err:
            raise MobyleError , "can't querying sge : %s :%s "%( sge_cmd , err )

        sge2mobyleStatus = { 'r' : 'running' ,
                             't' : 'running' ,
                             'R' : 'running' ,
                             's' : 'hold' ,
                             'S' : 'hold' ,
                             'T' : 'hold' ,
                             'h' : 'hold' ,
                             'w' : 'pending' ,
                             'd' : 'killed' ,
                             'E' : 'error'
                             }

        for job in pipe.stdout :
            
            jobState = job.split()
            try:
               number = int( jobState[ 0 ] )
               status = jobState[ 4 ]
               continue
            except ( ValueError , IndexError ):
               pass #it's not the fisrt line

            try:
               if not ( jobNum == jobState[2] and 'Full' == jobState[0] ):
	               continue #it's not the right jobNum
            except IndexError :
               continue #it's not the 2nde line

            pipe.stdout.close()
            pipe.wait()
            
            if pipe.returncode == 0 :
                return sge2mobyleStatus[ status ]
            else:
                raise MobyleError , "cannot get status " + str( pipe.returncode )
        raise MobyleError , "no job : " + str( jobNum )


        
    @staticmethod
    def kill( job_number ):
        """
        @todo: kill the Job
        """
        sge_prefix , sge_env = SGE._sgeInit( )
        sge_cmd = "%s %s 1>&2" %( os.path.join( sge_prefix , 'qdel' ) ,  job_number )
        os.environ.update( sge_env )
        os.system( sge_cmd )
        

class PBS(_Batch):
    """
    """

    @staticmethod
    def _pbsInit( ):
        """
        To get some PBS local information
        """

        # rc_log.debug("PBS _pbsInit\n")
        pbs_prefix =  _cfg.pbs() # return PBS_ROOT
        PBS_ROOT   = pbs_prefix
        # rc_log.debug("PBS_prefix %s\n" % pbs_prefix)
        pbs_env = {'PBS_ROOT': PBS_ROOT }
        # rc_log.debug("PBS_env %s\n" % pbs_env)
        return ( pbs_prefix , pbs_env )
    
##         PBS_QUEUES = _cfg.pbs()
                    
##         arch_path= os.path.join(SGE_ROOT, 'util' , 'arch')

##         try:
##             arch_pipe = Popen(arch_path         ,
##                               shell     = False ,
##                               stdout    = PIPE  ,
##                               stdin     = None  ,
##                               stderr    = None  ,
##                               close_fds = True  )

##             arch_pipe.wait() 
##             arch_rc = arch_pipe.returncode
            

##         except OSError , err:
##             #this error is log by calling method because I can't acces to jobKey , adm status ... from static method
##             msg = "SGE: I can't determined the system arch:"+str(err)
##             raise MobyleError , msg
     
##         if arch_rc != 0 :
##             msg = "I can't determined the system arch (return code = " + str( rc ) + " )"
##             raise MobyleError , msg
        
##         arch = ''.join( arch_pipe.stdout.readlines() ).strip()

##         sge_prefix = os.path.join( SGE_ROOT , 'bin' , arch )
##         sge_env = {'SGE_CELL': SGE_CELL , 'SGE_ROOT': SGE_ROOT }   
##         return ( pbs_prefix , pbs_env )

    def run(self):
        """
        We build the pbs script and we execute it.
        Unlike SGE, PDB is asynchronous, so we must manage the wait until
        job is finished.
        """
        # rc_log.debug("entering PBS.run()")


        if os.getcwd() != os.path.abspath( self.dirPath ):
            msg = "the child process execute itself in a wrong directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg 

        else:
            jobKey = self.getKey()

        # PBS_QUEUES = _cfg.pbsqueues()
        # rc_log.debug("PBS queue: %s\n" % self.queue)
        self.queueproperties = _cfg.queueProperties(self.queue)
        if self.queueproperties == None:
            msg = "undefined pbs queue invoked %s" % self.queue

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg

        # PBS
        # rc_log.debug("PBS queueproperties get\n")
        options = { '-d': self.dirPath            ,  #set the working dir to current dir
                    '-m': 'n'            ,  # no email
                    '-N' : jobKey        ,  #the id of the job
                    '-q' : self.queue    ,  #the queue 
                    '-p' : '0'           ,  #the priority
                    '-V' : ''            ,  #job inherits of the whole environment
                    '-u' : self.queueproperties['uid'] , # owner of the job
                    '-S' : '/bin/bash'   
                    }

        pbs_opts = ''

        for opt in options.keys():
            pbs_opts += opt + ' ' + options[opt]+' '

        # rc_log.debug("PBS opts: %s\n" % pbs_opts)

        try:
            pbs_prefix, pbs_env = PBS._pbsInit()
        except MobyleError , err:
            self._logError( admMsg = str( err ) ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = None )

            rc_log.critical( str( err ) ) 

            raise MobyleError , err

        # rc_log.debug("PBS prefix: %s\n" % pbs_prefix)

        # We must write a script for remote queing
        fName = "%s/batch.sh" % os.getcwd()
        # rc_log.debug("PBS script file: %s\n" % fName)
        f = open(fName,"w")
        f.write("#!/bin/bash\n")
        f.write("#PBS -S %s\n" % self.queueproperties['shell'])
        f.write("#PBS -q %s\n" % self.queue)
        f.write("#PBS -o %s/%s.out\n" % (os.getcwd(), jobKey))
        f.write("#PBS -e %s/%s.err\n" % (os.getcwd(), jobKey))
        f.write("#PBS -m n\n")
        f.write("#PBS -p %s\n" % self.queueproperties['priority'])
        f.write("#PBS -u %s\n" % self.queueproperties['uid'])
        f.write("cd %s\n" % os.getcwd())
        f.write("%s\n" % self.commandLine)
        f.close()

        cmd = os.path.join( pbs_prefix , 'qsub' ) + ' ' + fName
        rc_log.debug("PBS system cmd: %s\n" % cmd)

        try:
            fout = open( self.serviceName + ".out" , 'w' )
            ferr = open( self.serviceName + ".err" , 'w' )
        except IOError , err:
            msg= "PBS: can't open file for standard job output: "+ str(err)

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        self.xmlEnv.update( pbs_env )

        try:
            pipe = Popen( cmd ,
                          shell  = True ,
                          stdout = PIPE  ,
                          stdin  = None  ,
                          stderr = PIPE  ,
                          close_fds = True ,
                          env       = self.xmlEnv
                          )
        except OSError, err:
            msg= "PBS execution failed: "+ str(err)

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        # We get the pbsId
        pbsId = pipe.stdout.read()
        err   = pipe.stderr.read()
        if err != '':
            msg= "PBS execution returned an error: "+ str(err)

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        pbsId = pbsId.split()[0]
        if self.queueproperties['type'] == "execution":
            pbsKey = pbsId + "@localhost"
        elif self.queueproperties['type'] == "routage":
            pbsKey = pbsId + "@"+ self.queueproperties['remote']
        self.pbsKey = pbsKey


        # rc_log.debug("PBS key: %s\n" % pbsKey)

        pipe.wait()
        rc = pipe.returncode
        if rc != 0 :
            raise MobyleError , "error in pbs submission status"


        self._adm.setBatch( 'PBS')
        self._adm.setNumber( pbsKey )
        self._adm.commit()


        # rc_log.debug("PBS job id: %s\n" % pbsKey)


        linkName = ( "%s/ADMINDIR/%s.%s" %( _cfg.results_path() ,
                                            self.serviceName ,
                                            jobKey
                                            )
                     )

        try:
            os.link(
                os.path.join( self.dirPath , '.admin') ,
                linkName
                )
        except OSError , err:
            msg = "can't create link %s in ADMINDIR: %s" %( linkName , err )

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = None )

            rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                               jobKey  ,
                                               msg
                                               )
                             )

            raise MobyleError , msg

        # Unlike SGE, PBS is asynchronous. We already have the hand.
        # We must perform active wait.
        # We wait for the job to complete
        # rc_log.debug("PBS wait loop\n")
        self._pbswait(delay = 5)
        # rc_log.debug("PBS job completed\n")
        
        # To debug
        mobyleStatus = 4
        statusMsg = None
        return ( mobyleStatus , statusMsg )

        try:
            os.unlink( linkName )
        except OSError , err:
            msg = "can't remove link %s in ADMINDIR: %s" %( linkName , err )

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = None )

            rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                               jobKey ,
                                               msg
                                               )
                             )

            raise MobyleError , msg

        fout.close()
        ferr.close()

        ####################################
        #
        # verifier comment se comport pbs vis vis des code de retours !!!
        #
        #####################################


        if pipe.returncode == 0 :
            mobyleStatus = 4    # finished
            statusMsg = None
        elif pipe.returncode == 137: ## be careful if a job exit with a 137 exit status it will be labelled as killed 
            mobyleStatus = 6    # killed
            statusMsg = "Your job has been cancelled"
        else:
            mobyleStatus = 4    # finished
            statusMsg = "This job finished with an unusual status code ( %s ). be careful to your results" % pipe.returncode 


        return ( mobyleStatus , statusMsg )

    def _pbswait(self, delay = 5):
        """
        This method will wait for the submitted job (asynchronous) to be completed
        We can parameterize the delay between the refreshment of the
        information (60 seconds)
        """
        counter = 0
        while self._isCompleted() == 0:
##             fd = open("/tmp/pbs.dbg","w")
##             fd.write("Try %d\n" % counter)
##             fd.close()
            time.sleep(delay)
            counter += 1
        time.sleep(delay)
##         fd = open("/tmp/pbs.dbg","a")
##         fd.write("Done (2) ...\n")
##         fd.close()
        return

    def _isCompleted(self):
        """
        This queries the queueing system using qstat to determine if
        the job is completed
        """
        import os, string

        try:
            status = self.getStatus(self.pbsKey)
            # rc_log.debug("PBS wait status : %s\n" % status )
            return 0
        except:
            return 1

    @staticmethod
    def getStatus( pbsKey ):
        """
        @param pbsKey:
        @type pbsKey:
        @return: the status of job with number pbsKey
        @rtype: string
        @todo: for best performance, restrict the sge querying to one queue
        PBS qstat return if completed:
        qstat: Unknown Job Id 19347.bioserv.rpbs.jussieu.fr
        """

        pbs_prefix , pbs_env = PBS._pbsInit( )

        remote = ""
        if pbsKey.count("@"):
            if not pbsKey.count("localhost"):
                remote = pbsKey.split("@")[1]
        jobId = pbsKey.split("@")[0]
        if pbsKey.count(".") > 1:
            theIndex = pbsKey.index(".",pbsKey.index(".")+1)
            jobId = pbsKey[:theIndex]
        jobHandle = jobId
        if remote != "":
            jobHandle = jobId + "@" + remote

        pbs_cmd = os.path.join( pbs_prefix , 'qstat' )
        pbs_cmd = pbs_cmd + " " + jobHandle

        try:
            pipe = Popen( pbs_cmd,
                          shell = True ,
                          stdout = PIPE ,
                          stdin = None  ,
                          stderr= PIPE ,
                          close_fds = True ,
                          env= pbs_env
                          )
            pipe.wait()
            rc = pipe.returncode

        except OSError , err:
            raise MobyleError , "can't query pbs : %s :%s "%( pbs_cmd , err )
        if rc != 0 :
            raise MobyleError , "error in pbs status querying"

        pbs2mobyleStatus = { 'r' : 'running' ,
                             't' : 'running' ,
                             'R' : 'running' ,
                             's' : 'hold' ,
                             'S' : 'hold' ,
                             'T' : 'hold' ,
                             'h' : 'hold' ,
                             'w' : 'pending' ,
                             'd' : 'killed' ,
                             'E' : 'exiting',
                             'F' : 'Finished' # Not used
                             }

        err = pipe.stderr.read()
        pipe.stderr.close()
        if err.count("Unknown"):
            # return pbs2mobyleStatus[ 'F' ]
            raise MobyleError , "no job : " + str( pbsKey )
        
        pbsOutLines = pipe.stdout.readlines()

        for line in pbsOutLines :
            if line.count(jobId):
                jobState = line.split()
                try:
                   status = jobState[ 4 ]
                   pipe.stdout.close()
                   return pbs2mobyleStatus[ status ]
                   break
                except ( ValueError , IndexError ):
                   pass #it's not the fisrt line

        raise MobyleError , "no job : " + str( pbsKey )

    @staticmethod
    def kill( pbsKey ):
        """
        @todo: kill the Job
        """
        pbs_prefix , pbs_env = PBS._pbsInit( )


        remote = ""
        if pbsKey.count("@"):
            if not pbsKey.count("localhost"):
                remote = pbsKey.split("@")[1]
        jobId = pbsKey.split("@")[0]
        if pbsKey.count(".") > 1:
            theIndex = pbsKey.index(".",pbsKey.index(".")+1)
            jobId = pbsKey[:theIndex]
        jobHandle = jobId
        if remote != "":
            jobHandle = jobId + "@" + remote

        pbs_cmd = "%s %s 1>&2" %( os.path.join( pbs_prefix , 'qdel' ) ,  jobHandle )
        os.environ.update( pbs_env )
        os.system( pbs_cmd )




          ###############################################
          #                                             #
          #       to run without queuing system         #
          #                                             #
          ###############################################


class Sys(_Batch):
    """
    Run the commandline by a system call
    """

    def run(self):
        """
        Run the commandLine in a subshell (system call) and redirect the standard error and output on service.name.out and service.name.err, then restore the sys.stderr and sys.stdout
        @return: the L{Mobyle status code <Mobyle.Utils.status2code>} of this job and a message
        @rtype: ( int , string )
        @call: in _Batch.__init__
        """

        if (os.getcwd() != os.path.abspath(self.dirPath) ):
            msg = "the process is not in the working directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        else:
            jobKey = self.getKey()
            fout = open( self.serviceName + ".out" , 'w' )
            ferr = open( self.serviceName + ".err" , 'w' )

            try:
                #the new process launch by popen must be a session leader 
                # because it launch in backrgound and if we send a siggnal to the shell 
                # this signal will no propagate to all child process
                # and if we kill the processe leader we kill the python 
                # thus we execute our setsid which transform the new process as a session leader
                # and execute the command prior to return.
                # in these conditions we could kill the process leader and the signal is propagate to
                # all processes belonging to this session. 

                setsidPath = '%s/Tools/setsid' %os.environ[ 'MOBYLEHOME']
                command = [ setsidPath , setsidPath , '/bin/sh' , '-c' , self.commandLine ]
                pipe = Popen( command ,
                              shell     = False ,
                              stdout    = fout ,
                              stdin     = None ,
                              stderr    = ferr ,
                              close_fds = True ,
                              env       = self.xmlEnv
                              )


            except OSError, err:
                msg= "System execution failed: " + str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg 
                
            jobPid = pipe.pid
            
            self._adm.setBatch( 'Sys' )
            self._adm.setNumber( jobPid )
            self._adm.commit()


            linkName = ( "%s/ADMINDIR/%s.%s" %( _cfg.results_path() ,
                                                self.serviceName ,
                                                jobKey
                                                )
                         )

            
            try:
                os.link(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                msg = "can't create link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            #pipe.wait() doesn't work with signal.signal SIGUSR1 & SIGUSR2
            #OSError: [Errno 4] Interrupted system call
            #RESOLVED with used siginterrupt.siginterrupt( signal , False )

            pipe.wait()

            try:
                os.unlink( linkName )
            except OSError , err:
                msg = "can't remove link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            fout.close()
            ferr.close()
            self._returncode = pipe.returncode
            
            if pipe.returncode == 0 :
                mobyleStatus = 4    # finished
                statusMsg = None
            elif pipe.returncode == -15 or pipe.returncode == -9:
                mobyleStatus = 6    # killed
                statusMsg = "Your job has been cancelled"
            else:
                mobyleStatus = 4    # finished
                statusMsg = "This job finished with a status code = " + str( pipe.returncode )

            return ( mobyleStatus , statusMsg )
        

    @staticmethod
    def getStatus( pid ):
        """
        @param pid:
        @type pid:
        @return: the status of job with pid pid
        @rtype: string
        """
        pid = int( pid )
        try:
            os.kill( pid , signal.SIG_DFL )
        except OSError , err:
            if str( err ).find( 'No such process' ) != -1:
                return 'finished'
            else:
                raise OSError ,err 

        return 'running'


    @staticmethod
    def kill( job_pid ):
        """
        kill a job
        @param job_pid: the pid of the job
        @type job_pid: int
        @raise MobyleError: if can't kill the job
        @call: by L{Utils.Mkill}
        """
        try:
            job_pid = int( job_pid )
            job_pgid = os.getpgid( job_pid )
        except ValueError:
            raise MobyleError , "can't kill job  - %s" % err 
        try:
            os.killpg( job_pgid , signal.SIGTERM )
        except OSError , err:
            raise MobyleError , "can't kill job  - %s" % err 
        
               
        try:
            time.sleep(0.2)
            os.killpg( job_pgid , signal.SIG_DFL )
            try:
                os.killpg( job_pgid , signal.SIGKILL )
            except OSError , err:
                raise MobyleError , "can't kill job - %s" % err
                
        except OSError , err:
            return
        
class WSWrapper(_Batch):
    """
    """
    def run(self):
        raise NotImplementedError ,"ToDo"

class CGIWrapper(_Batch):
    """
    """
    def rub(self):
        raise NotImplementedError ,"ToDo"

if __name__ == '__main__':
    try:
        fh = open(".forChild.dump", "r")
        fromFather = cPickle.load( fh )
        fh.close() 
    except IOError ,err:
        pass
    except :
        pass

    try:
        os.chdir( fromFather[ 'dirPath' ] )
    except OSError, err:
        msg = fromFather[ 'serviceName' ] + ":" + str( err )
        rc_log.critical( msg )
        raise MobyleError ,msg

    email = fromFather[ 'email']
    if email is not None:
        email = Mobyle.Net.Email( email  , _cfg )
        
    child = AsynchronJob( fromFather[ 'commandLine' ] ,
                          fromFather[ 'dirPath' ] ,
                          fromFather[ 'serviceName' ] ,
                          fromFather[ 'resultsMask'] ,
                          fromFather[ 'session' ] ,
                          email = email  ,
                          xmlEnv = fromFather[ 'xmlEnv' ] ,
                          )
