import os , os.path
from time import localtime, strftime , strptime

import logging
u_log = logging.getLogger( 'mobyle.utils' )

from Mobyle.MobyleError import MobyleError , UserValueError


status2code = {"building" : 0 ,
               "submitted": 1 ,
               "pending"  : 2 ,
               "running"  : 3 ,
               "finished" : 4 ,
               "error"    : 5 , 
               "killed"   : 6 ,
               "hold"     : 7
               }

code2status = {0 : "building"  ,
               1 : "submitted" ,
               2 : "pending"   ,
               3 : "running"   ,
               4 : "finished"  ,
               5 : "error"     , 
               6 : "killed"    ,
               7 : "hold"     
              }


class Admin:
    """
    manage the informations in the .admin file.
    be careful there is no management of concurent access file
    thus if there is different instance of Admin with the same path
    it could be problem of data integrity
    """
    FIELDS = ( 'DATE'    ,
               'EMAIL'   ,
               'REMOTE'  ,
               'SESSION' ,
               'JOBNAME' ,
               'JOBID'   ,
               'MD5'     ,
               'BATCH'   ,
               'NUMBER'  ,
               'QUEUE'   ,
               'STATUS'  ,
               'MESSAGE'
               )

    
    def __init__(self , path ):
        self.me = {}
        path = os.path.abspath( path )
        
        if os.path.exists( path ):

            if os.path.isfile( path ):
               self.path = path
               self._parse()


            elif os.path.isdir( path ):
                self.path = os.path.join( path , ".admin" )

                if os.path.isfile( self.path ):
                    self._parse()
                else:
                    pass
                    
        else:
            basename , filename = os.path.split( path )
            if filename == ".admin":
                if os.path.exists( basename ):
                    self.path = path
                else:
                    raise MobyleError , "The path to the .admin is invalid : "+ basename
            else:
                raise MobyleError , "invalid path : " + path
                

            
        
    
                
    def __str__( self ):
        res = ''
        for key in self.__class__.FIELDS :
            try:
                if key == 'DATE':
                    value = strftime( "%x %X" , self.me['DATE'] )

                elif key == 'STATUS':
                    value = code2status[ self.me[ key ] ]
                else:
                    value =  self.me[ key ]
            except KeyError :
                continue
            
            res = res + "%s : %s\n" %( key , value )

        return res



    def _parse ( self ):
        """
        parse the file .admin
        """
        try:
            fh = open( self.path , 'r' )

            for line in fh:
                datas = line[:-1].split( ' : ' )
                key = datas[0]
                value = ' : '.join( datas[1:] )
                if key == 'DATE':
                    value = strptime( value , "%x %X" )
                elif key == 'STATUS':
                    value = status2code[ value ]
                self.me[ key ] = value
        finally:
            fh.close()

                 
    def refresh( self ):
        self._parse()

    def commit( self ):
        """
        Write the string representation of this instance on the file .admin
        """
        fh = open( self.path , 'w' )
        fh.write( self.__str__() )
        fh.close()


    def getDate( self ):
        """
        @return: the date of the job submission
        @rtype: time.struct_time
        """
        try:
            return  self.me[ 'DATE' ] 
        except KeyError :
            return None
 

    def setDate( self , date = None , format = None):
        """
        set the date of the job submission
        @param date: the date of the job submission. if date is None use localtime 
        @type date: a time.struct_time or a string
        @param format: if the date is a string represent the format of date. if it's None use \"%a %b %d %H:%M:%S %Y\" as default format.
        @type format: string
        """
        
        if date is None:
            myDate = localtime() 

        elif type( date ) == type( '' ) :
            if format is not None:
                myDate = strptime( date , format ) 
            else:
                myDate = strptime( date ) 
                    
        elif type( date ) == type( localtime( 0 ) ):
            myDate =  date 

        self.me[ 'DATE' ] = myDate



    def getEmail( self ):
        """
        @return: the email of the job 
        @rtype: string
        """
        try:
            return self.me[ 'EMAIL' ]
        except KeyError :
            return None


    def setEmail( self , email ):
        """
        set the email of the job
        @param jobName: the email of the job
        @type jobName: string
        """
        self.me[ 'EMAIL' ] = email


    def getRemote( self ):
        """
        @return: the remote of the job 
        @rtype: string
        """
        try:
            return self.me[ 'REMOTE' ]
        except KeyError :
            return None


    def setRemote( self , remote_ip ):
        """
        set the  of the job
        @param jobName: the  of the job
        @type jobName: string
        """
        self.me[ 'REMOTE' ] = remote_ip


    def getSession( self ):
        """
        @return: the Session path of the job 
        @rtype: string
        """
        try:
            return self.me[ 'SESSION' ]
        except KeyError :
            return None


    def setSession( self , sessionKey ):
        """
        set the  Session path of the job
        @param jobName: the  of the job
        @type jobName: string
        """
        self.me[ 'SESSION' ] = sessionKey



    def getJobName( self ):
        """
        @return: the name of the job ( blast2 , toppred )
        @rtype: string
        """
        try:
            return self.me[ 'JOBNAME' ]
        except KeyError :
            return None


    def setJobName( self , jobName ):
        """
        set the name of the job
        @param jobName: the name of the job
        @type jobName: string
        """
        self.me[ 'JOBNAME' ] = jobName

    def getJobID( self ):
        """
        @return: the job identifier.
        @rtype: string
        """
        try:
            return self.me[ 'JOBID' ]
        except KeyError :
            return None


    def setJobID( self , jobID ):
        """
        set the identifier of the job
        @param : the identifier of the job
        @type : string
        """
 
        self.me[ 'JOBID' ] = jobID


    def getMd5( self ):
        """
        @return: the md5 of the job
        @rtype: string
        """
        try:
            return self.me[ 'MD5' ]
        except KeyError :
            return None


    def setMd5( self , md5 ):
        """
        set the md5 of the job
        @param : the identifier of the job
        @type : string
        """
 
        self.me[ 'MD5' ] = md5



    def getBatch( self ):
        """
        @return: the batch system name used to run the job  
        @rtype: string
        """
        try:
            return self.me[ 'BATCH' ]
        except KeyError :
            return None


    def setBatch( self , batch):
        """
        set the  batch system used to run of the job
        @param : the name of the batch system 
        @type : string
        """
        self.me[ 'BATCH' ] = batch


    def getNumber( self ):
        """
        @return: the number the job the meaning of this value depend of BATCH value.
         - BATCH = Sys number is the job pid
         - BATCH = SGE number is the result of qstat
        @rtype: string
        """
        try:
            return self.me[ 'NUMBER' ]
        except KeyError :
            return None


    def setNumber( self , number ):
        """
        set the number of the job this number depend of the batch value
        if BATCH = Sys number is the pid
        if BATCH = SGE number is the 
        @param : the number of the job
        @type : string
        """
        self.me[ 'NUMBER' ] = number

    def getQueue( self ):
        """
        @return: return the queue name of the job 
        @rtype: string
        """
        try:
            return self.me[ 'QUEUE' ]
        except KeyError :
            return None


    def setQueue( self, queue ):
        """
        set the queue name of the job
        @param : the queuename of the job
        @type : string
        """
        self.me[ 'QUEUE' ] = queue


    def getStatus( self ):
        """
        @return: the job status. the values could be \"summitted\" ,\"pending\",\"running\" ,\"finished\",\"error\ ,\"killed\"
        @rtype: string
        """
        try:
            status = code2status[ self.me[ 'STATUS' ] ]
        except KeyError :
            status = None
        try:
            msg = self.me[ 'MESSAGE' ]
        except KeyError :
            msg = None

        return ( status , msg )
        

    def setStatus( self , statusCode , msg =None):
        """
        set the  status of the job
        @param : the status code of the job.The allowed value are [0-7] 
        @type : int
        """
        if statusCode in code2status.keys():
            try:
                self.me[ 'STATUS' ] = statusCode
            except KeyError:
                raise MobyleError, "status code out of range:" + str( statusCode )
            if msg:
                self.me[ 'MESSAGE' ] = msg
        else:
            raise MobyleError, "status code out of range:" + str( statusCode )

        
    def getMessage( self ):
        """
        @return: the job status. the values could be \"summitted\" ,\"pending\",\"running\" ,\"finished\",\"error\"
        @rtype: string
        """
        try:
            return self.me[ 'MESSAGE' ]
        except KeyError :
            return None


    def setMessage( self , message ):
        """
        set a  message used when status == error
        @param : a error message
        @type : string
        """
        self.me[ 'MESSAGE' ] = message




    
def getStatus( jobID ):
    """
    @param jobID: the url of the job
    @type jobID: string
    @return: the current status of the job
    @rtype: string
    @raise MobyleError: if the job has no number or if the job doesn't exist anymore
    @raise OSError: if the user is not the owner of the process
    @todo: tester la partie sge
    the different value for a status could be
     - building , the job is in preparation (creation of job environment etc...)
     - submitted , the job is ready to run (at the call of Job.run)
     - pending , the job is queued by the batch system   
     - running , the job is effectively running
     - hold , the job execution is hold by the mobyle administartor
     - finished , the job is finished
     - error , an error occured and the job is finished
     - killed , the job was killed by the mobyle administrator
     """
    
    import Mobyle.JobState
    import Mobyle.RunnerChild

    path = Mobyle.JobState.normUri( jobID )
    adm = Admin( path )
    oldStatus , msg = adm.getStatus()
    
    #'killed' , 'finished' , 'error' the status cannot change anymore
    #'submitted', 'building' these jobs have not yet sge number

    if oldStatus in ( 'killed' , 'finished' , 'error' , 'building' ):
        return ( oldStatus , msg )
    
    else:
        batch = adm.getBatch()
        jobNum = adm.getNumber()
        
        if batch is None :
            import Mobyle.MobyleError
            raise Mobyle.MobyleError.MobyleError , "BATCH is unknown for job : " + str( jobID )
        if jobNum is None:
            import Mobyle.MobyleError
            raise Mobyle.MobyleError.MobyleError ,"NUMBER is unknown for job : " + str( jobID )
        
        if batch == ( "Sys" ):
            newStatus = Mobyle.RunnerChild.Sys.getStatus( jobNum )
        elif batch == ( "PBS" ):
            newStatus = Mobyle.RunnerChild.PBS.getStatus( jobNum )
        elif batch == ( "SGE" ):
            newStatus = Mobyle.RunnerChild.SGE.getStatus( jobNum )
        else:
            raise NotImplementedError, "unknown batch \""+ batch +"\" "

        if newStatus != oldStatus:
            
            adm.setStatus( status2code[ newStatus ] )
            adm.commit()

            jobState = Mobyle.JobState.JobState( jobID )
            jobState.setStatus( status2code[ newStatus ] ) 
            jobState.commit()
            
        msg = adm.getMessage()
        return ( newStatus , msg )



def Mkill( keys ):
    import Local.Config.Config
    import string
    import types
    import glob
    import Mobyle.RunnerChild
    from Mobyle.MobyleError import MobyleError

    cfg = Mobyle.ConfigManager.Config()

    keysType = type( keys )

    if keysType == types.StringType :
       keys  = [ keys ]
    elif not( keysType == types.TupleType or keysType == types.ListType ) :
        raise MobyleError , "keys must be a string or a Sequence of strings"
    
    errors = []

    for key in keys:

        if len( key ) == 15  and key[0] in string.ascii_uppercase :
            try:
                int( key[1:] )
            except ValueError:
                errors.append( ( key , "invalid key" ) )
                continue
            else:
                search = "%s/ADMINDIR/*.%s" %( Local.Config.Config.RESULTS_PATH , key )
                admins = glob.glob( search )
                
                if len( admins ) == 1 :
                    adminPath = admins[0]
                elif len( admins ) == 0:
                    errors.append( ( key , "no running job with key : " + key  ) )
                    continue
                else:
                    raise MobyleError , "there is 2 running job with the same key : " + key 

            try:
                adm = Admin( adminPath )
            except MobyleError , err :
                errors.append( ( key , "invalid key" ) )
                continue
        
            batch = adm.getBatch()
            jobNum = adm.getNumber()
            job = adm.getJobName() + "/" + key 
            
            if jobNum is None:
                # an error occured on this jobID but I continue to kill the other jobIDs
                errors.append( ( job , 'no jobNum for this job' ) )
                continue

            try:

                try:
                    killer = "Mobyle.RunnerChild.%s.kill( '%s' )" %( batch , jobNum )
		    eval( killer )
                except ( NameError , AttributeError ) , err:
                    errors.append( ( job , "unknown batch \""+ batch +"\" " ) )
                    continue
                except Exception , err :
                    errors.append( ( job , str( err )  ) )
                    continue
                    
            except MobyleError ,err :
                errors.append( ( job , str( err ) ) )

        else: # len(key) != 15
            errors.append( ( key , "invalid key" ) )
            continue
  

    if errors:
        msg = ''
        for job , msgErr in errors :
            msg = "%s Mkill( %s ) - %s\n" %( msg , job , msgErr )

        raise MobyleError , msg

        
    



def killJob( jobID ):
    """
    @param jobID: the url of the job or a sequence of jobID
    @type jobID: string or sequence of jobID
    @return: 
    @rtype: 
    @raise MobyleError: if the job has no number or if the job doesn't exist anymore
    @todo: tester la partie sge
    """
    
    import Mobyle.RunnerChild
    import types
    from Mobyle.MobyleError import MobyleError
    
    jobIDType = type( jobID )
    
    if jobIDType == types.StringType :
        jobIDs = [ jobID ]
    elif jobIDType == types.TupleType or jobIDType == types.ListType :
        jobIDs = jobID
    else:
        raise MobyleError , "jobID must be a string ar a Sequence of strings"
    
    errors = []
    
    for jobID in jobIDs :
        try:
            path = Mobyle.JobState.normUri( jobID )
        except MobyleError , err :
            errors.append( ( jobID , str( err ) ) )
            continue
        
        if path[:4] == 'http' :
            #the jobID is not on this Mobyle server
            errors.append( ( jobID , "can't kill distant job" ) )
            continue

        try:
            adm = Admin( path )
        except MobyleError , err :
            errors.append( ( jobID , "invalid jobID" ) )
            continue
        
        batch = adm.getBatch()
        jobNum = adm.getNumber()
        if jobNum is None:
            # an error occured on this jobID but I continue to kill the other jobIDs
            errors.append( ( jobID , 'no jobNum for this job' ) )
            continue

        try:
            if batch == ( "Sys" ):
                Mobyle.RunnerChild.Sys.kill( jobNum )            
            elif batch == ( "SGE" ):
                Mobyle.RunnerChild.SGE.kill( jobNum )
            else:
                # an error occured on this jobId but I continue to kill the other jobId
                error.append( ( jobID , "unknown batch \""+ batch +"\" " ) )
                continue
        except MobyleError ,err :
            errors.append( ( jobID , str( err ) ) )
        
        
    if errors:
        msg = ''
        for jobID , msgErr in errors :
            msg = "%s killJob( %s ) - %s\n" %( msg , jobID , msgErr )
            
        raise MobyleError , msg



def safeFileName( fileName ):
    import string , re
    
    if fileName in ( 'index.xml' , '.admin' , '.command' ,'.forChild.dump' ,'.session.dump'):
        raise UserValueError , "value \"" + str( fileName ) + "\" is not allowed"
    
    for car in fileName :
        if car not in string.printable : #we don't allow  non ascii char
            fileName = fileName.replace( car , '_')
    
    #SECURITY: substitution of shell special characters
    fileName = re.sub( "[ #\"\'<>&\*;$`\|()\[\]\{\}\?\s ]" , '_' , fileName )
                  
    #SECURITY: transform an absolute path in relative path
    fileName = re.sub( "^.*[\\\:/]", "" , fileName )
    
    return fileName




def makeService( progName ):
    """
    Instantiate a Service object corresponding to the progName
    @param progName: the name of a service.
    @type progName: string
    @return: a L{Service} object corresponding to progName
    @rtype: a L{Service} instance
    @raise MobyleError: L{MobyleError} if the progName doesn't correspond to any xml file, or if
    the user Ip is not authorized to acces at this service.
    @call : L{Service} , L{CGI}
    """
    from Ft.Lib import Uri , UriException
    
    import Mobyle.Parser
    
    try:
        serviceLocator = ServiceLocator()
        servicePath = serviceLocator[ progName ]
    except KeyError :
        msg = "the service : \"%s\", can't be accessible from your IP" % progName
        raise UserValueError( msg = msg )

    try:
        parser = Mobyle.Parser.ServiceParser()
        serviceUri = Uri.OsPathToUri( servicePath )
        service = parser.parse( serviceUri )
        return service
    
    except UriException, err:
        raise MobyleError , str(err)


    
def sizeFormat(bytes, precision=2):
     """Returns a humanized string for a given amount of bytes"""
     import math
     bytes = int(bytes)
     if bytes is 0:
         return '0 bytes'
     log = math.floor( math.log( bytes , 1024 ) )
     return "%.*f%s" % ( precision , bytes / math.pow(1024, log), [ 'bytes', 'KiB', 'MiB' , 'GiB' ][ int(log) ] )
 
 
 
 
class ServiceLocator( object ):
    """
    manage paths services
    c'est ici que l'on pourra mettre un system de restriction d'acces aux interfaces
    (interfaces avec licences campus etc...)
    """
    import glob
    import os
    import Mobyle.ConfigManager
    
    _ref = None
    _cfg = Mobyle.ConfigManager.Config()


    def __new__( cls ):
      
        if cls._ref is None:
            self = super( ServiceLocator , cls ).__new__( cls )
            
            self._serviceInfo = {} 

            for path in ServiceLocator.glob.glob( os.path.join ( os.environ['MOBYLEHOME'] , "Local" , "Programs" ,"*.xml" ) ):
                
                self._serviceInfo [ os.path.basename( path )[:-4] ] = path

            for path in ServiceLocator.glob.glob( os.path.join ( os.environ['MOBYLEHOME']  , "Programs" ,"*.xml" ) ):
                service = os.path.basename( path )[:-4]

                if not self._serviceInfo.has_key( service ):
                    self._serviceInfo [ service ] = path

                    
            cls._ref = self
            
        return cls._ref



    def __getitem__( self, serviceName ):
        """
        @param serviceName: the name of the service
        @type serviceName: string
        @return: the path of the xml describing the service
        @rtype: string
        @raise: keyError if serviceName doesn't match with any services
        """
        return self._serviceInfo[ serviceName ]
        

    def iterServices( self ):
        """
        @return: a iterator on all services names
        """
        for s in self.services() :
            yield s

    def services( self ):
        """
        @return: the list of all service name
        @rtype : list of string
        """
        services = self._serviceInfo.keys()
        services.sort()
        return services

    def has_service( self, serviceName ):
        """
        @param serviceName: the name of the service
        @type serviceName: string
        @return: True if thlocator has a service named serviceName ,False otherwise
        @rtype: boolean
        """
        return self._serviceInfo.has_key( serviceName )

    def items( self ):
        """
        @return: a copy of the ServiceLocator's list of ( serviceName, path) pairs.
        @rtype: [ ( string , string ) ]
        """
        return [ ( s ,self._serviceInfo[ s ] )  for s in self._serviceInfo.keys() ]

    def iteritems( self ):
        """
        @return: an iterator on ServiceLocator's list of ( serviceName, path) pairs.
        """
        for v in self.items():
            yield v

    def paths( self ):
        """
        @return: a copy of serviceLocator's list of paths.
        """
        return self._serviceInfo.values()
    
    def iterPath( self ):
        """
        @return: an iterator on ServiceLocator's list of paths.
        """
        for path in self.paths():
            yield path



