/**
 * @class MobyleProgramsList is a MobyleWidget class that represents the list of
 * available programs on the server (and potentially elsewhere). It includes a search field
 * that lets users filter the programs using a search string.
 * It includes:
 * <ul>
 * <li>cgi call to get the list of programs</li>
 * <li>Tree behaviour initialization</li>
 * </ul>
 *
 * @author Herv Mnager
 * @organization Pasteur Institute
 * @contact mobyle@pasteur.fr
 * @extends MobyleWidget
 */
MobyleProgramsList = Class.create();
MobyleProgramsList.prototype = Object.extend({

    el:'programs_list_channel',

    cgiurl:'programs_list.py',

    initAjaxParams: function() {
      try{
      }catch(err){
        reportMobyleError(err);
      }
    },

	onCompleteFunction: function() {
      try{
        // initializing search form values and backing them up
        if(this.cgiparams && this.cgiparams.searchString){
          $("searchString").value = this.cgiparams.searchString;
          this.previousSearchString = this.cgiparams.searchString;
        }else{
          this.previousSearchString = '';
        }
        if(this.cgiparams && this.cgiparams.fullSearch){
          var fsEl = $("fullSearch");
          for(i=0; i<fsEl.options.length; i++){
            if(fsEl.options[i].value.strip()==this.cgiparams.fullSearch.strip()){
              fsEl.selectedIndex=i;
            }
          }
          this.previousFullSearch = this.cgiparams.fullSearch;
        }else{
          this.previousFullSearch = "on";
        }
  		this.initProgramNodes();
  		this.initCategoryNodes();
        Event.observe("programSearch", 'submit', 
          function(e){
		  	$("programSearch").disable();
			if($("programsList")){
				$("programsList").disable();
			}
            this.submitSearch();
            Event.stop(e);
        }.bindAsEventListener(this));    
        if($("programsList")){
          Event.observe("programsList", 'click', 
            function(e){
		  	  $("programSearch").disable();
			  if($("programsList")){
				$("programsList").disable();
			  }
              this.displayAll();
              Event.stop(e);
          }.bindAsEventListener(this));    
        }
      }catch(err){
        reportMobyleError(err);
      }
	},

	initProgramNodes: function() {
      try{
        var mpl = this;
        this.el.select(".tree_program").each(
          function(element){
            new MobyleTreeProgramNode(element, mpl);
          }
        );
      }catch(err){
        reportMobyleError(err);
      }
	},

	initCategoryNodes: function() {
      try{
  		this.el.select(".tree_category").each(
          function(element){
            new MobyleTreeCategoryNode(element);
          }
        );
      }catch(err){
        reportMobyleError(err);
      }
	},
    
    submitSearch: function() {
      try{
        // before to set the timeout, we check if the search parameters
        // have really changed
        if(this.previousSearchString!=$F("searchString").strip()){
          p = new Object();
          p.searchString = $F("searchString");
          this.cgiparams = p;
          this.build();
        }
      }catch(err){
        reportMobyleError(err);
      }
    },

    displayAll: function() {
      try{
        var p = new Object();
        this.cgiparams = p;
        this.build();
      }catch(err){
        reportMobyleError(err);
      }
    }
	
},MobyleWidget.prototype);

treeNodeMethods = {
	
    showTreeNode: function(element){
        element = $(element);
	  	element.ancestors().each(
			function(ancestorEl){
				if(ancestorEl.hasClassName("tree_category")){
					Element.removeClassName(ancestorEl,'tree_close');
				}
			}
		);
    },
	
    hideTreeNode: function(element){
        element = $(element);
	  	element.ancestors().each(
			function(ancestorEl){
				if(ancestorEl.hasClassName("tree_category")){
					Element.addClassName(ancestorEl,'tree_close');
				}
			}
		);		
    }

}

Element.addMethods(treeNodeMethods);	

/**
 * @class MobyleTreeProgramNode initializes the 
 * behaviour of a program link in the MobyleProgramsList
 *
 * @author Herv Mnager
 * @organization Pasteur Institute
 * @contact mobyle@pasteur.fr
 * @extends MobyleWidget
 */
MobyleTreeProgramNode = Class.create();
MobyleTreeProgramNode.prototype = {
	initialize: function(el) {
      try{
		this.el = el;
		this.initializeBehaviour();		
		Object.extend(this,this.el);
      }catch(err){
        reportMobyleError(err);
      }
	},
	
	initializeBehaviour: function(el) {
      try{
		this.linkEl = this.el.select(".program_link").first(); // N.B.: There is only one link for a category
        Event.observe(this.linkEl, 'click', 
          function(e){
            this.showService();
            Event.stop(e);
        }.bindAsEventListener(this));    
      }catch(err){
        reportMobyleError(err);
      }
	},

    showTooltip: function(){
      try{
        if(this.timeOutTooltipId){
          clearTimeout(this.timeOutTooltipId);
        }
        this.timeOutTooltipId=setTimeout((function() {Element.show(this.tooltipEl)}).bind(this), 500);
      }catch(err){
        reportMobyleError(err);
      }
    },

    hideTooltip: function(){
      try{
        if(this.timeOutTooltipId){
          clearTimeout(this.timeOutTooltipId);
        }
        Element.hide(this.tooltipEl);
      }catch(err){
        reportMobyleError(err);
      }
    },

    moveTooltip: function(e){
      try{
        Element.setStyle(this.tooltipEl, {
          position:"absolute",
          top:(Event.pointerY(e) - e.currentTarget.offsetParent.offsetTop) + "px",
          left:(Event.pointerX(e) - e.currentTarget.offsetParent.offsetLeft) + "px"
        });
      }catch(err){
        reportMobyleError(err);
      }
    },

	showService: function(){
      try{
        // get the program name corresponding to the form that has to be generated
        // create a form tab by invoking the tab's id (program name here)
        var programName = this.linkEl.innerHTML; // TODO the real name of the program could be stored in the href attribute of the link
		$('services_tabPanel').getFormTab(programName);
      }catch(err){
        reportMobyleError(err);
      }
	}
},

/**
 * @class MobyleTreeCategoryNode initializes the 
 * behaviour of a category node in the MobyleProgramsList
 *
 * @author Herv Mnager
 * @organization Pasteur Institute
 * @contact mobyle@pasteur.fr
 * @extends MobyleWidget
 */
MobyleTreeCategoryNode = Class.create();
MobyleTreeCategoryNode.prototype = {
	initialize: function(el) {
      try{
		this.el = el;
		this.initializeBehaviour();
      }catch(err){
        reportMobyleError(err);
      }
	},
	
	initializeBehaviour: function(el) {
      try{
		this.linkEl = this.el.select(".category_link").first(); // N.B.: There is only one link for a category
        Event.observe(this.linkEl,'click',
          function(e){
            this.toggleTreeStyle();
            Event.stop(e);
          }.bindAsEventListener(this));
        if($("searchString").value==''){
         this.toggleTreeStyle();        
        }
      }catch(err){
        reportMobyleError(err);
      }
	},
	
	toggleTreeStyle: function(){
      try{
    	if(!Element.hasClassName(this.el,'tree_close')){
          Element.addClassName(this.el,'tree_close');
    	}else{
          Element.removeClassName(this.el,'tree_close');
    	}
        return false;
      }catch(err){
        reportMobyleError(err);
      }
	}
}
