#############################################################
#                                                           #
#   Author: Herve Menager                                   #
#   Organization:'Biological Software and Databases' Group, #
#                Institut Pasteur, Paris.                   #
#   Distributed under GPLv2 Licence. Please refer to the    #
#   COPYING.LIB document.                                   #
#                                                           #
#############################################################
"""
Registry.py

This module holds:
- The Registry class that describes the list of installed/imported programs 
  and their servers
- The DefNode class, an abstract class that describes an element (node or
  leaf) of the programs tree
- The ProgramDef class that describes a program (leaf in the programs tree)
- The CategoryDef class that describes a category (node in the programs tree)
- The ServerDef class that describes a server (node in the programs tree)
"""
import os
import md5
import glob

import Mobyle.ConfigManager
from Mobyle.MobyleError import MobyleError

import logging
from Mobyle import MobyleLogger
MobyleLogger.MLogger()
r_log = logging.getLogger('mobyle.registry' )

_cfg = Mobyle.ConfigManager.Config()

class Registry(object):
    
    def __init__(self):
        self.load()
        
    def load(self, selectedPrograms=None):
        self.programs = []
        self.servers = []
        self.serversByName = {}
        self.serversByUrl = {}
        self.programsByUrl = {}
           
        serverProperties = self._getServerProperties()
        for name, properties in serverProperties.items():
            server = ServerDef( name = name,
                                url = properties['url'] ,
                                help = properties['help'],
                                repository = properties['repository'],
                                jobsBase = properties['jobsBase']
                               )
            self._addServer(server)
            for name in properties['programs']:
                try:                        
                    url = self.getProgramUrl(name, server.name)
                    if not(selectedPrograms) or (selectedPrograms and url in selectedPrograms.keys()):
                        path = self.getProgramPath(name, server.name)
                        program = ProgramDef( name= name,
                                              url= url,
                                              path=path,
                                              server= server
                                              )
                        # here we check the actual existence of the local file 
                        # corresponding to the imported program
                        if os.path.exists(self.getProgramPath(name, server.name)):
                            program.installed = True
                        else:
                            program.installed = False
                        # program disabled
                        if _cfg.isDisabled(programName=url): 
                            program.disabled = True
                        # program with restricted access
                        if not(_cfg.isAuthorized(url)):
                            program.authorized = False
                        self._addProgram(program)
                except IndexError, e:
                    r_log.error("Error while loading %s: %s" % (name, e))

    def _addServer(self, server):
        self.servers.append(server)
        self.serversByName[server.name] = server
        self.serversByUrl[server.url] = server

    def _addProgram(self, program):
        self.programs.append(program)
        self.programsByUrl[program.url] = program
        self.serversByName[program.server.name].programs.append(program)
        self.serversByName[program.server.name].\
            programsByName[program.name] = program

    def pruneProgram(self, program):
        if self.programsByUrl.has_key(program.url):
            self.programs.remove(program)
            del self.programsByUrl[program.url]
            self.serversByName[program.server.name].programs.remove(program)
            del self.serversByName[program.server.name].\
                programsByName[program.name]
            if (len(program.server.programs)==0):
                del self.serversByName[program.server.name]
                self.servers.remove(program.server)

    def _getServerProperties(self):
        localPaths = glob.glob(os.path.join(_cfg.programs_path(), "*.xml"))
        localNames = [os.path.basename(programPath)[:-4] \
                      for programPath in localPaths]
        serverProperties = _cfg.portals()
        serverProperties['local'] = {
           'url': _cfg.cgi_url(),
           'help' : _cfg.mailHelp(),
           'repository': _cfg.programs_url(),
           'programs': localNames,
           'jobsBase': _cfg.results_url()
        }
        return serverProperties
    
    def getProgramUrl(self, name, server='local'):
        return "%s/%s.xml" % \
            ( self.serversByName[server].repository , name )
        
    def getProgramPath(self, name, server='local'):
        if server != 'local':
            url = self.getProgramUrl(name, server)
            return os.path.join(_cfg.programs_imports_path(), \
                                md5.new(url).hexdigest()+'.xml')
        else:
            return os.path.join(_cfg.programs_path(), name+'.xml')
        
    def getServerByJobId(self,jobId):
        for server in self.servers:
            if jobId.startswith(server.jobsBase):
                return server

class DefNode(object):
    """
    DefNode is the class that provides the tree structure to the 
    server/categories/program hierarchy
    """

    def __init__(self):
        self.name = None
        self.children = []
        self.parents = []
              
    def getDescendants(self):
        """
        Returns the descendants of the node recursively
        @return: the list of descendant nodes
        @rtype: list
        """
        r = []
        r.extend(self.children)
        for p in self.children:
            r.extend(p.getDescendants())
        return r

    def getAncestors(self):
        """
        Returns the ancestors of the node recursively
        @return: the list of ancestor nodes
        @rtype: list
        """
        r = []
        r.extend(self.parents)
        for p in self.parents:
            r.extend(p.getAncestors())
        return r

    def addDescendant(self, subpath):
        """
        Adds a descendant to the node, by recursively adding the descendants from the provided subpath if they do not exist
        @param subpath: the subpath, a list of descendant objects 
                        (e.g.: [CategoryDef, CategoryDef, ProgramDef])
        @type subpath: string
        """
        if len(subpath)==0:
            return
        else:
            c = [c for c in self.children if c.name == subpath[0].name]
            if not(c):
                c = subpath[0]
                self.children.append(c)
                c.parents.append(self)
            else:
                c = c[0]
            c.addDescendant(subpath[1:])
            
    def __cmp__(self, other):
        """
        Comparison operator that allows to classify the tree structure alphabetically
        based on node names
        @param other: the other node to which self is compared
        @type other: DefNode
        @return: the comparison result
        @rtype: boolean
        """
        return cmp(self.name, other.name)
    
    def getChildCategories(self):
        """
        Returns child categories, i.e. child nodes which happen to be CategoryDef instances
        @return: a list of category nodes
        @rtype: list
        """
        l = [c for c in self.children if isinstance(c, CategoryDef)]
        l.sort()
        return l

    def getChildPrograms(self):
        """
        Returns child programs, i.e. child nodes which happen to be ProgramDef instances
        @return: a list of program nodes
        @rtype: list
        """
        l = [c for c in self.children if isinstance(c, ProgramDef)]
        l.sort()
        return l

class ProgramDef(DefNode):
    """
    ProgramDef is the class that provides the program information to the registry
    """
    
    def __init__(self, url, name=None, path=None, server=None):
        """
        @param url: the url of the definition of the program on the server where it is executed
        @type url: String
        @param name: the name of the program 
        @type name: String
        @param path: the os path to the local version of the file (local file definition or local cache of distant program)
        @type path: String
        @param server: the execution server definition
        @type server: ServerDef
        """
        DefNode.__init__(self)
        self.url = url
        self.name = name
        self.path = path
        self.server = server
        self.categories = []
        self.disabled = False
        self.authorized = True
        
    def isExported(self):
        return self.server.name == 'local' and \
            self.name in _cfg.exported_programs()

class ServerDef(DefNode):
    """
    ServerDef is the class that provides the server information to the registry
    """

    def __init__(self, name, url, help, repository, jobsBase):
        """    
        @param name: the short name, as displayed for instance in the programs tree menu
        @type name: String
        @param url: the url of the server (cgi-bin base url for now)
        @type url: String
        @param help: the help email contact
        @type help: String
        @param repository: the repository url
        @type repository: String
        @param jobsBase: the url of the directory that contains all of the server's jobs
        @type jobsBase: String
        """
        DefNode.__init__(self)
        self.name = name
        self.url = url
        self.help = help
        self.repository = repository
        self.programs = [] # top-level programs
        self.programsByName = {}
        self.categories = [] # top-level categories
        if self.name == 'local':
            self.path = _cfg.programs_path()
        else:
            self.path = _cfg.programs_imports_path()
        self.jobsBase = jobsBase

class CategoryDef(DefNode):
    """
    CategoryDef is the class that provides the category information to the registry
    """

    def __init__ (self, name, parentCategory=None, server=None):
        """    
        @param name: the name of the category
        @type name: String
        @param parentCategory: the parent category (if any)
        @type parentCategory: CategoryDef
        @param server: the server to which the Category belongs
        @type server: ServerDef
        """
        DefNode.__init__(self)
        self.name = name
        self.parentCategory = parentCategory
        self.server = server
        self.programs = [] # top-level programs
        self.categories = [] # top-level categories
        
registry = Registry()