#! /usr/bin/env python


########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


"""
Classes executing the command and managing the results  
"""
import os ,os.path
from subprocess import Popen, PIPE
import signal , siginterrupt
import sys
import glob
import time
import cPickle
import zipfile , StringIO
import Ft.Xml.Domlette
import atexit

import logging ,logging.config

#"the environment variable MOBYLEHOME must be defined in the environment"
#append Mobyle Home to the search modules path
MOBYLEHOME = None
if os.environ.has_key('MOBYLEHOME'):
    MOBYLEHOME = os.environ['MOBYLEHOME']
if not MOBYLEHOME:
    sys.exit('MOBYLEHOME must be defined in your environment')

if ( os.path.join( MOBYLEHOME , 'Src' ) ) not in sys.path:
    sys.path.append(  os.path.join( MOBYLEHOME , 'Src' )  )


from Mobyle import MobyleLogger
MobyleLogger.MLogger( child = True )

rc_log = logging.getLogger('mobyle.runnerchild')


import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
import Mobyle.Net

import Local.Policy
from Mobyle.MobyleError import *


_cfg = Mobyle.ConfigManager.Config()
__extra_epydoc_fields__ = [('call', 'Called by','Called by')]


class AsynchronJob:
    """
    is instantiated in child process
      - should listen the father's signals
      - instantiate a Script
    when the script is finished
      - manage the results
    """

    
    def __init__(self, commandLine, dirPath, serviceName, resultsMask , userEmail = None, jobState = None , xmlEnv = None):
        """
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (normaly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param resultsMask: the unix mask to retrieve the results of this job
        @type resultsMask: a dictionary { paramName : [ string prompt , ( string class , string or None superclass ) , string mask ] }
        @param jobState: the jobState link to this job
        @type jobState: a L{JobState} instance
        @call: by the main of this module which is call by L{AsynchronRunner}
        """
        self._command = commandLine
        self._dirPath = dirPath
        self.serviceName = serviceName
        self.father_pid = os.getppid()
        self.father_done = False

        signal.signal( signal.SIGUSR2, self.sigFromFather )
 
        # encapsulated module to siginterrupt system.
        # see Mobyle/Utils/Siginterrupt/siginterrupt.c
 
        # this cause that the signal SIGUSR1 send to the child 
        # will not break the system call ( pipe.wait )
        # the child process is notified a sigusr2 has been received
        # but the handler will be executed at the end of the system call ( pipe wait )
        
        siginterrupt.siginterrupt( signal.SIGUSR2 , False )
        
        if jobState is None:
            self.jobState = Mobyle.JobState.JobState( os.getcwd() )
        
        self.userEmail = userEmail
                
        if self._dirPath[-1] == '/':
            self._dirPath = self._dirPath[:-1]

        self.jobKey = os.path.split( self._dirPath )[ 1 ]
        
        atexit.register( self.childExit , "------------------- %s : %s -------------------" %( serviceName , self.jobKey ) )
     
        self._adm = Mobyle.Utils.Admin( self._dirPath )
        script = Script( self._command , self._dirPath , serviceName , jobState , xmlEnv )
        self.results = {}

        for paramName in resultsMask.keys():
            resultsFiles = []
            
            #type is a tuple ( klass , superKlass )
            prompt , Type , masks = resultsMask[ paramName ]

            for mask in masks :
                for File in  glob.glob( mask ):
                    size = os.path.getsize( File )
                    if size != 0:
                        resultsFiles.append(  ( str( File ) , size , None ) ) #we have not information about the output format 
            if resultsFiles: 
                self.results[ paramName ] = resultsFiles  #a list of tuple (string file name , int size ,  string format or None )
                self.jobState.setOutputDataFile( paramName , prompt , Type , resultsFiles )

        self.jobState.commit()
        try:
            zipFileName = self.zipResults()
        except Exception , err:
            msg = "an error occured during the zipping results :\n\n"
            rc_log.critical( "%s/%s : %s" %( self.serviceName , self.jobKey , msg ) , exc_info = True)
            zipFileName = None
            
        if self.father_done:
            self.emailResults(  FileName = zipFileName )
        else:
            #print >> sys.stderr, "AsynchronJob stderr: mon pere est encore la, je le previens"
            try:
                os.kill( self.father_pid , signal.SIGUSR1 )
            except OSError , err :
                msg = "Can't send a signal to my father process send results by email :" + str( err )
                rc_log.warning( msg )
                self.emailResults(  FileName = zipFileName )
                

    def childExit(self , message ):
        print >> sys.stderr , message
        if not self.father_done:
            os.kill( self.father_pid , signal.SIGUSR1 )

    def sigFromFather(self,signum ,frame):
        """
        @call: when the father is terminated and send a SIGUSR2 to the child process
        """
        self.father_done = True
        signal.signal(signal.SIGUSR2 , signal.SIG_IGN)


    def emailResults(self , FileName = None ):
        """
        """

        dont_email_result , maxmailsize =  _cfg.mailResults()
        if dont_email_result :
            return
        else:
            if self.userEmail :
                mail = Mobyle.Net.Email( self.userEmail )
                jobInPortalUrl = "%s/portal.py?jobs=%s" %( _cfg.cgi_url() ,
                                                           self.jobState.getID() )
                
                if FileName is not None:
                    zipSize = os.path.getsize( FileName )
                    mailDict = { 'SENDER'         : _cfg.sender() ,
                                 'HELP'           : _cfg.mailHelp() ,
                                 'SERVER_NAME'    : _cfg.portal_url() ,
                                 'JOB_URL'        : jobInPortalUrl , 
                                 'RESULTS_REMAIN' : _cfg.remainResults() ,
                                 'JOB_NAME'       : self.serviceName 
                                 }
                    if zipSize > maxmailsize - 2048 :
                        #2048 octet is an estimated size of email headers
                        try:
                            mail.send( 'RESULTS_TOOBIG' , mailDict )
                            return
                        except EmailError ,err :
                            msg = str(err)
                            self._adm.setMessage( msg )
                            rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                                          self.jobKey ,
                                                          msg
                                                          )
                            )

                            return
                    else:
                        try:   
                            mail.send( 'RESULTS_FILES' , mailDict , files = [ FileName ]  )
                            return
                        except TooBig ,err :
                            try:
                                mail.send( 'RESULTS_TOOBIG' , mailDict )
                            except EmailError ,err :
                                msg = str(err)
                                self._adm.setMessage( msg )
                                rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                                              self.jobKey ,
                                                              msg
                                                              )
                                )
                            
                            return
                else: #if there is a problem on zip creation
                    mail.send( 'RESULTS_NOTIFICATION' , mailDict )
            else:
                return


    def zipResults(self ):

        resultsFiles = []

        for File in self.results.values():
            resultsFiles += File  #File is tuple (string file name , int size , string format or None )
       
        xsl_path = os.path.join( _cfg.portal_path() , "xsl" ,)
        xslSize = os.path.getsize( xsl_path )
        resultsFiles.append( ( "index.xml" , xslSize , None ) )
        
        jobXslPath = os.path.join( xsl_path , "job.xsl" ) 
        jobXslSize = os.path.getsize( jobXslPath )
        resultsFiles.append( ( jobXslPath , jobXslSize , None ) )
                
        mobyleXslPath = os.path.join( xsl_path , "mobyle.xsl" )
        mobyleXslSize = os.path.getsize( mobyleXslPath )
        resultsFiles.append( ( mobyleXslPath , mobyleXslSize , None ) )
        
        paramsfiles = self.jobState.getParamfiles()
        if paramsfiles: 
            for paramsfile in paramsfiles:
	    	resultsFiles.append(  ( paramsfile[0] , paramsfile[1] , None ) )
        
        inputFiles = self.jobState.getInputFiles() #inputFiles = [ ( parameter , [ (fileName , format or None ) , ...) , ... ]
        if inputFiles is not None:
            for files in inputFiles:
                for item in files[1]: #item = ( filename , size , fmt ) 
                    resultsFiles.append( item )
            
	zipFileName = "%s_%s.zip" %(self.serviceName , self.jobKey )
        myZipFile = zipfile.ZipFile( zipFileName  , "w" )
        
	for fileName , size , fmt in resultsFiles :
            #for python up to 2.5 
            #If allowZip64 is True zipfile will create ZIP files that use the ZIP64 extensions when the zipfile is larger than 2 GB.
            #If it is false (the default) zipfile will raise an exception when the ZIP file would require ZIP64 extensions. 
            #ZIP64 extensions are disabled by default because the default zip and unzip v5.52 commands on Unix  don't support these extensions.
            
            
            if size > 0 and size < 2147483648 : #2Go 
                if fileName  == 'index.xml':
                    tree = self._indexPostProcess()
                    myZipInfo = zipfile.ZipInfo( 'index.xml', time.localtime()[:6] )
                    myZipInfo.external_attr = 2175008768   # set perms to 644
                    myZipFile.writestr(  myZipInfo  , tree )
                elif size < 10 :
                    myZipFile.write(   fileName   , os.path.basename( fileName ) , zipfile.ZIP_STORED )
                else:
                    myZipFile.write(   fileName   , os.path.basename( fileName ) , zipfile.ZIP_DEFLATED )
                    
            elif size >= 2147483648 :
                myZipInfo = zipfile.ZipInfo( os.path.basename( fileName ) , time.localtime()[:6] )
                myZipInfo.external_attr = 2175008768   # set perms to 644
                myZipFile.writestr(  myZipInfo  , "Sorry this file result is too large ( > 2GiB ) to be included in this archive." )    
            else:
                #the file is empty we don't add it to this archive
                pass
            
        myZipFile.close()
        return  zipFileName


    def _indexPostProcess( self ):
        fh_tree = open( 'index.xml' , 'r')
        tree = ''.join( fh_tree.readlines() )
        doc = Ft.Xml.Domlette.NoExtDtdReader.parseString( tree , uri = "file://%s" %os.path.abspath(fh_tree.name) )
        
        old_pi = doc.childNodes[0]
        new_pi = doc.createProcessingInstruction('xml-stylesheet' ,
                                                 'href="job.xsl" type="text/xsl"')
        doc.replaceChild( new_pi , old_pi)
        
        fh = StringIO.StringIO()
        Ft.Xml.Domlette.PrettyPrint( doc ,fh )
        tree = fh.getvalue()
        return tree 


class Script:
    """
    Choose de right class of Batch (in Config.Config.py) , instantiate it and do run()
    """

    def __init__(self,commandLine, dirPath, serviceName, jobState = None , xmlEnv = None ):
        """
        Choose de right class of Batch (in Config.Config.py) , instantiate it and do run()
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (normaly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param jobState:
        @type jobState: a L{JobState} instance
        @call: L{AsynchronJob.__init__}
        """
        # rc_log.debug( "serviceName = "+ serviceName)
        batch = _cfg.batch( serviceName )
        # rc_log.debug( "batch = "+batch)
        
        if batch == "SGE":
            self.batch = SGE( commandLine , dirPath , serviceName , jobState = jobState , xmlEnv = xmlEnv)
        elif batch == "PBS":
            # rc_log.debug("Considering a PBS submission for %s: %s" % (serviceName , commandLine) )
            self.batch = PBS( commandLine , dirPath , serviceName , jobState = jobState , xmlEnv = xmlEnv)
            # rc_log.debug("PBS submission batch instanced")
            
            # raise MobyleError,"try to instantiate PBS but PBS not yet implemented"
        elif batch == "LSF":
            raise MobyleError,"try to instantiate LSF but LSF not yet implemented"
        elif batch == "SYS":            
            self.batch = Sys( commandLine , dirPath , serviceName , jobState = jobState , xmlEnv = xmlEnv)
        else:
            msg = "unknown batch system : " + str(batch)
            rc_log.critical("%s : %s" %( serviceName ,
                                         msg
                                         )
            )

            if jobState is None:
                jobState = Mobyle.JobState.JobState( uri = dirPath )
            jobState.setStatus( Mobyle.Utils.Status( code = 5 , message = 'Mobyle internal server error' ) )
            jobState.commit()

            self._adm = Mobyle.Utils.Admin( dirPath )
            self._adm.setStatus( Mobyle.Utils.Status( code = 5 , message = msg ) )
            self._adm.commit()
            
            raise MobyleError, msg


class _Batch:
    """
    abstract class
    manage the status by updating the file index.xml
    """

    def __init__( self , commandLine , dirPath , serviceName , jobState = None , xmlEnv = None ):
        """
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (normaly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param jobState:
        @type jobState: a L{JobState} instance
        @call: L{Script.__init__}

        """

        self.commandLine = commandLine
        
        self._protectedCommandLine = ''
        for c in self.commandLine:
            self._protectedCommandLine += '\\'+ c
            
        self.dirPath = dirPath
        self.serviceName = serviceName

        if not jobState :
            self.jobState = Mobyle.JobState.JobState( self.dirPath )

        jobID = self.jobState.getID()
        status = Mobyle.Utils.Status( code = 3 )
        self.jobState.setStatus( status ) #running
        self.jobState.commit()

        self._adm = Mobyle.Utils.Admin( self.dirPath )
        self._adm.setStatus( status )
        self._adm.commit()
        
        if xmlEnv:
            self.xmlEnv = xmlEnv
        else:
            self.xmlEnv = {}
            
        # queue is a dictionary
        queue = _cfg.queue( self.serviceName )
        
        self.queue = Local.Policy.queue( queue["queue"] )

        new_path = ''
        binary_path = _cfg.binary_path()
        if binary_path :
                new_path = ":".join( binary_path )        

        if self.xmlEnv.has_key( 'PATH' ) :      
                new_path = "%s:%s" %( self.xmlEnv[ 'PATH' ] , new_path )

        if new_path :
            self.xmlEnv[ 'PATH' ] = "%s:%s" %( new_path , os.environ[ 'PATH' ] ) 
        else:
            self.xmlEnv[ 'PATH' ] = os.environ[ 'PATH' ] 
            
        for var in os.environ.keys():
            if var != 'PATH':
                self.xmlEnv[ var ] = os.environ[ var ]

        self._returncode = None
        accounting = _cfg.accounting()

        if accounting:
            beg_time = time.time()

        ###################################
        mobyleStatus = self.run()
        ###################################
        
        if accounting:
            end_time = time.time()
            elapsed_time = end_time - beg_time
            
            a_log = logging.getLogger( 'mobyle.account' )
            #%d trunc time to second
            #%f for millisecond
            a_log.info("%s/%s %d-%d %d %s" %( serviceName ,
                                           _cfg.batch( serviceName ) ,
                                           beg_time ,
                                           end_time ,
                                           elapsed_time ,
                                           self._returncode 
                                           )
                       )
        self.jobState.setStatus( mobyleStatus )

        self.jobState.commit()
        self._adm.setStatus( mobyleStatus  )
        self._adm.commit()


    def getKey( self ):
        """
        extract the Job key from the dirPath
        """
        if self.dirPath[-1] == '/':
            self.dirPath = self.dirPath[:-1]
        return os.path.split( self.dirPath )[1]
    
    def run( self ):
        """
        run the job
        abstract method. this method must be implemented in child classes
        """
        raise NotImplementedError, "Must be Implemented in child classes"

    @staticmethod
    def getStatus( number ):
        """
        @param number:
        @type number:
        @return the status of the job
        @rtype:
        abstract method. this method must be implemented in child classes
        """
        raise NotImplementedError, "Must be Implemented in child classes"

    @staticmethod
    def kill( self , number ):
        """
        kill the Job
        @param number:
        @type number:
        abstract method. this method must be implemented in child classes
        """
        raise NotImplementedError, "Must be Implemented in child classes"
            



    def _logError( self , admMsg = None , userMsg = None , logMsg = None ):

        if  admMsg :
            self._adm.setStatus( Mobyle.Utils.Status( code = 5 , message = admMsg ) )
            self._adm.commit()

        if userMsg :
            self.jobState.setStatus( Mobyle.Utils.Status( code = 5 , message = userMsg ) )
            self.jobState.commit()

        if logMsg :
            rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                          self.getKey() ,
                                          logMsg
                                          )
                        )
          
          
          
          
          ###########################################################
          #                                                         #
          #       to run with queuing system Sun Grid Engine        #
          #                                                         #
          ###########################################################
                
class SGE (_Batch):
    """
    
    """
        
    @staticmethod
    def _sgeInit( ):

        SGE_ROOT , SGE_CELL = _cfg.sge()
                    
        arch_path= os.path.join( SGE_ROOT , 'util' , 'arch' )

        try:
            arch_pipe = Popen(arch_path         ,
                              shell     = False ,
                              stdout    = PIPE  ,
                              stdin     = None  ,
                              stderr    = None  ,
                              close_fds = True  )
            arch_pipe.wait() 
            arch_rc = arch_pipe.returncode
            
        except OSError , err:
            #this error is log by calling method because I can't access to jobKey , adm status ... from static method
            msg = "SGE: I can't determined the system arch:"+str(err)
            raise MobyleError , msg
     
        if arch_rc != 0 :
            msg = "I can't determined the system arch (return code = " + str( rc ) + " )"
            raise MobyleError , msg
        
        arch = ''.join( arch_pipe.stdout.readlines() ).strip()

        sge_prefix = os.path.join( SGE_ROOT , 'bin' , arch )
        sge_env = {'SGE_CELL': SGE_CELL , 'SGE_ROOT': SGE_ROOT }   
        return ( sge_prefix , sge_env )


    def run(self):
        """
        """
        if os.getcwd() != os.path.abspath( self.dirPath ):
            msg = "the child process execute itself in a wrong directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg 
        else:
            jobKey = self.getKey()
            options = { '-cwd': ''           ,  #set the working dir to current dir
                        '-now': 'n'          ,  #the job is not executed immediately
                        '-N' : jobKey        ,  #the id of the job
                        '-q' : self.queue    ,  #the queue 
                        '-V' : ''               #job inherits of the whole environment
                        }
            sge_opts = ''

            for opt in options.keys():
                sge_opts += opt + ' ' + options[opt]+' '
            
            try:
                sge_prefix , sge_env = SGE._sgeInit()
            except MobyleError , err:
                self._logError( admMsg = str( err ) ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( str( err ) ) 
                    
                raise MobyleError , err
                      
            sge_cmd = os.path.join( sge_prefix , 'qrsh' ) + ' ' + sge_opts
            cmd = sge_cmd + " " + self._protectedCommandLine

            try:
                fout = open( self.serviceName + ".out" , 'w' )
                ferr = open( self.serviceName + ".err" , 'w' )
            except IOError , err:
                msg= "SGE: can't open file for standard job output: "+ str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = msg )

                raise MobyleError , msg

            self.xmlEnv.update( sge_env )
            
            try:
                pipe = Popen( cmd ,
                              shell  = True ,
                              stdout = fout  ,
                              stdin  = None  ,
                              stderr = ferr  ,
                              close_fds = True ,
                              env       = self.xmlEnv
                              )

            except OSError, err:
                msg= "SGE execution failed: "+ str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = msg )

                raise MobyleError , msg

            self._adm.setBatch( 'SGE')
            self._adm.setNumber( jobKey )
            self._adm.commit()
            
            linkName = ( "%s/%s.%s" %( _cfg.admindir() ,
                                       self.serviceName ,
                                       jobKey
                                    )
                         )
            try:
                os.link(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                msg = "can't create link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )

                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey  ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg
            pipe.wait()

            try:
                os.unlink( linkName )
            except OSError , err:
                msg = "can't remove link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )

                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            fout.close()
            ferr.close()

            self._returncode = pipe.returncode
            
            if pipe.returncode == 0 :# finished
                status = Mobyle.Utils.Status( code = 4 ) #finished
            elif pipe.returncode == 137 or pipe.returncode == 153: #killed
                ## 137 =  9 ( SIGKILL ) + 128 ( python add 128 )
                ## 153 = 25 ( SIGXFSZ ) + 128
                ## be careful if a job exit with a 137 or 153 exit status it will be labelled as killed
                status = Mobyle.Utils.Status( code = 6 , message = "Your job has been cancelled" ) 
            else:
                status = Mobyle.Utils.Status( code = 4 , message = "Your job finished with an unusual status code ( %s ), check your results carefully." % pipe.returncode )
            return status


    @staticmethod
    def getStatus( jobNum ):
        """
        @param jobNum:
        @type jobNum:
        @return: the status of job with number jobNum
        @rtype: string
        @todo: for best performance, restrict the sge querying to one queue
        """
        sge_prefix , sge_env = SGE._sgeInit( )
        sge_cmd = [ os.path.join( sge_prefix , 'qstat' ) , '-r' ]
        
        try:
            pipe = Popen( sge_cmd,
                          executable = sge_cmd[0] ,
                          shell = False ,
                          stdout = PIPE ,
                          stdin = None  ,
                          stderr= None  ,
                          close_fds = True ,
                          env= sge_env
                          )
        except OSError , err:
            raise MobyleError , "can't querying sge : %s :%s "%( sge_cmd , err )

        sge2mobyleStatus = { 'r' : 3 , # running
                             't' : 3 ,
                             'R' : 3 ,
                             's' : 7 , #hold
                             'S' : 7 , 
                             'T' : 7 ,
                             'h' : 7 ,
                             'w' : 2 , #pending
                             'd' : 6 , #killed 
                             'E' : 5 , #error
                             }

        for job in pipe.stdout :
            
            jobState = job.split()
            try:
               number = int( jobState[ 0 ] )
               status = jobState[ 4 ][ -1 ]
               continue
            except ( ValueError , IndexError ):
               pass #it's not the fisrt line

            try:
               if not ( jobNum == jobState[2] and 'Full' == jobState[0] ):
	               continue #it's not the right jobNum
            except IndexError :
               continue #it's not the 2nde line

            pipe.stdout.close()
            pipe.wait()
            
            if pipe.returncode == 0 :
                try:
                    return Mobyle.Utils.Status( code = sge2mobyleStatus[ status ]  )
                except MobyleError , err :
                    raise MobyleError , "unexpected sge status: " +str( status )
            else:
                raise MobyleError , "cannot get status " + str( pipe.returncode )
        return None

        
    @staticmethod
    def kill( job_number ):
        """
        @todo: kill the Job
        """
        sge_prefix , sge_env = SGE._sgeInit( )
        sge_cmd = "%s %s 1>&2" %( os.path.join( sge_prefix , 'qdel' ) ,  job_number )
        os.environ.update( sge_env )
        os.system( sge_cmd )
        

class PBS(_Batch):
    """
    @author: Pierre Tuffery
    PBS class to work for execution or route queues.
    """

    @staticmethod
    def _pbsInit( ):
        """
        To get some PBS local information
        """

        # rc_log.debug("PBS _pbsInit\n")
        (PBS_ROOT, PBS_WAIT_TIME, PBS_PRIORITY) =  _cfg.pbs() # return (PBS_ROOT, PBS_ENV)
        # ( PBS_ROOT, PBS_PATH ) =  _cfg.pbs() # return (PBS_ROOT, PBS_ENV)
        pbs_prefix =  PBS_ROOT 
        # rc_log.debug("PBS_prefix %s\n" % pbs_prefix)
        # pbs_env = {'PBS_ROOT': PBS_ROOT, 'PATH': PBS_PATH}
        pbs_env = {'PBS_ROOT': PBS_ROOT}
        pbs_defaults = {'PBS_WAIT_TIME' : PBS_WAIT_TIME, 'PBS_PRIORITY': PBS_PRIORITY}
        # rc_log.debug("PBS_env %s\n" % pbs_env)
        return ( pbs_prefix , pbs_env, pbs_defaults )
    
##         PBS_QUEUES = _cfg.pbs()
                    
##         arch_path= os.path.join(SGE_ROOT, 'util' , 'arch')

##         try:
##             arch_pipe = Popen(arch_path         ,
##                               shell     = False ,
##                               stdout    = PIPE  ,
##                               stdin     = None  ,
##                               stderr    = None  ,
##                               close_fds = True  )

##             arch_pipe.wait() 
##             arch_rc = arch_pipe.returncode
            

##         except OSError , err:
##             #this error is log by calling method because I can't access to jobKey , adm status ... from static method
##             msg = "SGE: I can't determined the system arch:"+str(err)
##             raise MobyleError , msg
     
##         if arch_rc != 0 :
##             msg = "I can't determined the system arch (return code = " + str( rc ) + " )"
##             raise MobyleError , msg
        
##         arch = ''.join( arch_pipe.stdout.readlines() ).strip()

##         sge_prefix = os.path.join( SGE_ROOT , 'bin' , arch )
##         sge_env = {'SGE_CELL': SGE_CELL , 'SGE_ROOT': SGE_ROOT }   
##         return ( pbs_prefix , pbs_env )

    def run(self):
        """
        We build the pbs script and we execute it.
        Unlike SGE, PDB is asynchronous, so we must manage the wait until
        job is finished.
        """
        # rc_log.debug("entering PBS.run()")


        if os.getcwd() != os.path.abspath( self.dirPath ):
            msg = "the child process execute itself in a wrong directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg 

        else:
            jobKey = self.getKey()

        # PBS_QUEUES = _cfg.pbsqueues()
        thisservicequeue = _cfg.queue( self.serviceName )
        # rc_log.debug("PBS service queue: %s\n" % thisservicequeue)

        self.queueproperties = _cfg.queueProperties(self.queue)
        # rc_log.debug("PBS queue (%s) properties: %s\n" % (self.queue, self.queueproperties) )
        if self.queueproperties == None:
            msg = "undefined pbs queue invoked %s" % self.queue

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg

        # PBS
        # rc_log.debug("PBS queueproperties get\n")
        options = { '-d': self.dirPath            ,  #set the working dir to current dir
                    '-m': 'n'            ,  # no email
                    '-N' : jobKey        ,  #the id of the job
                    '-q' : self.queue    ,  #the queue 
                    '-p' : '0'           ,  #the priority
                    '-V' : ''            ,  #job inherits of the whole environment
                    '-u' : self.queueproperties['uid'] , # owner of the job
                    '-S' : '/bin/bash'   
                    }

        pbs_opts = ''

        for opt in options.keys():
            pbs_opts += opt + ' ' + options[opt]+' '

        # rc_log.debug("PBS opts: %s\n" % pbs_opts)

        try:
            pbs_prefix, pbs_env, pbs_defaults = PBS._pbsInit()
        except MobyleError , err:
            self._logError( admMsg = str( err ) ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = None )

            rc_log.critical( str( err ) ) 

            raise MobyleError , err

        # rc_log.debug("PBS prefix: %s\n" % pbs_prefix)

        # We must write a script for remote queuing
        fName = "%s/batch.sh" % os.getcwd()
        # rc_log.debug("PBS script file: %s\n" % fName)
        f = open(fName,"w")
        f.write("#!/bin/bash\n")
        f.write("#PBS -S %s\n" % self.queueproperties['shell'])
        f.write("#PBS -q %s\n" % self.queue)
#        f.write("#PBS -o %s/%s.out\n" % (os.getcwd(), jobKey))
#        f.write("#PBS -e %s/%s.err\n" % (os.getcwd(), jobKey))
        f.write("#PBS -o %s/%s.out\n" % (os.getcwd(), self.serviceName))
        f.write("#PBS -e %s/%s.err\n" % (os.getcwd(), self.serviceName))
        f.write("#PBS -m n\n")
        f.write("#PBS -p %s\n" % self.queueproperties['priority'])
        f.write("#PBS -u %s\n" % self.queueproperties['uid'])
        f.write("#PBS -N %s\n" % self.serviceName)
	try:
		f.write("#PBS -l nodes=%s:ppn=%s\n" % (thisservicequeue["nodes"],thisservicequeue["ppn"] ))
	except:
		rc_log.debug("PBS exception: no multi_proc instructions for service %s\n" % self.serviceName)	
		pass
        f.write("cd %s\n" % os.getcwd())
        f.write("%s\n" % self.commandLine)
        f.close()

        cmd = os.path.join( pbs_prefix , 'qsub' ) + ' ' + fName
        # rc_log.debug("PBS system cmd: %s\n" % cmd)

        try:
            fout = open( self.serviceName + ".out" , 'w' )
            ferr = open( self.serviceName + ".err" , 'w' )
        except IOError , err:
            msg= "PBS: can't open file for standard job output: "+ str(err)

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        # rc_log.debug("PBS env: %s\n" % pbs_env)

        self.xmlEnv.update( pbs_env )

        # rc_log.debug("PBS env: %s\n" % self.xmlEnv)

        try:
            pipe = Popen( cmd ,
                          shell  = True ,
                          stdout = PIPE  ,
                          stdin  = None  ,
                          stderr = PIPE  ,
                          close_fds = True ,
                          env       = self.xmlEnv
                          )
        except OSError, err:
            msg= "PBS execution failed: "+ str(err)

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        # We get the pbsId
        pbsId = pipe.stdout.read()
        err   = pipe.stderr.read()
        if err != '':
            msg= "PBS execution returned an error: "+ str(err)

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        # make sure the Id contains the information of the host
        pbsId = pbsId.split()[0]
        if self.queueproperties['type'] == "execution":
            pbsKey = pbsId + "@localhost"
        elif self.queueproperties['type'] == "routage":
            pbsKey = pbsId + "@"+ self.queueproperties['remote']
        self.pbsKey = pbsKey

        # rc_log.debug("PBS key: %s\n" % pbsKey)

        pipe.wait()
        rc = pipe.returncode
        if rc != 0 :
            raise MobyleError , "error in pbs submission status"

        self._adm.setBatch( 'PBS')
        self._adm.setNumber( pbsKey )
        self._adm.commit()

        # rc_log.debug("PBS job id: %s\n" % pbsKey)

        linkName = ( "%s/%s.%s" %( _cfg.admindir() ,
                                   self.serviceName ,
                                   jobKey
                                )
                     )

        try:
            os.link(
                os.path.join( self.dirPath , '.admin') ,
                linkName
                )
        except OSError , err:
            msg = "can't create link %s in ADMINDIR: %s" %( linkName , err )

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = None )

            rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                               jobKey  ,
                                               msg
                                               )
                             )

            raise MobyleError , msg

        # Unlike SGE, PBS is asynchronous. We already have the hand.
        # We must perform active wait.
        # We wait for the job to complete
        # rc_log.debug("PBS wait loop (wait : %s %s)\n" % (thisservicequeue["wait_time"], pbs_defaults["PBS_WAIT_TIME"]))
        try:
            wait_time = thisservicequeue["wait_time"]
        except:
            try:
                wait_time = pbs_defaults["PBS_WAIT_TIME"]
            except:
                wait_time = 5 # should never get there, but stupid in case.

        # rc_log.debug("PBS wait_time %d\n" % wait_time)
        self._pbswait(wait_time = wait_time)
        # rc_log.debug("PBS job completed\n")
        
        # To debug
        mobyleStatus = 4
        statusMsg = None

        try:
            os.unlink( linkName )
        except OSError , err:
            msg = "can't remove link %s in ADMINDIR: %s" %( linkName , err )

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = None )

            rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                               jobKey ,
                                               msg
                                               )
                             )

            raise MobyleError , msg

        fout.close()
        ferr.close()

        # return ( mobyleStatus , statusMsg )

        ####################################
        #
        # verifier comment se comport pbs vis vis des code de retours !!!
        #
        #####################################


        if pipe.returncode == 0 :
            status = Mobyle.Utils.Status( code = 4 ) #finished
        elif pipe.returncode == 137: ## be careful if a job exit with a 137 exit status it will be labelled as killed 
            status = Mobyle.Utils.Status( code = 6 , message = "Your job has been cancelled" )  # killed
        else:
            status = Mobyle.Utils.Status( code= 4 ,
                                          message = "Your job finished with an unusual status code ( %s ), check your results carefully." % pipe.returncode )

        return status

    def _pbswait(self, wait_time = 5):
        """
        This method will wait for the submitted job (asynchronous) to be completed
        We can parameterize the wait_time between the refreshment of the
        information (60 seconds)
        """
        counter = 0
        # rc_log.debug("_pbswait %d\n" % wait_time)
        while counter < 2:
            while self._isCompleted() == 0:
##             fd = open("/tmp/pbs.dbg","w")
##             fd.write("Try %d\n" % counter)
##             fd.close()
                counter = 0
                time.sleep(wait_time)
            counter += 1
            time.sleep(2) # extra delay for NFS, sync, etc
##         fd = open("/tmp/pbs.dbg","a")
##         fd.write("Done (2) ...\n")
##         fd.close()
        return

    def _isCompleted(self):
        """
        This queries the queuing system using qstat to determine if
        the job is completed
        """
        import os, string

        # rc_log.debug("_isCompleted ?\n")
        try:
            status = self.getStatus(self.pbsKey)
            # rc_log.debug("_isCompleted wait status : %s %s msg\n" % (status, msg) )
            if status.isEnded():
                # rc_log.debug("_isCompleted wait status : isEnded %d\n" % (status.isEnded() ) )
                return 1
            return 0
        except:
            # rc_log.debug("_isCompleted !\n")
            return 1

    @staticmethod
    def getStatus( pbsKey ):
        """
        @param pbsKey:
        @type pbsKey:
        @return: the status of job with number pbsKey, a message relative to its status (useful for pending jobs)
        @rtype: string
        @todo: for best performance, restrict the sge querying to one queue
        PBS qstat return if completed:
        qstat: Unknown Job Id 19347.bioserv.rpbs.jussieu.fr
        """

        # rc_log.debug("getStatus from %s\n" % sys.argv[0])
        pbs_prefix , pbs_env, pbs_defaults = PBS._pbsInit( )

        remote = ""
        if pbsKey.count("@"):
            if not pbsKey.count("localhost"):
                remote = pbsKey.split("@")[1]
        jobId = pbsKey.split("@")[0]

        # We have 2 choices:
        # qstat -f (full description of properties)
        # qstat (short description)
        # here, we use short desc.

        # qstat will return info on the form:
        # 32796.bioserv       STDIN            aw3                    0 R Xeon
        # Hence, we truncate jobId if necessary
        if pbsKey.count(".") > 1:
            theIndex = pbsKey.index(".",pbsKey.index(".")+1)
            jobId = pbsKey[:theIndex]
        jobHandle = jobId
        qstatMachine = ""
        if remote != "":
            jobHandle = jobId + "@" + remote
            qstatMachine = "@" + remote

        # 2008, P. Tuffery:
        # safer not to specify jobId in qstat
        # better to get full info and search for the jobId
        pbs_cmd = os.path.join( pbs_prefix , 'qstat' )
        # pbs_cmd = pbs_cmd + " " + jobHandle
        pbs_cmd = pbs_cmd + " " + qstatMachine
        # rc_log.debug("PBS qstat_cmd %s\n" % pbs_cmd)
        # rc_log.debug("PBS for job %s\n" % jobId)

        try:
            pipe = Popen( pbs_cmd,
                          shell = True ,
                          stdout = PIPE ,
                          stdin = None  ,
                          stderr= PIPE ,
                          close_fds = True ,
                          env= pbs_env
                          )
            pipe.wait()
            rc = pipe.returncode

        except OSError , err:
            raise MobyleError , "can't query pbs : %s :%s "%( pbs_cmd , err )
        if rc != 0 :
            raise MobyleError , "error in pbs status querying"

        pbs2mobyleStatus = { 'r' : 3 , #running
                             't' : 3 ,
                             'R' : 3 ,
                             's' : 7 , #hold
                             'S' : 7 ,
                             'T' : 7 ,
                             'h' : 7 ,
                             'w' : 2 , #pending
                             'd' : 6 , #killed
                             'E' : 3 , #'exiting',
                             'C' : 3 , #'exiting',
                             'Q' : 2 , #pending
                             'F' : 4 , #finished # Not used
                             }

        err = pipe.stderr.read()
        pipe.stderr.close()
        if err.count("Unknown"):
            # return pbs2mobyleStatus[ 'F' ]
            raise MobyleError , "no job (1) : " + str( pbsKey )

        pbsOutLines = pipe.stdout.readlines()
        # To debug PBS
        # rc_log.debug("PBS qstat_cmd got: %s\n" % pbsOutLines)

        Qcount = None
	Qmsg   = None
        for line in pbsOutLines :
	    if Qcount:
                Qcount += 1
            else:
                lState = line.split()
                try:
                    lstatus = lState[4]		
                    if lstatus == "Q":
                        Qcount = 1
                except:
                    continue
            
	    lState = line.split()
	    try:
		lstatus = lState[4]		
	    except:
	        continue
            if line.count(jobId):
                jobState = line.split()
                try:
                   status = jobState[ 4 ]
                   pipe.stdout.close()
                   # rc_log.debug("PBS qstat_cmd got status: %s\n" % status)
                   mobyleStatusCode = pbs2mobyleStatus[ status ]
                   if Qcount and (status == "Q"):
		      Qmsg = "Pending %d" % Qcount
                   #     returnStr = "%s_%d" % ( returnStr, Qcount )
                   # rc_log.debug("PBS get_status got : %s %s\n" % (status, Qmsg))
                   return Mobyle.Utils.Status( code = mobyleStatusCode , message = Qmsg )
                   break
                except ( ValueError , IndexError ):
                   pass #it's not the first line
        # rc_log.debug("PBS qstat_cmd got no job. Msg: %s\n" % Qmsg)

        # return 'finished'
        # time.sleep(3)
        rs = Mobyle.Utils.Status( code = 4 , message = Qmsg )

        # rc_log.debug("Will exit getStatus\n")
        return rs
        # raise MobyleError , "no job (2) : " + str( pbsKey )

    @staticmethod
    def kill( pbsKey ):
        """
        @todo: kill the Job
        """

        # rc_log.debug("PBS kill %s\n" % pbsKey)

        pbs_prefix , pbs_env, pbs_defaults = PBS._pbsInit( )

        remote = ""
        if pbsKey.count("@"):
            if not pbsKey.count("localhost"):
                remote = pbsKey.split("@")[1]
        jobId = pbsKey.split("@")[0]
        if pbsKey.count(".") > 1:
            theIndex = pbsKey.index(".",pbsKey.index(".")+1)
            jobId = pbsKey[:theIndex]
        jobHandle = jobId
        if remote != "":
            jobHandle = jobId + "@" + remote

        pbs_cmd = "%s %s 1>&2" %( os.path.join( pbs_prefix , 'qdel' ) ,  jobHandle )
        os.environ.update( pbs_env )
        os.system( pbs_cmd )


          ###############################################
          #                                             #
          #       to run without queuing system         #
          #                                             #
          ###############################################


class Sys(_Batch):
    """
    Run the commandline by a system call
    """

    def run(self):
        """
        Run the commandLine in a subshell (system call) and redirect the standard error and output on service.name.out and service.name.err, then restore the sys.stderr and sys.stdout
        @return: the L{Mobyle.Utils.Status} of this job and a message
        @rtype: ( int , string )
        @call: in _Batch.__init__
        """

        if (os.getcwd() != os.path.abspath(self.dirPath) ):
            msg = "the process is not in the working directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        else:
            jobKey = self.getKey()
            fout = open( self.serviceName + ".out" , 'w' )
            ferr = open( self.serviceName + ".err" , 'w' )

            try:
                #the new process launch by popen must be a session leader 
                # because it launch in background and if we send a siggnal to the shell 
                # this signal will no propagate to all child process
                # and if we kill the process leader we kill the python 
                # thus we execute our setsid which transform the new process as a session leader
                # and execute the command prior to return.
                # in these conditions we could kill the process leader and the signal is propagate to
                # all processes belonging to this session. 

                setsidPath = '%s/Tools/setsid' %_cfg.mobylehome()
                command = [ setsidPath , setsidPath , '/bin/sh' , '-c' , self.commandLine ]
                # rc_log.debug("Sys env : %s\n" % str(self.xmlEnv) )

                pipe = Popen( command ,
                              shell     = False ,
                              stdout    = fout ,
                              stdin     = None ,
                              stderr    = ferr ,
                              close_fds = True ,
                              env       = self.xmlEnv
                              )

            except OSError, err:
                msg= "System execution failed: " + str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg 
                
            jobPid = pipe.pid
            
            self._adm.setBatch( 'Sys' )
            self._adm.setNumber( jobPid )
            self._adm.commit()

            linkName = ( "%s/%s.%s" %( _cfg.admindir() ,
                                       self.serviceName ,
                                       jobKey
                                    )
                         )
            
            try:
                os.link(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                msg = "can't create link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            #pipe.wait() doesn't work with signal.signal SIGUSR1 & SIGUSR2 (until python 2.5)
            #OSError: [Errno 4] Interrupted system call
            #RESOLVED with used siginterrupt.siginterrupt( signal , False )

            pipe.wait()

            try:
                os.unlink( linkName )
            except OSError , err:
                msg = "can't remove link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                rc_log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            fout.close()
            ferr.close()
            self._returncode = pipe.returncode
            
            if pipe.returncode == 0 :# finished
                status = Mobyle.Utils.Status( code = 4 ) #finished
            elif pipe.returncode == -15 or pipe.returncode == -9: #killed
                status = Mobyle.Utils.Status( code = 6 , message = "Your job has been cancelled" ) 
            else:
                status = Mobyle.Utils.Status( code= 4 , message = "Your job finished with an unusual status code ( %s ), check your results carefully." % pipe.returncode )

            return status
        

    @staticmethod
    def getStatus( pid ):
        """
        @param pid:
        @type pid:
        @return: the status of job with pid pid
        @rtype: string
        """
        pid = int( pid )
        try:
            message = None
            os.kill( pid , signal.SIG_DFL )
        except OSError , err:
            if str( err ).find( 'No such process' ) != -1:
                return None 
            else:
                raise MobyleError , err 

        return Mobyle.Utils.Status( code = 3 , message = message ) #running

    @staticmethod
    def kill( job_pid ):
        """
        kill a job
        @param job_pid: the pid of the job
        @type job_pid: int
        @raise MobyleError: if can't kill the job
        @call: by L{Utils.Mkill}
        """
        try:
            job_pid = int( job_pid )
            job_pgid = os.getpgid( job_pid )
        except ValueError:
            raise MobyleError , "can't kill job  - %s" % err 
        try:
            os.killpg( job_pgid , signal.SIGTERM )
        except OSError , err:
            raise MobyleError , "can't kill job  - %s" % err 
        
               
        try:
            time.sleep(0.2)
            os.killpg( job_pgid , signal.SIG_DFL )
            try:
                os.killpg( job_pgid , signal.SIGKILL )
            except OSError , err:
                raise MobyleError , "can't kill job - %s" % err
                
        except OSError , err:
            return
        
class WSWrapper(_Batch):
    """
    """
    def run(self):
        raise NotImplementedError ,"ToDo"

class CGIWrapper(_Batch):
    """
    """
    def run(self):
        raise NotImplementedError ,"ToDo"

if __name__ == '__main__':
    try:
        fh = open(".forChild.dump", "r")
        fromFather = cPickle.load( fh )
        fh.close() 
    except IOError ,err:
        pass
    except :
        pass

    try:
        os.chdir( fromFather[ 'dirPath' ] )
    except OSError, err:
        msg = fromFather[ 'serviceName' ] + ":" + str( err )
        rc_log.critical( msg )
        raise MobyleError ,msg

    userEmail = fromFather[ 'email']
    if userEmail is not None:
        userEmail = Mobyle.Net.EmailAddress( userEmail  )
        
    child = AsynchronJob( fromFather[ 'commandLine' ] , # string the unix command line
                          fromFather[ 'dirPath' ] ,     # absolute path of the working directory
                          fromFather[ 'serviceName' ] , # string 
                          fromFather[ 'resultsMask'] ,  # 
                          userEmail = userEmail  ,      # Net.EmailAddress to
                          xmlEnv = fromFather[ 'xmlEnv' ] , #a dict
                          )
