########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


import os , os.path
import sys
import fcntl
import Ft.Xml , Ft.Xml.Domlette
from Ft.Xml import EMPTY_NAMESPACE
from time import strptime , strftime , time ,sleep
import logging


from Mobyle.Parser import ServiceParser
from  Mobyle.Classes.DataType import DataTypeFactory
from Mobyle.MobyleError import MobyleError , SessionError , ParserError
from Mobyle.Utils import Status

class Transaction( object ):
    """
    This class defines a session, that stores all the information
    about a user that should be persistent on the server
    @author: Bertrand Neron
    @organization: Institut Pasteur
    @contact:mobyle@pasteur.fr
    """
    
    __ref = {}
    
    WRITE  = fcntl.LOCK_EX
    READ   = fcntl.LOCK_SH
    
    def __new__( cls , fileName , lockType ):
        if not cls.__ref.has_key( fileName ) :
            self = super( Transaction , cls ).__new__( cls )
            self._log = logging.getLogger( 'mobyle.session' )
            fileName = os.path.normpath( fileName )
            try:
                if lockType == self.READ :
                    self.__File = open( fileName , 'r' )
                elif lockType == self.WRITE :
                    self.__File = open( fileName , 'r+' )
                else:
                    raise MobyleError , 'unvalid lockType : ' +  str( lockType )
            except IOError , err:
                msg = "can't open session %s : %s" %( os.path.basename( os.path.dirname( fileName ) ) , err )
                self._log.critical( msg )
                raise SessionError , msg
            
            self.__lockType = lockType
            
            self._lock() #try to acquire a lock 
            #If I do not got a lock a IOError is raised
            try:
                self._doc = Ft.Xml.Domlette.NoExtDtdReader.parseStream( self.__File , uri = "file://%s" % os.path.abspath( self.__File.name )) 
                self._root = self._doc.documentElement
            except Exception , err:
                import shutil
                shutil.copy2( self.__File.name , "%s.copy.%d" %( self.__File.name , int( time() ) ))
                msg = " an error occured during the session %s parsing : %s " %( os.path.basename( os.path.dirname( fileName ) ) , err )
                self._log.debug( msg  , exc_info = True )
                self._log.error( msg , exc_info = True )      
                raise SessionError , err
            
            self._modified = False
            cls.__ref[ self.__File.name ]= True
            return self
        else:
            raise SessionError , "try to open 2 transactions on the same file at the same time"
        
    
    
    @staticmethod    
    def create( fileName , authenticated , activated , activatingKey = None , userEmail = None , passwd = None):
        """
        create a minimal xml session 
        @param fileName: the absolute path of the xml session file
        @type fileName: string
        @param authenticated: True if the session is authenticated , false otherwise
        @type authenticated: boolean
        @param activated: True if the session is activated, False otherwise.
        @type activated: boolean
        @param activatingKey: the key send to the user to activate this session
        @type activatingKey: boolean
        @param userEmail: the userEmail of the user
        @type userEmail: string
        """
        id = os.path.basename( os.path.dirname( fileName ))
        if authenticated and not passwd:
            msg = "cannot create authenticated session %s for %s without passwd" %( id , userEmail )
            raise SessionError, msg
        
        if not authenticated and passwd:
            msg = "cannot create anonymous session %s for %s with passwd" %( id , userEmail )
            raise SessionError, msg
        
        if authenticated and not userEmail:
            msg = "cannot create authenticated session %s without userEmail" %( id )
            raise SessionError, msg
        
        doc = Ft.Xml.Domlette.implementation.createDocument( Ft.Xml.EMPTY_NAMESPACE , "userSpace" , None )
        root = doc.documentElement
        root.setAttributeNS( EMPTY_NAMESPACE , 'id' , id )
        
        authenticatedNode = doc.createElementNS( EMPTY_NAMESPACE , "authenticated" )
        if authenticated:
            text_node = doc.createTextNode( 'true' )
        else:
            text_node = doc.createTextNode( 'false' )
        authenticatedNode.appendChild( text_node )    
        root.appendChild( authenticatedNode )    
        
        activatedNode = doc.createElementNS( EMPTY_NAMESPACE , "activated" )
        if activated:
            text_node = doc.createTextNode( 'true' )
        else:
            text_node = doc.createTextNode( 'false' )
        activatedNode.appendChild( text_node )    
        root.appendChild( activatedNode ) 
         
        if userEmail:
            emailNode = doc.createElementNS( EMPTY_NAMESPACE , "email" )
            text_node = doc.createTextNode( str( userEmail ) )
            emailNode.appendChild( text_node )   
            root.appendChild( emailNode ) 
        
        if passwd:
            passwdNode = doc.createElementNS( EMPTY_NAMESPACE , "passwd" )
            text_node = doc.createTextNode( passwd )
            passwdNode.appendChild( text_node )   
            root.appendChild( passwdNode )             
        
        if activatingKey:
            activatingKeyNode = doc.createElementNS( EMPTY_NAMESPACE , "activatingKey" )
            text_node = doc.createTextNode( activatingKey )
            activatingKeyNode.appendChild( text_node )   
            root.appendChild( activatingKeyNode ) 
                   
        #xmlFileName = os.path.normpath(  os.path.join( self.Dir , Session.FILENAME  )  )
        xmlFile = open( fileName , 'w' )
        Ft.Xml.Domlette.PrettyPrint( doc , xmlFile )
        xmlFile.close()        


    def _lock(self):
        """
        try to acquire a lock of self.__lockType on self.__File
        @raise IOError: when it could not acquire a lock
        """
        IGotALock = False
        ID = os.path.basename( os.path.dirname( self.__File.name ))
        self._log.debug( "%f : %s : _lock Type= %s ( call by= %s )"  %( time() ,
                                                                        ID,
                                                                        ( 'UNKNOWN LOCK', 'READ' , 'WRITE' )[ self.__lockType ] ,
                                                                        os.path.basename( sys.argv[0] ) ,
                                                                        ))
        for attempt in range( 4 ):
            try:             
                fcntl.lockf( self.__File , self.__lockType | fcntl.LOCK_NB )
                IGotALock  = True
                self._log.debug( "%f : %s : _lock IGotALock = True" %(time() , ID ))
                break
            except IOError , err:
                self._log.debug( "%f : %s : _lock IGotALock = False" %(time() , ID))
                sleep( 0.2 )

        if not IGotALock :
            self._log.error( "%s : %s" %( ID , err ) )
            self._log.debug( "%f : %s : _lock Type= %s ( call by= %s )"  %( time() ,
                                                                            ID ,
                                                                            ( 'UNKNOWN LOCK', 'READ' , 'WRITE' )[ self.__lockType ] ,
                                                                            os.path.basename( sys.argv[0] ) 
                                                                            ))
            self.__File.close()
            self.__File = None
            self.__lockType = None
               
            raise IOError , err



    
    def _setModified(self , modified ):
        """
        to avoid that a 1rst method set modified to True and a 2 method reset it to False
        """
        if self._modified is False:
            self._modified = modified
        
    def _addTextNode( self, node , nodeName , content , attr = None):
        """ 
        add a text node named nodeName with content to the node node
        @param node: the node on which the new node will append 
        @type node: node element
        @param nodeName: the name of the new node
        @type nodeName: string
        @param content: the content of the new node
        @type content: string
        """

        newNode = self._doc.createElementNS( EMPTY_NAMESPACE , nodeName )
        if attr:
            for attrName in attr.keys():
                newNode.setAttributeNS( EMPTY_NAMESPACE , attrName , str( attr[ attrName ] ) )
        
        text_node = self._doc.createTextNode( str( content ) )
        newNode.appendChild( text_node )
        node.appendChild( newNode )

    
    
    def _updateNode( self, node , nodeName , content ):
        """
        replace the content of the text_node child of a node by content. if the node nodeName doesn't exist, it make it
        @param nodeName: the name of the Node to update
        @type nodeName: String
        @param content: the content of the text-node child of nodeName
        @type content: String
        """
        nodes = node.xpath( "./" + nodeName )
        text_node = self._doc.createTextNode( str( content ) )
        if nodes:
            if nodes[0].xpath( "text()")[0].data == content:
                return False
            else:
                nodes[0].replaceChild( text_node , nodes[0].firstChild )
                return True
        else:
            newNode = self._doc.createElementNS( EMPTY_NAMESPACE , nodeName )
            newNode.appendChild( text_node )
            node.appendChild( newNode )
            return True
    
    
    def commit(self):
        """
        if the transaction was open in WRITE
         - write the dom in xml file if it was modified 
        and release the lock and close the session File this transaction could not be used again
        """
        
        if self.__lockType == self.READ :
            logType = 'READ'
        elif self.__lockType == self.WRITE :
            logType = 'WRITE'
        else:
            logType = 'UNKNOWN LOCK( ' + str( self.__lockType ) +' )'

        self._log.debug( "%f : %s : _commit Type= %s ( call by= %s )"  %( time(),
                                                                         self.getID() ,
                                                                         logType ,
                                                                         os.path.basename( sys.argv[0] ) 
                                                                         ))
        try:
            if self.__lockType == self.WRITE and self._modified :
                try:
                    tmpFile = open( "%s.%d" %( self.__File.name , os.getpid() )  , 'w' )
                    Ft.Xml.Domlette.PrettyPrint( self._doc , tmpFile )
                    tmpFile.close()
                    os.rename( tmpFile.name , self.__File.name )
                except IOError , err :
                    msg = "can't commit this transaction: " + str( err )
                    self._log.error( msg )
                    raise SessionError , msg
                except Exception ,err:
                    self._log.error( "session/%s self.__File = %s" %( self.getID(), self.__File ) , 
                                 exc_info = True )
                    raise err
        finally:
            key2del = self.__File.name 
            try:
                self._log.debug("%f : %s : commit UNLOCK type= %s modified = %s" %(time() ,
                                                                                          self.getID() ,
                                                                                          logType, 
                                                                                          self._modified
                                                                                          ))   
                fcntl.lockf( self.__File , fcntl.LOCK_UN  )
                self.__File.close()
                self.__File = None
                self.__lockType = None
                self._modified = False #reset _modifiedTransaction to false for the next transaction
                del( Transaction.__ref[ key2del ] )
            except IOError , err :
                self._log.error( "session/%s : cannot UNLOCK  transaction type= %s modified= %s: " %(self.getID(),
                                                                                                            logType,
                                                                                                            self._modified,
                                                                                                            err ) ) 
                del( Transaction.__ref[ key2del ] )

                
    def rollback(self):
        """
        release the lock , close the file without modification of the xml file.
        this transaction could not be used again
        """
        if self.__lockType == self.READ :
            logType = 'READ'
        elif self.__lockType == self.WRITE :
            logType = 'WRITE'
        else:
            logType = 'UNKNOWN LOCK( ' + str( self.__lockType ) +' )'

        self._log.debug( "%f : %s : _close Type= %s ( call by= %s )"  %( time(),
                                                                         self.getID() ,
                                                                         logType ,
                                                                         os.path.basename( sys.argv[0] ) 
                                                                         ))
            
        try:
            key2del = self.__File.name 
            fcntl.lockf( self.__File , fcntl.LOCK_UN  )
            self.__File.close()
            self.__File = None
            self.__lockType = None
            self._modified = False
            self._modified = False
        except IOError , err :
            self._log.error( "session/%s : an error occured during closing transaction : %s" %( self.getID() , err )) 
            # a gerer   
        except Exception , err :
            msg = "session/%s : an error occured during closing transaction : %s" %( self.getID() , err )
            self._log.error( msg ) 
            self._log.debug( "%f : %s : self.__File = %s" %(time() , self.getID(), self.__File ))
            self._log.debug( "%f : %s : self.__lockType = %s" %(time() , self.getID(), logType ))
            raise SessionError , msg
        finally:
            del( Transaction.__ref[ key2del ] )
            
            
    def getID(self):
        """
        @return: the id of the session
        @rtype: string
        """
        try:
            return self._root.xpath( '@id')[0].value
        except indexError:
            msg = "the session %s has no identifier" % self.__File.name
            self._log.error( msg )
            raise SessionError , msg   
   
    def getEmail(self):
        """
        @return: the email of the user session
        @rtype: string 
        """
        try:
            return str( self._root.xpath( 'email/text()' )[0].data )
        except IndexError:
            return None
        
    def setEmail( self , email ):
        """
        set the user email of this session 
        @param email: the email of the user of this session
        @type email: string
        """
        modified = self._updateNode( self._root , 'email' ,  email )
        self._setModified( modified )
        
        
    def isActivated( self ):
        """
        @return: True if this session is activated, False otherwise.
        @rtype: boolean
        """
        try:
            activated = str( self._root.xpath( 'activated/text()' )[0].data )
            if activated == 'true':
                return True
            else:
                return False
        except IndexError:
            raise Exception
    
    def activate(self):
        """
        activate this session
        """
        modified = self._updateNode( self._root , 'activated' ,  'true' )
        self._setModified( modified )
        
    def getActivatingKey(self):
        """
        @return: the key needed to activate this session ( which is send by email to the user )
        @rtype: string
        """
        try:
            return str( self._root.xpath( 'activatingKey/text()' )[0].data )
        except IndexError:
            return None
    
    def isAuthenticated(self):
        """
        @return: True if this session is authenticated, False otherwise.
        @rtype: boolean
        """
        try:
            auth = str( self._root.xpath( 'authenticated/text()' )[0].data )
            if auth == 'true':
                return True
            else:
                return False
        except IndexError:
            return None
    
    
    def getPasswd(self):
        """
        @return: the password of the user ( encoded )
        @rtype: string
        """
        try:
            return str( self._root.xpath( 'passwd/text()' )[0].data )
        except IndexError:
            return None
    
    def setPasswd( self , passwd ):
        """
        set the session user pass word to passwd
        @param passwd: the encoded user password
        @type passwd: string
        """
        modified = self._updateNode( self._root , 'passwd' ,  passwd )
        self._setModified( modified )
        
    def getCaptchaSolution( self ):
        """
        @return: the solution of the captcha problem submitted to the user.
        @rtype: string
        """
        try:
            return str( self._root.xpath( 'captchaSolution/text()' )[0].data )
        except IndexError:
            return None
    
    def setCaptchaSolution( self  , solution ):
        """
        store the solution of the captcha problem submitted to the user.
        @param solution: the solution of the captcha
        @type solution: string
        """
        modified = self._updateNode( self._root , 'captchaSolution' ,  solution )
        self._setModified( modified )
    
    
    ################################
    #
    #  operations on Data
    #
    #################################
    def hasData(self , dataID ):
        """
        @param dataID: the identifier of the data in the session ( mdm5 name + ext )
        @type dataID: string
        @return: True if the session structure has an entry for the data corresponding to this ID, False otherwise.
        @rtype: boolean 
        """
        try:
            self._root.xpath( 'dataList/data[@id = "%s"]'% dataID )[0]
            return True
        except IndexError:
            return False 
            
    def renameData( self , dataID  , newUserName ):
        """
        change the user name of the data corresponding to the dataID.
        @param dataID: the identifier of the data in the session ( mdm5 name + ext )
        @type dataID: string
        @param newUserName: the new user name of the data.
        @type newUserName: string
        @raise ValueError: if dataID does not match any entry in session.
        """
        dataNode = self._getDataNode( dataID )
        modified = self._updateNode( dataNode , 'userName' ,  newUserName )
        self._setModified( modified )
    
    def getData(self , dataID ):
        """
        @param dataID: the identifier of the data in the session ( mdm5 name + ext )
        @type dataID: string
        @return: the data corresponding to the dataID
        @rtype: { 'dataName'   =  string ,
                  'userName'   =  string ,
                  'size'       =  int ,
                  'Type'       =  Mobyletype instance,
                  'dataBeg'    =  string ,
                  'inputModes' = [ string inputMode1 , string inputMode2 ,... ],
                  'format'     = (  string format , int count , string 'to fill the api (unused)',''),
                  'producedBy' = [ string jobID1 , string jobID2 , ... ],  
                  'usedBy'     = [ string jobID1 , string jobID2 , ... ]
                  }
        @raise ValueError: if dataID does not match any entry in session.
        """
        dataNode = self._getDataNode( dataID )
        dtf = DataTypeFactory()
        return  self._dataNode2dataDict( dataNode ,dtf )

             
    def getAllData(self):
        """
        @return: all data in this session
        @rtype: [ {'dataName'   =  string ,
                  'userName'   =  string ,
                  'size'       =  int ,
                  'Type'       =  Mobyletype instance,
                  'dataBeg'    =  string ,
                  'inputModes' = [ string inputMode1 , string inputMode2 ,... ],
                  'format'     = (  string format , int count , string 'to fill the api (unused)'),
                  'producedBy' = [ string jobID1 , string jobID2 , ... ],  
                  'usedBy'     = [ string jobID1 , string jobID2 , ... ]
                  } , ... ]
        """
        datas = []
        dtf = DataTypeFactory()
        for dataNode in self._root.xpath( 'dataList/data') :
            data = self._dataNode2dataDict( dataNode , dtf )
            datas.append( data )
        return datas


    def removeData(self , dataID ):
        """
        remove the data entry corresponding to dataID
        @param dataID: the identifier of the data in the session ( mdm5 name + ext )
        @type dataID: string
        @raise ValueError: if dataID does not match any entry in session.
        """
        dataNode = self._getDataNode( dataID )
        dataListNode = dataNode.xpath( '..' )[0]
        dataListNode.removeChild( dataNode )
        ## remove the ref of this data from the jobs
        usedBy = [ str( node.value ) for node in dataNode.xpath('usedBy/@ref' )]
        for jobID in usedBy:
            try:
                jobNode = self._getJobNode( jobID )
            except ValueError:
                continue
            else:
                try:
                    dataUsedNode = jobNode.xpath( 'dataUsed[@ref="%s"]' %dataID )[0]
                except IndexError:
                    continue
                else:
                    jobNode.removeChild( dataUsedNode )
            
        producedBy = [ str( node.value ) for node in dataNode.xpath('producedBy/@ref' )]
        for jobID in producedBy:    
            try:
                jobNode = self._getJobNode( jobID )
            except ValueError:
                continue
            else:
                try:
                    dataProducedNode = jobNode.xpath( 'dataProduced[@ref="%s"]' %dataID )[0]
                except IndexError:
                    continue
                else:
                    jobNode.removeChild( dataProducedNode )            
        self._setModified( True ) 
    
    
    def linkJobInput2Data(self , datasID ,  jobsID ):
        """
        add a ref of each job using this data and a ref of this data in the corresponding job
        @param datasID: the IDs of data which are used by these jobsID
        @type datasID: list of string
        @param jobsID: the IDs of job which has used these data
        @type jobsID: list of string
        @raise ValueError: if one dataID or jobID does not match any entry in session.
        """
        for dataID in datasID:
            dataNode = self._getDataNode( dataID )
            for jobID in jobsID :
                alreadyUsedBy = [ str( node.value ) for node in dataNode.xpath('usedBy/@ref' )]
                if  jobID not in alreadyUsedBy :
                    newUsedByNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'usedBy' )
                    newUsedByNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , jobID )
                    dataNode.appendChild( newUsedByNode )
                    self._setModified( True )
                
                try:
                    jobNode = self._getJobNode( jobID )
                except ValueError:
                    continue
                dataUsed = [ str( node.value ) for node in jobNode.xpath( 'dataUsed/@ref' ) ]
                if dataID not in dataUsed:
                    newDataUsedNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'dataUsed' )
                    newDataUsedNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , dataID )
                    jobNode.appendChild( newDataUsedNode )
                    self._setModified( True )
    
    
    
    def linkJobOutput2Data(self , datasID , jobsID ):
        """
        add a ref of each job producing these data and a ref of these data in corresponding jobs.
        @param dataID: the list of the data ID
        @type dataID: list of string
        @param jobsID: the IDs of job which has produced this data
        @type jobsID: list of string
        @raise ValueError: if one dataID or jobID does not match any entry in session.
        """
        for dataID in datasID:
            dataNode = self._getDataNode( dataID )
            for jobID in jobsID:
                alreadyDataProduced = [ str( node.value ) for node in dataNode.xpath('producedBy/@ref' )]
                if jobID not in alreadyDataProduced :
                    newUsedByNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'producedBy' )
                    newUsedByNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , jobID )
                    dataNode.appendChild( newUsedByNode )
                    self._setModified( True )

                try:
                    jobNode = self._getJobNode( jobID )
                except ValueError:
                    continue
                dataProduced = [ str( node.value ) for node in jobNode.xpath( 'dataProduced/@ref' ) ]
                if dataID not in dataProduced:
                    newDataProducedNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'dataProduced' )
                    newDataProducedNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , dataID )
                    jobNode.appendChild( newDataProducedNode )
                    self._setModified( True )

                
    def addInputModes(self , dataID , inputModes ):
        """
        add an inputmode in the list of inputModes of the data corresponding to the dataID
        @param dataID: The identifier of one data in this session
        @type dataID: string
        @param inputModes: the list of inputMode to add the accepted value are 'db' , 'paste' , 'upload' , 'result'
        @type inputModes: list of string
        @raise ValueError: if dataID does not match any entry in session.
        """
        dataNode = self._getDataNode( dataID )
        try:
            modesNode = dataNode.xpath( 'inputModes')[0]
            oldModes = [ modeNode.data for modeNode in modesNode.xpath( 'inputMode/text()') ]
        except IndexError:
            modesNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'inputModes' )
            oldModes = []
        for newMode in inputModes:
            if newMode in oldModes:
                continue
            else:
                newInputNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'inputMode' )
                textNode = self._doc.createTextNode( newMode )
                newInputNode.appendChild( textNode )
                modesNode.appendChild( newInputNode )
                self._setModified( True )
     
     
    def createData( self , dataID , userName , size , Type , dataBegining , inputModes , format = None , producedBy = [] , usedBy = []):
        """
        add a new data entry in the session.
        @param dataID: The identifier of one data in this session
        @type dataID: string
        @param userName: the name given by the user to this data
        @type userName: string
        @param size: the size of the file corresponding to this data
        @type size: int
        @param Type: the MobyleType of this data
        @type Type: L{MobyleType} instance
        @param dataBegining: the 50 first char of the data (if data inherits from Binary a string)
        @type dataBegining: string
        @param inputModes: the input modes of this data the accepted data are: 'paste' , 'db' , 'upload' , 'result'
        @type inputModes: list of string
        @param format: the format , count , of this data
        @type format: ( string format , int count , string 'to fill the api (unused)' )
        @param producedBy: the jobs Id which produced this data
        @type producedBy: list of string
        @param usedBy: the jobs ID which used this data
        @type usedBy: list of string
        """
        try:
            dataListNode = self._root.xpath( 'dataList')[0]
        except IndexError:
            dataListNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'dataList')
            self._root.appendChild( dataListNode )
            
        dataNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'data' )
        dataNode.setAttributeNS( EMPTY_NAMESPACE , 'id' , dataID )
        dataNode.setAttributeNS( EMPTY_NAMESPACE , 'size' , str( size ) )
        
        if format:
            format , count , progConverter =  format
        else:
            format = None
            count = None
        if count:
            dataNode.setAttributeNS( EMPTY_NAMESPACE , 'count' , str( count ) )
        
        self._addTextNode( dataNode , "userName" ,  userName )
        
        typeNode = Type.toDom()
        dataNode.appendChild( typeNode )
        
        ## the format is already in Type and the node created by mobyleType.toDom() 
#        if format:
#            self._addTextNode( typeNode , "dataFormat" , format )
#            dataNode.appendChild( typeNode )
        
        self._addTextNode( dataNode , "headOfData" ,  dataBegining )
        
        if inputModes:   
            inputModesNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'inputModes' )
            for inputMode in inputModes:
                self._addTextNode( inputModesNode , "inputMode" ,  inputMode )
                dataNode.appendChild( inputModesNode )
        
        for jobId in producedBy:
            producedByNode = self._doc.createElementNS( EMPTY_NAMESPACE , "producedBy" )
            producedByNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , jobId )
            dataNode.appendChild( producedByNode )
            
        for jobId in usedBy:
            usedByNode = self._doc.createElementNS( EMPTY_NAMESPACE , "usedBy" )
            usedByNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , jobId )
            dataNode.appendChild( usedByNode )           
            
        dataListNode.appendChild( dataNode )  
        self._setModified( True )
        
      
    def _getDataNode(self , dataID ):
        """
        @param dataID: the identifier of one data in this session
        @type dataID: string
        @return: the Node corresponding to the dataID
        @rtype: Ft.Xml.Node instance
        @raise ValueError: if dataID does not match any entry in this session.
        """
        try:
            dataNode = self._root.xpath( 'dataList/data[@id = "%s"]'% dataID )[0]
            return dataNode
        except IndexError:
            msg = "the data %s does not exist in the session %s" %( dataID , self.getID() )
            raise ValueError , msg
               
                           
    def _dataNode2dataDict(self, dataNode , dtf ):
        """
        @param dataNode: a node representing a data
        @type dataNode: Ft.Xml.Node instance
        @param dtf: a dataTypeFactory
        @type dtf: a L{Mobyle.Classes.Core.DataType.DataTypeFactory} instance
        """
        data = {}
        data[ 'dataName' ] = str( dataNode.xpath( '@id' )[0].value )
        try:
            data[ 'userName' ] = str( dataNode.xpath( 'userName/text()' )[0].data )
        except IndexError:
            msg = "data %s has no userName" % data[ 'dataName' ]
            self._log.error( msg )
            SessionError , msg
        
        try:
            data[ 'Type' ] = ServiceParser.parseType( dataNode.xpath( 'type' )[0] , dataTypeFactory = dtf )
        except ( ParserError , MobyleError ) ,err :
            msg = "error in type parsing for data %s "% data[ 'dataName' ]
            self._log.error( msg , exc_info = True )
            raise SessionError , msg
        try:
            data[ 'dataBeg' ] = str( dataNode.xpath( 'headOfData/text()' )[0].data )
        except IndexError:
            data[ 'dataBeg' ] = ''
        try:
            data[ 'inputModes' ] = [ str( inputmode.xpath( 'text()' )[0].data )  for inputmode  in dataNode.xpath( 'inputModes/inputMode' ) ]
        except IndexError:
            pass
        
        data[ 'producedBy' ] =  [ str( jobID.value ) for jobID in dataNode.xpath( 'producedBy/@ref' ) ]
        data[ 'usedBy' ] =  [ str( jobID.value ) for jobID in dataNode.xpath( 'usedBy/@ref' ) ]
        
        try:
            data[ 'format' ] = ( str( dataNode.xpath('type/dataFormat/text()')[0].data ) ,
                                 int( dataNode.xpath( '@count' )[0].value ),
                                 'to fill the api (unused)' , 
                                 )
        except IndexError:
            data[ 'format' ] = None
        data[ 'size' ] = int( dataNode.xpath( '@size' )[0].value )
        
        return data 


    #####################
    #
    #   jobs methods
    #
    #####################
    
    def hasJob( self , jobID ):
        """
        @param jobID: the identifier of the job in the session ( the url without index.xml )
        @type jobID: string
        @return: True if the session structure has an entry for the job corresponding to this ID, False otherwise.
        @rtype: boolean 
        """
        try:
            self._root.xpath( 'jobList/job[@id="%s"]'% jobID )[0]
            return True
        except IndexError:
            return False 

    
    def getJob(self , jobID ):    
        """
        @param jobID: the identifier of the job in the session ( the url without index.xml )
        @type jobID: string
        @return: the job corresponding to the jobID
        @rtype: {'jobID'       : string ,
                 'userName'    : string ,
                 'programName' : string ,
                 'status'      : Status object ,
                 'date'        : time struct ,
                 'dataProduced': [ string dataID1 , string dataID2 , ...] ,
                 'dataUsed'    : [ string dataID1 , string dataID2 , ...] ,
                }
        @raise ValueError: if the jobID does not match any entry in this session
        """
        jobNode = self._getJobNode( jobID )
        return self._jobNode2jobDict( jobNode )

    
    def getAllJobs(self):
        """
        @return: the list of jobs in this session
        @rtype:  [ {'jobID'       : string ,
                    'userName'    : string ,
                    'programName' : string ,
                    'status'      : Status object ,
                    'date'        : time struct ,
                    'dataProduced': [ string dataID1 , string dataID2 , ...] ,
                    'dataUsed'    : [ string dataID1 , string dataID2 , ...] ,
                   } , ... ]
        """
        jobs = []
        jobList = self._root.xpath( 'jobList/job' )
        for jobNode in jobList:
            job = self._jobNode2jobDict( jobNode )
            jobs.append( job )
        return jobs
    

    def createJob(self , jobID , userName , programName , status , date , dataUsed ,  dataProduced ):
        """
        add a new job entry in this session
        @param jobID: the identifier of the job in the session ( the url without index.xml )
        @type jobID: string
        @param userName: the name of this job given by the user
        @type userName: string
        @param programName: the name of the program which generate this job
        @type programName: string
        @param status: the status of this job
        @type status: L{Utils.Status} instance
        @param date: the date of job submission
        @type date: time struct 
        @param dataUsed: the list of data stored in this session used by this job.
        @type dataUsed: list of strings
        @param dataProduced: the list of data stored in this session and produced by this job.
        @type dataProduced: list of strings
        """
        try:
            jobListNode = self._root.xpath( 'jobList' )[0]
        except IndexError:
            jobListNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'jobList')
            self._root.appendChild( jobListNode )
        jobexist = self._doc.xpath( 'joblist/job[@id="%s"]'%jobID )
        if jobexist:
            raise ValueError , "can't create a new job entry in session %s :the jobID %s already exist" %( self.getID() , jobID ) 
        
        jobNode = self._doc.createElementNS( EMPTY_NAMESPACE , 'job' )
        jobNode.setAttributeNS( EMPTY_NAMESPACE , 'id' , jobID )
        self._addTextNode( jobNode, 'userName' , userName )
        self._addTextNode( jobNode, 'programName' , programName )
        self._addTextNode( jobNode, 'date' , strftime( "%x  %X" , date ) )
        self._addTextNode( jobNode, 'status' , str( status ) ) 
        
        for dataID in dataUsed:
            dataUsedNode = self._doc.createElementNS( EMPTY_NAMESPACE , "dataUsed" )
            dataUsedNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , dataID )
            jobNode.appendChild( dataUsedNode )
            
        for dataID in dataProduced:
            dataProducedNode = self._doc.createElementNS( EMPTY_NAMESPACE , "dataProduced" )
            dataProducedNode.setAttributeNS( EMPTY_NAMESPACE , 'ref' , dataID )
            jobNode.appendChild( dataProducedNode )     
        
        jobListNode.appendChild( jobNode )
        self._setModified( True )
  
    
    def removeJob(self , jobID ):
        """
        remove the entry corresponding to this jobID 
        @param jobID: the identifier of the job in the session ( the url without index.xml )
        @type jobID: string
        @return: the job corresponding to jobID as a dict
        @rtype: dictionary
        @raise ValueError: if the jobID does not match any entry in this session
        """
        jobNode = self._getJobNode(jobID)
        jobDict = self._jobNode2jobDict( jobNode )
        jobListNode = jobNode.xpath( '..' )[0]
        jobListNode.removeChild( jobNode )
        ## remove the ref of this job from the data
        dataUsedNodes = jobNode.xpath( 'dataUsed' )
        for dataUsedNode in dataUsedNodes:
            dataID = str( dataUsedNode.xpath( '@ref' )[0].value )
            try:
                dataNode = self._getDataNode( dataID )
            except ValueError:
                continue
            else:
                try:
                    usedByNode = dataNode.xpath( 'usedBy[@ref="%s"]' %jobID )[0]
                except IndexError:
                    continue
                else:
                    dataNode.removeChild( usedByNode )
            
        dataProducedNodes = jobNode.xpath( 'dataProduced' )
        for dataProducedNode in dataProducedNodes:
            dataID = str( dataProducedNode.xpath( '@ref' )[0].value )
            try:
                dataNode = self._getDataNode( dataID )
            except ValueError:
                continue
            else:
                try:
                    producedByNode = dataNode.xpath( 'producedBy[@ref="%s"]' %jobID )[0]
                except IndexError:
                    continue
                else:
                    dataNode.removeChild( producedByNode )
        
        self._setModified( True ) 
        return jobDict       
               
    def renameJob( self , jobID , newUserName ):
        """
        change the user name of the data corresponding to the dataID.
        @param jobID: the identifier of the job in the session ( url without index.xml )
        @type jobID: string
        @param newUserName: the new user name of the job.
        @type newUserName: string
        @raise ValueError: if jobID does not match any entry in session.        
        """
        jobNode = self._getJobNode( jobID )
        modified = self._updateNode( jobNode , 'userName' ,  newUserName )
        self._setModified( modified )
        
        
    def updateJobStatus( self ,jobID , status ):
        """
        update the status of the job corresponding to the jobID
        @param jobID: the identifier of the job in the session ( url without index.xml )
        @type jobID: string
        @param status: the status of this job
        @type status: L{Utils.Status} instance
        @raise ValueError: 
        """
        jobNode = self._getJobNode( jobID )
        modified = self._updateNode( jobNode , 'status' ,  str( status ) )
        self._setModified( modified )
        
            
    def _getJobNode( self , jobID ):
        """
        @param jobID: the identifier of one job in this session
        @type jobID: string
        @return: the Node corresponding to the jobID
        @rtype: Ft.Xml.Node instance
        @raise ValueError: if jobID does not match any entry in this session.
        """
        try:
            return self._root.xpath( 'jobList/job[@id="%s"]'% jobID )[0]
        except IndexError:
            msg = "the job %s does not exist in the session %s" %( jobID , self.getID() )
            raise ValueError , msg
                    
    
    def _jobNode2jobDict( self , jobNode ):
        """
        @param jobNode: a node representing a job
        @type jobNode: Ft.Xml.Node instance
        """
        job = {}
        job [ 'jobID' ] = str( jobNode.xpath( '@id' )[0].value )
        try:
            job[ 'userName' ] = str( jobNode.xpath( 'userName/text()' )[0].data )
        except IndexError:
            self._log.error( "the job %s in session %s has no userName" %( job[ 'jobID' ] , self.getID()) )
        try:
            job[ 'programName' ] =  str( jobNode.xpath( 'programName/text()' )[0].data ) 
        except ( ParserError , MobyleError ) ,err :
            self._log.error( "the job %s in session %s has no programName" %( job[ 'jobID' ] , self.getID()) )
        try:
            statusString = str( jobNode.xpath( 'status/text()' )[0].data )
        except IndexError :
            self._log.error( "the job %s in session %s has no status" %( job[ 'jobID' ] , self.getID()) )
            raise MobyleError 
        else:
            try:
                job[ 'status' ] = Status( string= statusString )
            except KeyError:
                msg = "error in status %s for job %s in session %s" %( statusString ,
                                                                       job[ 'jobID' ] , 
                                                                       self.getID()
                                                                       ) 
                self._log.error( msg )
                raise MobyleError ,msg
        try:
            job[ 'date' ] = strptime( jobNode.xpath( 'date/text()' )[0].data , "%x  %X")
        except IndexError:
            pass  
      
        job[ 'dataProduced' ] =  [ str( dataID.value ) for dataID in jobNode.xpath( 'dataProduced/@ref' ) ]
        job[ 'dataUsed' ] =  [ str( dataID.value ) for dataID in jobNode.xpath( 'dataUsed/@ref' ) ]
        
        return job 
       
       
       
