########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


import os , os.path
from time import localtime, strftime , strptime

import logging
u_log = logging.getLogger( 'mobyle.utils' )

from Mobyle.MobyleError import MobyleError , UserValueError


class Admin:
    """
    manage the informations in the .admin file.
    be careful there is no management of concurrent access file
    thus if there is different instance of Admin with the same path
    it could be problem of data integrity
    """
    FIELDS = ( 'DATE'    ,
               'EMAIL'   ,
               'REMOTE'  ,
               'SESSION' ,
               'JOBNAME' ,
               'JOBID'   ,
               'MD5'     ,
               'BATCH'   ,
               'NUMBER'  ,
               'QUEUE'   ,
               'STATUS'  ,
               'MESSAGE'
               )

    
    def __init__(self , path ):
        self.me = {}
        path = os.path.abspath( path )
        
        if os.path.exists( path ):

            if os.path.isfile( path ):
               self.path = path
               self._parse()


            elif os.path.isdir( path ):
                self.path = os.path.join( path , ".admin" )

                if os.path.isfile( self.path ):
                    self._parse()
                else:
                    pass
                    
        else:
            basename , filename = os.path.split( path )
            if filename == ".admin":
                if os.path.exists( basename ):
                    self.path = path
                else:
                    raise MobyleError , "The path to the .admin is invalid : "+ basename
            else:
                raise MobyleError , "invalid path : " + path
    
                
    def __str__( self ):
        res = ''
        for key in self.__class__.FIELDS :
            try:
                if key == 'DATE':
                    value = strftime( "%x %X" , self.me['DATE'] )

                elif key == 'STATUS':
                    value = str( self.me[ key ] )
                else:
                    value =  self.me[ key ]
            except KeyError :
                continue
            
            res = res + "%s : %s\n" %( key , value )

        return res



    def _parse ( self ):
        """
        parse the file .admin
        """
        try:
            fh = open( self.path , 'r' )

            for line in fh:
                datas = line[:-1].split( ' : ' )
                key = datas[0]
                value = ' : '.join( datas[1:] )
                if key == 'DATE':
                    value = strptime( value , "%x %X" )
                self.me[ key ] = value
        finally:
            fh.close()

                 
    def refresh( self ):
        self._parse()

    def commit( self ):
        """
        Write the string representation of this instance on the file .admin
        """
        fh = open( self.path , 'w' )
        fh.write( self.__str__() )
        fh.close()


    def getDate( self ):
        """
        @return: the date of the job submission
        @rtype: time.struct_time
        """
        try:
            return  self.me[ 'DATE' ] 
        except KeyError :
            return None
 

    def setDate( self , date = None , format = None):
        """
        set the date of the job submission
        @param date: the date of the job submission. if date is None use localtime 
        @type date: a time.struct_time or a string
        @param format: if the date is a string represent the format of date. if it's None use \"%a %b %d %H:%M:%S %Y\" as default format.
        @type format: string
        """
        
        if date is None:
            myDate = localtime() 

        elif type( date ) == type( '' ) :
            if format is not None:
                myDate = strptime( date , format ) 
            else:
                myDate = strptime( date ) 
                    
        elif type( date ) == type( localtime( 0 ) ):
            myDate =  date 

        self.me[ 'DATE' ] = myDate



    def getEmail( self ):
        """
        @return: the email of the user who run the job 
        @rtype: string
        """
        try:
            return self.me[ 'EMAIL' ]
        except KeyError :
            return None


    def setEmail( self , email ):
        """
        set the email of the job
        @param jobName: the email of the job
        @type jobName: string
        """
        self.me[ 'EMAIL' ] = str( email )


    def getRemote( self ):
        """
        @return: the remote of the job 
        @rtype: string
        """
        try:
            return self.me[ 'REMOTE' ]
        except KeyError :
            return None


    def setRemote( self , remote_ip ):
        """
        set the  of the job
        @param jobName: the  of the job
        @type jobName: string
        """
        self.me[ 'REMOTE' ] = remote_ip


    def getSession( self ):
        """
        @return: the Session path of the job 
        @rtype: string
        """
        try:
            return self.me[ 'SESSION' ]
        except KeyError :
            return None


    def setSession( self , sessionKey ):
        """
        set the  Session path of the job
        @param jobName: the  of the job
        @type jobName: string
        """
        self.me[ 'SESSION' ] = sessionKey


    def getJobName( self ):
        """
        @return: the name of the job ( blast2 , toppred )
        @rtype: string
        """
        try:
            return self.me[ 'JOBNAME' ]
        except KeyError :
            return None


    def setJobName( self , jobName ):
        """
        set the name of the job
        @param jobName: the name of the job
        @type jobName: string
        """
        self.me[ 'JOBNAME' ] = jobName

    def getJobID( self ):
        """
        @return: the job identifier.
        @rtype: string
        """
        try:
            return self.me[ 'JOBID' ]
        except KeyError :
            return None


    def setJobID( self , jobID ):
        """
        set the identifier of the job
        @param : the identifier of the job
        @type : string
        """
 
        self.me[ 'JOBID' ] = jobID


    def getMd5( self ):
        """
        @return: the md5 of the job
        @rtype: string
        """
        try:
            return self.me[ 'MD5' ]
        except KeyError :
            return None


    def setMd5( self , md5 ):
        """
        set the md5 of the job
        @param : the identifier of the job
        @type : string
        """
 
        self.me[ 'MD5' ] = md5



    def getBatch( self ):
        """
        @return: the batch system name used to run the job  
        @rtype: string
        """
        try:
            return self.me[ 'BATCH' ]
        except KeyError :
            return None


    def setBatch( self , batch):
        """
        set the  batch system used to run of the job
        @param : the name of the batch system 
        @type : string
        """
        self.me[ 'BATCH' ] = batch


    def getNumber( self ):
        """
        @return: the number the job the meaning of this value depend of BATCH value.
         - BATCH = Sys number is the job pid
         - BATCH = SGE number is the result of qstat
        @rtype: string
        """
        try:
            return self.me[ 'NUMBER' ]
        except KeyError :
            return None


    def setNumber( self , number ):
        """
        set the number of the job this number depend of the batch value
        if BATCH = Sys number is the pid
        if BATCH = SGE number is the 
        @param : the number of the job
        @type : string
        """
        self.me[ 'NUMBER' ] = number

    def getQueue( self ):
        """
        @return: return the queue name of the job 
        @rtype: string
        """
        try:
            return self.me[ 'QUEUE' ]
        except KeyError :
            return None


    def setQueue( self, queue ):
        """
        set the queue name of the job
        @param : the queuename of the job
        @type : string
        """
        self.me[ 'QUEUE' ] = queue


    def getStatus( self ):
        """
        @return: the job status."
        @rtype: L{Status} instance
        """
        try:
            status = self.me[ 'STATUS' ] 
        except KeyError :
            return None
        try:
            msg = self.me[ 'MESSAGE' ]
        except KeyError :
            msg = None
        return  Status( string= status, message =  msg )
        

    def setStatus( self , status):
        """
        set the  status of the job
        @param : the status of the job.
        @type : L{Status} instance.
        """
        self.me[ 'STATUS' ] = str( status )
        if status.message:
            self.me[ 'MESSAGE' ] = status.message

        
    def getMessage( self ):
        """
        @return: the job status. the values could be \"summitted\" ,\"pending\",\"running\" ,\"finished\",\"error\"
        @rtype: string
        """
        try:
            return self.me[ 'MESSAGE' ]
        except KeyError :
            return None


    def setMessage( self , message ):
        """
        set a  message used when status == error
        @param : a error message
        @type : string
        """
        self.me[ 'MESSAGE' ] = message


class Status:

    _CODE_2_STATUS = { 
                      0 : "building"  , # the job directory has been created
                      1 : "submitted" , # the job.run method has been called
                      2 : "pending"   , # the job has been submitted to the batch manager but wait for a slot 
                      3 : "running"   , # the job is running in the batch manager
                      4 : "finished"  , # the job is finished without error from Mobyle
                      5 : "error"     , # the job is encounter a MobyleError 
                      6 : "killed"    , # the job has been removed by the user, or killed by the admin
                      7 : "hold"      , # the job is hold by the batch manager
                      8 : "finishing"   # the job is not anymore in the batch manager, but Mobyle is not yet finished
                      }
    
    def __init__(self , code = None , string = None , message= ''):
        """
        @param code: the code of the status 
        @type code: integer 
        @param string: the code of the status representing by a string
        @type string: string
        @param message: the message associated to the status
        @type message: string
        """
        if code is None and not string :
            raise MobyleError , "status code or string must be specify"
        elif code is not None and string :
            raise MobyleError, "status code or string must be specify, NOT both"
        elif code is not None :
            if code in self._CODE_2_STATUS.keys():
                self.code = code
            else:
                raise MobyleError , "invalid status code :" + str( code )
        elif string:
            if string in  self._CODE_2_STATUS.values():
                iterator = self._CODE_2_STATUS.iteritems()
                for code , status in iterator:
                    if status == string:
                        self.code = code
            else:
               raise MobyleError , "invalid status :" + str( string ) 
        
        if message:
            self.message = message
        else:
            self.message = ''
            
        
    def __eq__(self , other):
        return self.code == other.code and self.message == other.message
    
    def __str__(self):
        return self._CODE_2_STATUS[ self.code ]
    
    def isEnded(self):
        """
        """
        return self.code in ( 4 , 5 , 6 , 8 )
        # finishing (8) is not properly ended but is't too late to kill it
        
    def isQueryable(self):
        """
        """
        return self.code in( 2 , 3 , 7 )
    
    
    
    
def getStatus( jobID ):
    """
    @param jobID: the url of the job
    @type jobID: string
    @return: the current status of the job
    @rtype: string
    @raise MobyleError: if the job has no number or if the job doesn't exist anymore
    @raise OSError: if the user is not the owner of the process
    """
    import Mobyle.JobState
    import Mobyle.RunnerChild
    import urlparse
    
    path = Mobyle.JobState.normUri( jobID )
    protocol ,host, path, a,b,c = urlparse.urlparse( path )
    if protocol == "http":
        raise  NotImplementedError  , "trying to querying a distant server"
    
    if path[-9:] == "index.xml":
        path = path[:-10 ]
    adm = Admin( path )
    oldStatus = adm.getStatus()
    #'killed' , 'finished' , 'error' the status cannot change anymore
    #'submitted', 'building' these jobs have not yet sge number

    #  ( 'finished' , 'error' , 'killed' , 'building' , 'submitted' ):
    if oldStatus.isEnded():
        return oldStatus
    
    else:
        batch = adm.getBatch()
        jobNum = adm.getNumber()
        
        if batch is None :
            import Mobyle.MobyleError
            raise Mobyle.MobyleError.MobyleError , "BATCH is unknown for job : " + str( jobID )
        if jobNum is None:
            import Mobyle.MobyleError
            raise Mobyle.MobyleError.MobyleError ,"NUMBER is unknown for job : " + str( jobID )
        try:
            if batch == ( "Sys" ):
                newStatus = Mobyle.RunnerChild.Sys.getStatus( jobNum )
            elif batch == ( "PBS" ):
                newStatus = Mobyle.RunnerChild.PBS.getStatus( jobNum )
            elif batch == ( "SGE" ):
                newStatus = Mobyle.RunnerChild.SGE.getStatus( jobNum )
            else:
                raise NotImplementedError, "unknown batch \""+ batch +"\" "
        except MobyleError ,err : 
            u_log.error( str( err ) )
            raise err
        if newStatus != oldStatus :
            if newStatus is None:
                newStatus = Status( code = 8 )#finishing
            
            adm.setStatus( newStatus )
            adm.commit()

            jobState = Mobyle.JobState.JobState( jobID )
            jobState.setStatus( newStatus ) 
            jobState.commit()
            
        return newStatus 





def killJob( jobID ):
    """
    @param jobID: the url of the job or a sequence of jobID
    @type jobID: string or sequence of jobID
    @return: 
    @rtype: 
    @raise MobyleError: if the job has no number or if the job doesn't exist anymore
    @todo: tester la partie sge
    """
    
    import Mobyle.RunnerChild
    import types
    from Mobyle.MobyleError import MobyleError
    
    jobIDType = type( jobID )
    
    if jobIDType == types.StringType :
        jobIDs = [ jobID ]
    elif jobIDType == types.TupleType or jobIDType == types.ListType :
        jobIDs = jobID
    else:
        raise MobyleError , "jobID must be a string ar a Sequence of strings"
    
    errors = []
    for jobID in jobIDs :
        try:
            path = Mobyle.JobState.normUri( jobID )
        except MobyleError , err :
            errors.append( ( jobID , str( err ) ) )
            continue
        if path[:4] == 'http' :
            #the jobID is not on this Mobyle server
            errors.append( ( jobID , "can't kill distant job" ) )
            continue

        try:
            adm = Admin( path )
        except MobyleError , err :
            errors.append( ( jobID , "invalid jobID" ) )
            continue
        
        batch = adm.getBatch()
        jobNum = adm.getNumber()
        if jobNum is None:
            # an error occured on this jobID but I continue to kill the other jobIDs
            errors.append( ( jobID , 'no jobNum for this job' ) )
            continue
        try:
            if batch == ( "Sys" ):
                Mobyle.RunnerChild.Sys.kill( jobNum )            
            elif batch == ( "SGE" ):
                Mobyle.RunnerChild.SGE.kill( jobNum )
            elif batch == ( "PBS" ):
                Mobyle.RunnerChild.PBS.kill( jobNum )
            else:
                # an error occured on this jobId but I continue to kill the other jobId
                error.append( ( jobID , "unknown batch \""+ batch +"\" " ) )
                continue
        except MobyleError ,err :
            errors.append( ( jobID , str( err ) ) )
    if errors:
        msg = ''
        for jobID , msgErr in errors :
            msg = "%s killJob( %s ) - %s\n" %( msg , jobID , msgErr )
            
        raise MobyleError , msg



def safeFileName( fileName ):
    import string , re
    
    if fileName in ( 'index.xml' , '.admin' , '.command' ,'.forChild.dump' ,'.session.xml'):
        raise UserValueError( msg = "value \"" + str( fileName ) + "\" is not allowed" )
    
    for car in fileName :
        if car not in string.printable : #we don't allow  non ascii char
            fileName = fileName.replace( car , '_')
    
    #SECURITY: substitution of shell special characters
    fileName = re.sub( "[ #\"\'<>&\*;$`\|()\[\]\{\}\?\s ]" , '_' , fileName )
                  
    #SECURITY: transform an absolute path in relative path
    fileName = re.sub( "^.*[\\\:/]", "" , fileName )
    
    return fileName



def makeService( serviceUrl ):
    from Ft.Lib import Uri , UriException
    import Mobyle.Parser
    
    try:
        parser = Mobyle.Parser.ServiceParser()
        service = parser.parse( serviceUrl )
        return service
    except IOError , err:
        raise MobyleError , str( err )

      
def sizeFormat(bytes, precision=2):
     """Returns a humanized string for a given amount of bytes"""
     import math
     bytes = int(bytes)
     if bytes is 0:
         return '0 bytes'
     log = math.floor( math.log( bytes , 1024 ) )
     return "%.*f%s" % ( precision , bytes / math.pow(1024, log), [ 'bytes', 'KiB', 'MiB' , 'GiB' ][ int(log) ] )