

########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


"""
Classes executing the command and managing the results  
"""
import os 
from subprocess import Popen, PIPE

import logging 
_log = logging.getLogger(__name__)

#import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
from Mobyle.Execution.Batch import _Batch

import Local.Policy
from Mobyle.MobyleError import *


_cfg = Mobyle.ConfigManager.Config()
__extra_epydoc_fields__ = [('call', 'Called by','Called by')]



          ###########################################################
          #                                                         #
          #       to run with queuing system Sun Grid Engine        #
          #                                                         #
          ###########################################################
                
class SGE (_Batch):
    """
    
    """
        
    @staticmethod
    def _sgeInit( ):

        SGE_ROOT , SGE_CELL = _cfg.sge()
                    
        arch_path= os.path.join( SGE_ROOT , 'util' , 'arch' )

        try:
            arch_pipe = Popen(arch_path         ,
                              shell     = False ,
                              stdout    = PIPE  ,
                              stdin     = None  ,
                              stderr    = None  ,
                              close_fds = True  )
            arch_pipe.wait() 
            arch_rc = arch_pipe.returncode
            
        except OSError , err:
            #this error is log by calling method because I can't access to jobKey , adm status ... from static method
            msg = "SGE: I can't determined the system arch:"+str(err)
            raise MobyleError , msg
     
        if arch_rc != 0 :
            msg = "I can't determined the system arch (return code = " + str( arch_rc ) + " )"
            raise MobyleError , msg
        
        arch = ''.join( arch_pipe.stdout.readlines() ).strip()

        sge_prefix = os.path.join( SGE_ROOT , 'bin' , arch )
        sge_env = {'SGE_CELL': SGE_CELL , 'SGE_ROOT': SGE_ROOT }   
        return ( sge_prefix , sge_env )


    def run(self):
        """
        """
        if os.getcwd() != os.path.abspath( self.dirPath ):
            msg = "the child process execute itself in a wrong directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg  )

            raise MobyleError , msg 
        else:
            jobKey = self.getKey()
            options = { '-cwd': ''           ,  #set the working dir to current dir
                        '-now': 'n'          ,  #the job is not executed immediately
                        '-N' : jobKey        ,  #the id of the job
                        '-q' : self.queue    ,  #the queue 
                        '-V' : ''               #job inherits of the whole environment
                        }
            sge_opts = ''

            for opt in options.keys():
                sge_opts += opt + ' ' + options[opt]+' '
            
            try:
                sge_prefix , sge_env = SGE._sgeInit()
            except MobyleError , err:
                self._logError( admMsg = str( err ) ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( str( err ) ) 
                    
                raise MobyleError , err
                      
            sge_cmd = os.path.join( sge_prefix , 'qrsh' ) + ' ' + sge_opts
            cmd = sge_cmd + " " + self._protectedCommandLine

            try:
                fout = open( self.serviceName + ".out" , 'w' )
                ferr = open( self.serviceName + ".err" , 'w' )
            except IOError , err:
                msg= "SGE: can't open file for standard job output: "+ str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = msg )

                raise MobyleError , msg

            self.xmlEnv.update( sge_env )
            
            try:
                pipe = Popen( cmd ,
                              shell  = True ,
                              stdout = fout  ,
                              stdin  = None  ,
                              stderr = ferr  ,
                              close_fds = True ,
                              env       = self.xmlEnv
                              )

            except OSError, err:
                msg= "SGE execution failed: "+ str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = msg )

                raise MobyleError , msg

            self._adm.setBatch( 'SGE')
            self._adm.setNumber( jobKey )
            self._adm.commit()
            
            linkName = ( "%s/%s.%s" %( _cfg.admindir() ,
                                       self.serviceName ,
                                       jobKey
                                    )
                         )
            try:
                os.symlink(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                msg = "can't create symbolic link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )

                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey  ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg
            pipe.wait()

            try:
                os.unlink( linkName )
            except OSError , err:
                msg = "can't remove symbolic link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )

                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            fout.close()
            ferr.close()
            self._returncode = pipe.returncode
            self._adm.refresh()
            oldStatus = self._adm.getStatus()
            if oldStatus.isEnded():
                return oldStatus
            else:
                if pipe.returncode == 0 :# finished
                    status = Mobyle.Utils.Status( code = 4 ) #finished
                elif pipe.returncode in ( 137 , 143 , 153 ): #killed
                    ## 137 =  9 ( SIGKILL ) + 128 ( python add 128 )
                    ## 143 = 15 ( SIGTERM )
                    ## 153 = 25 ( SIGXFSZ ) + 128 (file size exceeded )
                    ## be careful if a job exit with a 137 or 153 exit status it will be labelled as killed
                    status = Mobyle.Utils.Status( code = 6 , message = "Your job has been cancelled" ) 
                elif pipe.returncode > 128 :
                    status = Mobyle.Utils.Status( code = 6 , message = "Your job execution failed ( %s )" %pipe.returncode ) 
                    ## if return code > 128 it's a signal
                    ## the 9 , 15 25 signal are send by administrator
                    ## the other are self aborting signal see signal(7)   
                else:
                    status = Mobyle.Utils.Status( code = 4 , message = "Your job finished with an unusual status code ( %s ), check your results carefully." % pipe.returncode )
                return status


    @staticmethod
    def getStatus( jobNum ):
        """
        @param jobNum:
        @type jobNum:
        @return: the status of job with number jobNum
        @rtype: string
        @todo: for best performance, restrict the sge querying to one queue
        """
        sge_prefix , sge_env = SGE._sgeInit( )
        sge_cmd = [ os.path.join( sge_prefix , 'qstat' ) , '-r' ]
        
        try:
            pipe = Popen( sge_cmd,
                          executable = sge_cmd[0] ,
                          shell = False ,
                          stdout = PIPE ,
                          stdin = None  ,
                          stderr= None  ,
                          close_fds = True ,
                          env= sge_env
                          )
        except OSError , err:
            raise MobyleError , "can't querying sge : %s :%s "%( sge_cmd , err )

        sge2mobyleStatus = { 'r' : 3 , # running
                             't' : 3 ,
                             'R' : 3 ,
                             's' : 7 , #hold
                             'S' : 7 , 
                             'T' : 7 ,
                             'h' : 7 ,
                             'w' : 2 , #pending
                             'd' : 6 , #killed 
                             'E' : 5 , #error
                             }

        for job in pipe.stdout :
            
            jobState = job.split()
            try:
                status = jobState[ 4 ][ -1 ]
                continue
            except ( ValueError , IndexError ):
                pass #it's not the fisrt line
            try:
                if not ( jobNum == jobState[2] and 'Full' == jobState[0] ):
                    continue #it's not the right jobNum
            except IndexError :
                continue #it's not the 2nde line
            pipe.stdout.close()
            pipe.wait()
            if pipe.returncode == 0 :
                try:
                    return Mobyle.Utils.Status( code = sge2mobyleStatus[ status ]  )
                except MobyleError , err :
                    raise MobyleError , "unexpected sge status: " +str( status )
            else:
                raise MobyleError , "cannot get status " + str( pipe.returncode )
        return Mobyle.Utils.Status( code = -1 ) #unknown

        
    @staticmethod
    def kill( job_number ):
        """
        @todo: kill the Job
        """
        sge_prefix , sge_env = SGE._sgeInit( )
        sge_cmd = "%s %s 1>&2" %( os.path.join( sge_prefix , 'qdel' ) ,  job_number )
        os.environ.update( sge_env )
        os.system( sge_cmd )
        

