#! /usr/bin/env python


########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


"""
Classes executing the command and managing the results  
"""
import os 
import signal , siginterrupt
import sys
import glob
import time
import cPickle
import zipfile , StringIO
import Ft.Xml.Domlette
import atexit

import logging 

#"the environment variable MOBYLEHOME must be defined in the environment"
#append Mobyle Home to the search modules path
MOBYLEHOME = None
if os.environ.has_key('MOBYLEHOME'):
    MOBYLEHOME = os.environ['MOBYLEHOME']
if not MOBYLEHOME:
    sys.exit('MOBYLEHOME must be defined in your environment')

if ( os.path.join( MOBYLEHOME , 'Src' ) ) not in sys.path:
    sys.path.append(  os.path.join( MOBYLEHOME , 'Src' )  )


from Mobyle import MobyleLogger
MobyleLogger.MLogger( child = True )

rc_log = logging.getLogger( 'Mobyle.RunnerChild' )

import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
import Mobyle.Net

import Local.Policy
from Mobyle.MobyleError import MobyleError , EmailError , TooBigError


_cfg = Mobyle.ConfigManager.Config()
__extra_epydoc_fields__ = [('call', 'Called by','Called by')]


class AsynchronJob:
    """
    is instantiated in child process instantiate the object corresponding
    to the execution manager defined in Config, and after the completion of the job
    manage the results
    """

    
    def __init__(self, commandLine, dirPath, serviceName, resultsMask , userEmail = None, jobState = None , xmlEnv = None):
        """
        @param commandLine: the command to be executed
        @type commandLine: String
        @param dirPath: the absolute path to directory where the job will be executed (normaly we are already in)
        @type dirPath: String
        @param serviceName: the name of the service
        @type serviceName: string
        @param resultsMask: the unix mask to retrieve the results of this job
        @type resultsMask: a dictionary { paramName : [ string prompt , ( string class , string or None superclass ) , string mask ] }
        @param jobState: the jobState link to this job
        @type jobState: a L{JobState} instance
        @call: by the main of this module which is call by L{AsynchronRunner}
        """
        self._command = commandLine
        self._dirPath = dirPath
        self.serviceName = serviceName
        self.father_pid = os.getppid()
        self.father_done = False

        signal.signal( signal.SIGUSR2, self.sigFromFather )
 
        # encapsulated module to siginterrupt system.
        # see Mobyle/Utils/Siginterrupt/siginterrupt.c
 
        # this cause that the signal SIGUSR1 send to the child 
        # will not break the system call ( pipe.wait )
        # the child process is notified a sigusr2 has been received
        # but the handler will be executed at the end of the system call ( pipe wait )
        
        siginterrupt.siginterrupt( signal.SIGUSR2 , False )
        
        if jobState is None:
            self.jobState = Mobyle.JobState.JobState( os.getcwd() )
        self.userEmail = userEmail
        if self._dirPath[-1] == '/':
            self._dirPath = self._dirPath[:-1]
        self.jobKey = os.path.split( self._dirPath )[ 1 ]
        atexit.register( self.childExit , "------------------- %s : %s -------------------" %( serviceName , self.jobKey ) )
        self._adm = Mobyle.Utils.Admin( self._dirPath )
        
        ############################################
        self._run( serviceName , jobState , xmlEnv )
        #############################################
        
        
        self.results = {}

        for paramName in resultsMask.keys():
            resultsFiles = []
            
            #type is a tuple ( klass , superKlass )
            prompt , Type , masks = resultsMask[ paramName ]

            for mask in masks :
                for File in  glob.glob( mask ):
                    size = os.path.getsize( File )
                    if size != 0:
                        resultsFiles.append(  ( str( File ) , size , None ) ) #we have not information about the output format 
            if resultsFiles: 
                self.results[ paramName ] = resultsFiles  #a list of tuple (string file name , int size ,  string format or None )
                self.jobState.setOutputDataFile( paramName , prompt , Type , resultsFiles )

        self.jobState.commit()
        try:
            zipFileName = self.zipResults()
        except Exception , err:
            msg = "an error occured during the zipping results :\n\n"
            rc_log.critical( "%s/%s : %s" %( self.serviceName , self.jobKey , msg ) , exc_info = True)
            zipFileName = None
            
        if self.father_done:
            self.emailResults(  FileName = zipFileName )
        else:
            #print >> sys.stderr, "AsynchronJob stderr: mon pere est encore la, je le previens"
            try:
                os.kill( self.father_pid , signal.SIGUSR1 )
            except OSError , err :
                msg = "Can't send a signal to my father process send results by email :" + str( err )
                rc_log.warning( msg )
                self.emailResults(  FileName = zipFileName )
                

    def childExit(self , message ):
        print >> sys.stderr , message
        if not self.father_done:
            os.kill( self.father_pid , signal.SIGUSR1 )

    def sigFromFather(self,signum ,frame):
        """
        @call: when the father is terminated and send a SIGUSR2 to the child process
        """
        self.father_done = True
        signal.signal(signal.SIGUSR2 , signal.SIG_IGN)


    def _run(self , serviceName , jobState , xmlEnv ):
        try:
            executionClass = Mobyle.Utils.executionClassLoader( serviceName )
        except MobyleError ,err :
            msg = "unknown execution system : %s" %err
            rc_log.critical("%s : %s" %( serviceName ,
                                         msg
                                         )
            )

            if jobState is None:
                jobState = Mobyle.JobState.JobState( uri = self._dirPath )
            jobState.setStatus( Mobyle.Utils.Status( code = 5 , message = 'Mobyle internal server error' ) )
            jobState.commit()

            self._adm = Mobyle.Utils.Admin( self._dirPath )
            self._adm.setStatus( Mobyle.Utils.Status( code = 5 , message = msg ) )
            self._adm.commit()
            
            raise MobyleError, msg
        except Exception , err:
            rc_log.error( str(err ), exc_info=True) 
            raise err
        executionClass( self._command , self._dirPath , serviceName , jobState = jobState , xmlEnv = xmlEnv)
    
    
    def emailResults(self , FileName = None ):
        """
        """

        dont_email_result , maxmailsize =  _cfg.mailResults()
        if dont_email_result :
            return
        else:
            if self.userEmail :
                mail = Mobyle.Net.Email( self.userEmail )
                jobInPortalUrl = "%s/portal.py?jobs=%s" %( _cfg.cgi_url() ,
                                                           self.jobState.getID() )
                
                if FileName is not None:
                    zipSize = os.path.getsize( FileName )
                    mailDict = { 'SENDER'         : _cfg.sender() ,
                                 'HELP'           : _cfg.mailHelp() ,
                                 'SERVER_NAME'    : _cfg.portal_url() ,
                                 'JOB_URL'        : jobInPortalUrl , 
                                 'RESULTS_REMAIN' : _cfg.remainResults() ,
                                 'JOB_NAME'       : self.serviceName ,
                                 'JOB_KEY'        : self.jobKey ,
                                 }
                    if zipSize > maxmailsize - 2048 :
                        #2048 octet is an estimated size of email headers
                        try:
                            mail.send( 'RESULTS_TOOBIG' , mailDict )
                            return
                        except EmailError ,err :
                            msg = str(err)
                            self._adm.setMessage( msg )
                            self._adm.commit()
                            rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                                          self.jobKey ,
                                                          msg
                                                          )
                            )
                            
                            return
                    else:
                        try:   
                            mail.send( 'RESULTS_FILES' , mailDict , files = [ FileName ]  )
                            return
                        except TooBigError ,err :
                            try:
                                mail.send( 'RESULTS_TOOBIG' , mailDict )
                            except EmailError ,err :
                                msg = str(err)
                                self._adm.setMessage( msg )
                                self._adm.commit()
                                rc_log.error( "%s/%s : %s" %( self.serviceName ,
                                                              self.jobKey ,
                                                              msg
                                                              )
                                )
                            
                            return
                else: #if there is a problem on zip creation
                    mail.send( 'RESULTS_NOTIFICATION' , mailDict )
            else:
                return


    def zipResults(self ):

        resultsFiles = []

        for File in self.results.values():
            resultsFiles += File  #File is tuple (string file name , int size , string format or None )
       
        xsl_path = os.path.join( _cfg.portal_path() , "xsl" ,)
        xslSize = os.path.getsize( xsl_path )
        resultsFiles.append( ( "index.xml" , xslSize , None ) )
        
        jobXslPath = os.path.join( xsl_path , "job.xsl" ) 
        jobXslSize = os.path.getsize( jobXslPath )
        resultsFiles.append( ( jobXslPath , jobXslSize , None ) )
                
        mobyleXslPath = os.path.join( xsl_path , "mobyle.xsl" )
        mobyleXslSize = os.path.getsize( mobyleXslPath )
        resultsFiles.append( ( mobyleXslPath , mobyleXslSize , None ) )
        
        paramsfiles = self.jobState.getParamfiles()
        if paramsfiles: 
            for paramsfile in paramsfiles:
                resultsFiles.append(  ( paramsfile[0] , paramsfile[1] , None ) )
        inputFiles = self.jobState.getInputFiles() #inputFiles = [ ( parameter , [ (fileName , format or None ) , ...) , ... ]
        if inputFiles is not None:
            for files in inputFiles:
                for item in files[1]: #item = ( filename , size , fmt ) 
                    resultsFiles.append( item )
            
        zipFileName = "%s_%s.zip" %(self.serviceName , self.jobKey )
        myZipFile = zipfile.ZipFile( zipFileName  , "w" )
        
        for fileName , size , fmt in resultsFiles :
            #for python up to 2.5 
            #If allowZip64 is True zipfile will create ZIP files that use the ZIP64 extensions when the zipfile is larger than 2 GB.
            #If it is false (the default) zipfile will raise an exception when the ZIP file would require ZIP64 extensions. 
            #ZIP64 extensions are disabled by default because the default zip and unzip v5.52 commands on Unix  don't support these extensions.
            
            if size > 0 and size < 2147483648 : #2Go 
                if fileName  == 'index.xml':
                    tree = self._indexPostProcess()
                    myZipInfo = zipfile.ZipInfo( 'index.xml', time.localtime()[:6] )
                    myZipInfo.external_attr = 2175008768   # set perms to 644
                    myZipFile.writestr(  myZipInfo  , tree )
                elif size < 10 :
                    myZipFile.write(   fileName   , os.path.basename( fileName ) , zipfile.ZIP_STORED )
                else:
                    myZipFile.write(   fileName   , os.path.basename( fileName ) , zipfile.ZIP_DEFLATED )
                    
            elif size >= 2147483648 :
                myZipInfo = zipfile.ZipInfo( os.path.basename( fileName ) , time.localtime()[:6] )
                myZipInfo.external_attr = 2175008768   # set perms to 644
                myZipFile.writestr(  myZipInfo  , "Sorry this file result is too large ( > 2GiB ) to be included in this archive." )    
            else:
                #the file is empty we don't add it to this archive
                pass
        myZipFile.close()
        return  zipFileName


    def _indexPostProcess( self ):
        fh_tree = open( 'index.xml' , 'r')
        tree = ''.join( fh_tree.readlines() )
        doc = Ft.Xml.Domlette.NoExtDtdReader.parseString( tree , uri = "file://%s" %os.path.abspath(fh_tree.name) )
        
        old_pi = doc.childNodes[0]
        new_pi = doc.createProcessingInstruction('xml-stylesheet' ,
                                                 'href="job.xsl" type="text/xsl"')
        doc.replaceChild( new_pi , old_pi)
        
        fh = StringIO.StringIO()
        Ft.Xml.Domlette.PrettyPrint( doc ,fh )
        tree = fh.getvalue()
        return tree 


if __name__ == '__main__':
    try:
        fh = open(".forChild.dump", "r")
        fromFather = cPickle.load( fh )
        fh.close() 
    except IOError ,err:
        pass
    except :
        pass

    try:
        os.chdir( fromFather[ 'dirPath' ] )
    except OSError, err:
        msg = fromFather[ 'serviceName' ] + ":" + str( err )
        rc_log.critical( msg )
        raise MobyleError ,msg

    userEmail = fromFather[ 'email']
    if userEmail is not None:
        userEmail = Mobyle.Net.EmailAddress( userEmail  )
        
    child = AsynchronJob( fromFather[ 'commandLine' ] , # string the unix command line
                          fromFather[ 'dirPath' ] ,     # absolute path of the working directory
                          fromFather[ 'serviceName' ] , # string 
                          fromFather[ 'resultsMask'] ,  # 
                          userEmail = userEmail  ,      # Net.EmailAddress to
                          xmlEnv = fromFather[ 'xmlEnv' ] , #a dict
                          )
