"""


This is an adaptation of PiseWorkflow.py by P. Tuffery (2007)

Former PiseWorkflow.py is itself an adaptation of PiseWorkflow.pm
(C. Letondal, S. Thoraval)
by P.Tuffery and C. Letondal

General workflows including multiple input and output are possible.
Compared to the PISE module, differences are mostly in the ClientAPI.

Simple example:

from MobyleWorkflow import *
from Mobyle import ClientAPI
import Bio.Clustalw

# alig = Bio.Clustalw.parse_file(sys.argv[1])          

factory  = PiseFactory()
dnapars  = factory.program('dnapars', infile=alig, print_treefile=True,verbose=True)
drawtree = factory.program('drawtree', verbose=True)

from PiseWorkflow import *
x = PiseWorkflow(verbose=0)
x.addpipe(method=clustalw, tomethod=drawtree, pipetype="phylip_tree")
x.run()

print x

Oct 29 2007, P. Tuffery

from MobyleWorkflow import *
wf = MobyleWorkflow()
t1 = wf.wfTask("clustalw-multialign", infile = "../Example/Client/aln.fst", quicktree="fast",typeseq = "protein", output = "PHYLIP")
t2 = wf.wfTask("protpars")
wf.wfPipe(t1, t2, "aligfile", "infile", verbose = 1)
print wf
wf.run(verbose = 1)


# t2 = wf.wfTask("clustalw-multialign", args = {"quicktree":"fast", "typeseq":"protein", "output":"PHYLIP"})
# t2.data["job"].__dict__
# t1 = wf.tasks[0]

20 Oct. 2007:
Started to convert evertyhing from PISE.
Could launch from t2.data["job"].run() : OK
NOW: must check pipe.

"""

import sys
import string
import os
import random
# import cPickle
import fcntl
import time


from Mobyle import ClientAPI

class Task:
    """
    Class for a single task of the workflow
	fields are  :
	service     : MOBYLE service instance
	taskid      : unique id of the task in the workflow (from 0).
        args        : a dictionnary of arguments specific of the task.

        data        : internal data representation of the Task
	\"job\"     : Mobyle job instance (job.run())
	\"service\" : Mobyle service instance (createService())
	\"id\"      : workflow internal unique id for the task.
	\"args\"    : arguments to pass to the service for this job.

	tomethodids : a list of ids to which the method is piped (output)
	ffrom       : a list of ids pied to the method (input)
	pipetype    : a list of the types of the parameters for each of ffrom
	status      : a string describing the status of the task.
		One of: declared, lauchned, waiting, running, done, error
	error       : error string (if any)

        __str__()   : a method to print the status and error string.
		could be html as well.
    """
    def __init__(self, service = None, taskId = None, tomethodids = None, ffrom= None , pipetype= None, status= "declared", error= "", job= None, **kargs):
        self.data = {}
        self.data ["service"]     = service
        self.data ["id"]          = taskId
        self.data ["args"]        = kargs
        if tomethodids == None:
            self.data ["tto"] = []
        else:
            self.data ["tto"] = tomethodids
        if ffrom == None:
            self.data ["ffrom"] = []
        else:
            self.data ["ffrom"] = ffrom
        self.data ["job"]         = None
        self.data ["jobid"]       = None

	# for print purpose 
	self.__status = ""
	self.__error = ""

        if service != None:
            print service.name, kargs
            self.data ["job"] = service.createJob(**kargs) # ClientJob instance
            
    def taskid(self,value = None):
        if value == None:
            return self.data ["id"]
        else:
            self.data ["id"]      = value
    
    def setTo(self,value = None):
        if value == None:
            return self.data ["tto"]
        else:
            self.data ["tto"]      = value
    
    def addTo(self,value = None):
        if value == None:
            return self.data ["tto"]
        else:
            self.data ["tto"].append( value )
            
    def setFrom(self,value = None):
        if value == None:
            return self.data ["ffrom"]
        else:
            self.data ["ffrom"]      = value
     
    def addFrom(self,value = None):
        if value == None:
            return self.data ["ffrom"]
        else:
            self.data ["ffrom"].append( value )
            
    def job(self):
        return self.data ["job"]

    def _print(self, status, error):
	self.__status = status
	self.__error = error
	return self.__str__()

    def __str__(self):
	s = ''
        s += ("task %s " % self.data["id"]) + "\n"
        s += ("ffrom       : %s" % self.data["ffrom"]) + "\n"
        s +=  ("tomethodids : %s" % self.data["tto"]) + "\n"
        s += ("status      : %s" % self.__status) + "\n"
        s += ("error       : %s" % self.__error) + "\n"
        s += ("job         : %s" % self.data["job"]) + "\n"
        s += ("jobid       : %s" % self.data["jobid"]) + "\n"
        s += "--------\n"
	self.__status = ""
	self.__error = ""
 	return s
       
class MobyleWorkflow:
    """
    class MobyleWorkflow:
    Relies on ClientAPI.
    internal dictionnary:
    services: programs member of the workflow (generic so far)
    tasks   : individual calls to services (might exist several calls to identical services)
              i.e. a list of the MOBYLE tasks, the way their are chained, etc
    status: the status and error message for each task
    """
    def __init__(self, service = None, email = None, location = None, verbose = 0, **args):
        """
        MobyleWorkflow(service = None, email = None, location = None, verbose = 0)
        Will create a ClientFactory
        If service != None: will instantiate the service class.


        @param service : a service to initialize the workflow (not mandatory)
        @type : string
        
        @param args : 
        @param email : email address to propagate to the ClientFactory
        @type email  : string

        @param location: location to propagate to the ClientFactory
        @type location: string ??

        """
        self.factory    = ClientAPI.ClientFactory(email=email, location = location, verbose = verbose)
        self.services   = None
        self.ntasks     = 0
        self.tasks      = []
        self.status     = []
        self.error      = []
        self.interval   = 0.01
        self.verbose    = verbose
	self.parent     = True
        self.level      = 0
        key             = int(random.random()*100000)
        self.statusFile = ("/tmp/%d.mwf" % key)
        sys.stderr.write("MobyleWorkflow: statusFile file is  %s\n" % self.statusFile)

        if service != None:
            t = self.wfTask(service, location, **args)
##             self.services = {service : self.factory.createService(service)}
##             task = Task(service = self.services[service], taskId = self.ntasks, args = args)
##             self.ntasks += 1
##             self.tasks.append(task)
##             # self.methods.append(method)
##             self.status.append("declared")
##             self.error.append("")

##     def _method(self, method = None):
##         """
##         _method
##         This one scans for the registered methods and returns the one
##         (if any) that corresponds to the 'method'.
##         method must be a PiseApplication, but this _method will
##         work with strings as well for debug.
##         """
##         if method == None:
##             return None
##         for task in self.tasks:
##             if task.method() == method:
##                 return task
##         return None

    def __repr__(self):
        out = ""
        
        if self.services != None:
            out += "Services: \n"
            for service in self.services.keys():
                out += "  %s\n" % service
        else:
            out += "Services: None\n"

        if self.tasks != []:
            out += "Tasks: \n"

            for task in self.tasks:
                out += "  %d %s \n" % (task.data["id"], task.data["service"].name)
                for i in range(0,len(task.data["ffrom"])):
                    id = task.data["ffrom"][i]["id"]
                    out += "    <- %d %s %s\n" % (id, self.tasks[id].data["service"].name, task.data["ffrom"][i]["out"])

                for i in range(0,len(task.data["tto"])):
                    id = task.data["tto"][i]["id"]
                    out += "    -> %d %s %s\n" % (id, self.tasks[id].data["service"].name, task.data["tto"][i]["in"])
        else:
            out += "Tasks: None\n"

        html = False
        # 1. We read the content
        try:
            self._statusread()
        except:
            pass
	s = "======================\n"
        s += "Complete worklow tasks status:\n\n"
        # 2. We print the information
        for rank in range(0,len(self.tasks)):
            if not html:
                s += self.tasks[rank]._print(self.status[rank],self.error[rank]) + "\n"
	s += "======================\n"
	

                
        return out+s
    
    def wfTask(self, service = None, location = None, **kargs):
        """
        wfTask(service = None, **args)
        create instance of the service. (also register service in factory if not).
        
        @param service: the class of the service (e.g. clustalw, toppred)
        @type service : string

        @param location: the location where the jobs will be runned
        @type location : None or an url ?
        
        @param args   : the args for this particular job instance
        @type args    : a dictionnary

        id = 0
        wf.tasks[id].data["job"].run(infile="/data/aw3/Mobyle/Example/Client/aln.fst")
        """
        if (self.services == None):
            self.services = {service : self.factory.createService(service)}
        elif (service not in self.services.keys()):
            self.services[service] = self.factory.createService(service)
        task = Task(service = self.services[service], taskId = self.ntasks, **kargs)
        self.ntasks += 1
        self.tasks.append(task)
        # self.methods.append(method)
        self.status.append("declared")
        self.error.append("")
        
        self._statuswrite()
        
        return task

    def wfPipe(self,fromTask = None, toTask = None, outParam = None, inParam = None, verbose = 0):
        """
        wfPipe(fromTask = None, toTAsk = None, outParam = None, inParam = None, verbose = 0)
        Defines a pipe between two tasks. The data passed is from outParam (task1) to inParam (task2) pipetype

        @param fromTask, toTask: MobyleWorkflow tasks.
        @type  fromTask, toTask: Task
        
        @param outParam, inParam: parameters to connect
        @type outParam, inParam: string
        the verbose flag may help debug if addpipe fails (0 -> 1)

        Usage:
        wfPipe(task1, task2, "aligfile", "infile")

        return: True if everything OK (instances, params)
                False otherwise

        toTask.data["service"].parameters.keys()
        """
        
        if (fromTask == None) or (toTask == None):
            if verbose:
                sys.stderr.write("%s\n" % "wfPipe: undefined task. wfPipe requires 2 valid Task instances.")
            return False
        
        try:
            p = fromTask.data["job"].service.parameters.keys()
        except:
            if verbose:
                sys.stderr.write("%s\n" % "wfPipe: invalid fromTask. ")
            return False
        
        try:
            p = toTask.data["job"].service.parameters.keys()
        except:
            if verbose:
                sys.stderr.write("%s\n" % "wfPipe: invalid toTask. ")
            return False

        if outParam not in fromTask.data["service"].parameters.keys():
            if verbose:
                sys.stderr.write("wfPipe: invalid parameter %s for fromTask. " % outParam)
            return False
        if inParam not in toTask.data["service"].parameters.keys():
            if verbose:
                sys.stderr.write("wfPipe: invalid parameter %s for toTask. " % inParam)
            return False
        toTask.data["ffrom"].append( {"id" : fromTask.taskid(), "out": outParam, "in": inParam} )
        fromTask.data["tto"].append( {"id" : toTask.taskid(), "out": outParam, "in": inParam} )
        
        return True


    def run(self, interval = 10, statusFile = None, html = False, verbose = 0):
        """
        run(interval = 10, statusFile = None, html = False, verbose = 0)
        
        run the complete workflow
        Workflows MUST be run by calling run.
        interval : seconds slept when waiting job completion. Can be 0.001 as well
        statusFile : if specified, this file will be employed as current status log file. Else, a xxx.pws file is created, where xxx is a random integer between 0 and 100000.
        html     : left for pm compatibility. Not used yet.
        verbose  : some debugging facility
        """
        
        self.interval = int(interval)
        self.verbose = verbose
        self.html = html
        if statusFile != None:
            self.statusFile = statusFile
        
        if html != None:
            self.html = html
            try:
                f = open(self.html)
                f.close()
            except:
                self.html = None

        if self.statusFile == None:
            key = int(random.random()*100000)
##            if self.html == None:
##                 self.html = ("jobs_%5d" % key)
##                 f = open(self.html,"w")
##                 f.close()
##                 sys.stderr.write("MobyleWorkflow/run: Jobs\'s informations will be written and updated in file %s\n" % self.html)
            self.statusFile = ("/tmp/%d.mwf" % key)
            sys.stderr.write("MobyleWorkflow/run: statusFile file is  %s\n" % self.statusFile)

        # before we start, we MUST write the tasks to the statusFile file
        # We reset all the tasks (if mutliple runs, but unmanaged yet)
        for i in range(0,len(self.tasks)):
            self.status[i] = "declared"
        self._statuswrite()

        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/run: initial states:"
            print >>sys.stderr, self
        

        self._launch() # here the run really starts

        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/run: final workflow status:"
            print >>sys.stderr, self

    def _statuswrite(self):
        try: 
            f = open(self.statusFile, "w")
            fcntl.flock(f.fileno(), fcntl.LOCK_EX)
            for i in range(0,len(self.tasks)):
                f.write("%d %s \"%s\" \"%s\" \"%s\"\n" % (self.tasks[i].taskid(),self.status[i],self.error[i],self.tasks[i].data["job"], self.tasks[i].data["jobid"]))
            f.flush()
            fcntl.flock(f.fileno(), fcntl.LOCK_UN)
            f.close()
        except:
            return


    def _statusread(self, verbose = 0):
        try: 
            f = open(self.statusFile, "r")
            try:
                fcntl.flock(f.fileno(), fcntl.LOCK_EX)
            except IOError, (error, message):
                if verbose:
                    print >>sys.stderr, "MobyleWorkflow/statusread:  (parent ? ", self.parent, ")", self.statusFile, " fileno: ", f.fileno()
                    print >>sys.stderr, "MobyleWorkflow/statusread: exception during flock()", message
            if verbose: 
                print >>sys.stderr, "MobyleWorkflow/_statusread:"
            lines = f.readlines()
            for i in range(0,len(self.tasks)):
                if verbose: 
                    print >>sys.stderr, "MobyleWorkflow/_statusread: lines[i]=", lines[i]
                s  = string.split(lines[i],"\"")
                if verbose: 
                    print >>sys.stderr, "MobyleWorkflow/_statusread: s=", s
                it = string.split(s[0])
                self.status[i] = it[1]
                self.error[i]  = s[1]
                self.tasks[i].jobid = s[5]
                print "_statusread : ", lines[i]
                print "_statusread : ", s
                print "_statusread : ", s[5]
                # self.tasks[i].job()
                if verbose: 
                    print >>sys.stderr, "MobyleWorkflow/_statusread: t1[1]=%s s[1]=%s s[3] =%s" % (it[1],s[1],s[3])
                    # sys.exit(0)

            fcntl.flock(f.fileno(), fcntl.LOCK_UN)
            f.close()
        except Exception, e:
            print >>sys.stderr, "MobyleWorkflow/_statusread: exception during open: ", e


    def _statusupdate(self, rank, verbose = 0):
        """
        rank is the rank of the task (current task) to update in the status statusFile file
        """
        
        # 1. We read the content
        f = open(self.statusFile, "r+")
        fcntl.flock(f.fileno(), fcntl.LOCK_EX)

        lines  = f.readlines()
        status = []
        error  = []
	job    = []
        jobid  = []
        for i in range(0,len(self.tasks)):
            s  = string.split(lines[i],"\"")
            it = string.split(s[0])
            status.append(it[1])
            error.append(s[1])
	    job.append(s[3])
            jobid.append(s[4])

        # 2. We update it
        status[rank] = self.status[rank]
        error[rank]  = self.error[rank]
        job[rank]    = self.tasks[rank].data["job"]
        jobid[rank]  = self.tasks[rank].data["jobid"]
        if verbose:
            print >>sys.stderr, "MobyleWorkflow/_statusupdate: status job=", self.status[rank], job[rank]
        
        # 3. we write it back
        f.seek(0)
        for i in range(0,len(self.tasks)):
            f.write("%d %s \"%s\" \"%s\" \"%s\"\n" % (self.tasks[i].taskid(),status[i],error[i],job[i],jobid[i]))
        
        # 4. unlock, close file
        f.flush() # make sure we have updated the file
        fcntl.flock(f.fileno(), fcntl.LOCK_UN)
        f.close()
        if verbose:
            print >>sys.stderr, "MobyleWorkflow/_statusupdate: done"

    def _launch(self, methodids = [0]):
        """
        Recursive travel among the graph of the workflow.
        This one makes the forks.
        From PiseWorkflow.pm _init -> launch to avoid confusion here.
        """

        self.level += 1
        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/_launch: considering :",methodids, "lauch level ",self.level
            
        children  = 0 # Obsolete since we have a list of children pids (next line)
        childpids = []

        # We process a list of methods
        for methodid in methodids:

            if self.status[methodid] != "declared":
                continue

            # We make sure all parents are completed
            self._statusread()
            if self.verbose:
                print >>sys.stderr, "MobyleWorkflow/_launch: considering methodid :",methodid

            # for parent in self.tasks[methodid].ffrom():
            parents = self.tasks[methodid].setFrom()
            print >>sys.stderr, "MobyleWorkflow/_launch: parents : ",parents

##             """
##             All the parent tasks must be completed before we start the present one
##             Note: this manages multiple pipes as well!
##             """
            for i in range(0,len(parents)):
                parent  = parents[i]["id"]
                pipeout = parents[i]["out"]
                pipein  = parents[i]["in"]
                if self.status[parent] != "done":
                    self.status[methodid] = "waiting"
                    self._statusupdate(methodid)
                    if self.verbose:
                        print >>sys.stderr, "MobyleWorkflow/_launch ",methodid,": will wait for ", parent,">>>>"
                    pstatus = self._wait(parent)
                    if self.verbose:
                        print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": completed wait for ", parent," got status",pstatus,"<<<<"
                    if pstatus != "ok":
                        self.status[methodid] = "error"
                        self.error[methodid] = "broken pipe"
                        self._statusupdate(methodid)
                        if self.verbose:
                            print >>sys.stderr, "MobyleWorkflow/_launch: ERROR BROKEN PIPE"
                        self.level -= 1
                        return "waiterror"
                self._pipe(parent, methodid, pipeout, pipein)
                if self.verbose:
                    print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": pipe ",parent, "to", methodid,"type",pipeout, "->", pipein

            # We run the process
            try:
                childpid = os.fork()
            except:
                # This includes a raise SystemExit
                if self.verbose:
                    sys.stderr.write("Could not fork. Workflow execution broken.")
                self.level -= 1
                return "forkerror"

            """
            Fork is successful. We manage processes.
            """
            if childpid==0:
                # child
		self.parent = False
                os.getpid()
                if self.verbose:
                    print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": child=", children , childpids

                # we close the file descriptors
                try:
                    maxfd = os.sysconf("SC_OPEN_MAX")
                except (AttributeError, ValueError):
                    maxfd = 256       # default maximum
        
                for fd in range(4, maxfd):
                    try:
                        os.close(fd)
                    except OSError:   # ERROR (ignore)
                        pass

                if self.verbose:
                    print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": will submit"
                if self._submit(methodid) == "ok":
                    if self.verbose:
                        print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": returning from submitting the task, OK"
                    sys.exit(0)
                else:
                    if self.verbose:
                        print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": some error occurred when submitting the task"
                    sys.exit(1) # some error occurred
            else:
                # parent
                childpids.append([childpid, methodid])
                children += 1
                if self.verbose:
                    print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": parent=", children , childpids

        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/_launch: status :"
            print >>sys.stderr, self
            print >>sys.stderr, "MobyleWorkflow/_launch",methodids,": will consider children :",childpids
            print >>sys.stderr, "MobyleWorkflow/_launch",methodids,": children , childpids: ", children , childpids
        # Here, we recursively process a list of method children

        for achild in childpids:
            os.wait() # wait until the child is terminated

            if self.verbose:
                print >>sys.stderr, "MobyleWorkflow/_launch: child.tomethodids is : ",self.tasks[achild[1]].setTo()
                
            if self.status[achild[1]] == "error":
                if self.tasks[achild[1]].setTo()  != []:
                    for aPipe in self.tasks[achild[1]]["tto"]:
                        id = aPipe["id"]
                        self.status[id] = "error"
                        self.status[id] = "broken pipe!"
            elif self.tasks[achild[1]].setTo()  != []:
                # we care for childs waiting: do not lauch !
                childlist = []
                for aPipe in self.tasks[achild[1]].setTo():
                    rank = aPipe["id"]
                    if self.verbose:
                        print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": child is ",rank,"status:",self.status[rank]
                    if self.status[rank] == "waiting":
                        pass
                    else:
                        childlist.append(rank)
                if self.verbose:
                    print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": will launch childlist:",childlist
                self._launch(childlist)
            else:
                self.level -= 1
                print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": completed!"
                return "ok"
        self.level -= 1
        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/_launch",methodid,": completed!"
        return "ok"

    def _wait(self, rank):
        """
        possible status values are:
        declared, launched, waiting, running, done, error
        """
        self._statusread()

        if self.verbose:
            print >>sys.stderr, ">>>>>>>>>> MobyleWorkflow/entering _wait for ",rank, "status:",self.status[rank]

        # the task is not even launched: we trigger its run
        # (made through recursive call backwards)
        ostatus = "ok"
        if self.status[rank] == "declared":
            if self.verbose:
                print >>sys.stderr, ">>>>>>>>>> MobyleWorkflow/_wait for ",rank, "will wait for parents"
            if self.tasks[rank].ffrom() != []:
                # run the parents
                for parent in self.tasks[rank].ffrom():
                    if self.status[parent] == "error":
                        self.status[rank] = "error"
                        ostatus = "error"
                        break
                    elif self.status[parent] != "done":
                        if self.verbose:
                            print >>sys.stderr, "MobyleWorkflow/_wait for ",rank, ": will wait_wait for",parent,"from ", rank
                        self.status[rank] = "waiting"
                        self._statusupdate(rank)
                        self._wait(parent)
                self.status[rank] = "declared"
                self._statusupdate(rank)
##                 self._statusread()
##                 if self.status[rank] == "declared":
##                     ostatus = self._launch([rank])
##             else:
##                 # run the task directly: todo
##                 ostatus =  self._launch([rank])
            if self.verbose:
                print >>sys.stderr, ">>>>>>>>>MobyleWorkflow/_wait for ",rank, ": finished wait_wait for parents"
            ostatus =  self._launch([rank])
            
        # There was a problem
        elif self.status[rank] == "error":
            if self.verbose:
                print >>sys.stderr, ">>>>>> MobyleWorkflow/_wait for ",rank, ": stopping wait with status:",self.status[rank]
            ostatus = "error"

        # It is running: we wait
	if ostatus == "error":
		pass
        else:
            while(self.status[rank] not in  ["done", "error"]):
                time.sleep(1)
                self._statusread()
            if self.verbose:
                print >>sys.stderr, ">>>>>> MobyleWorkflow/_wait for ",rank, ": awaiking wait with status:",self.status[rank]
            if self.status[rank] == "error":
                ostatus = "error"
        if self.verbose:
            print >>sys.stderr, ">>>>>> MobyleWorkflow/_wait for ",rank, ": exiting wait with task status:",self.status[rank], "wait exit status",ostatus
        return ostatus
            
    
    def _submit(self, rank, **args):
        """
        submit one job
        wait until it is finished
        """
##         import time

        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/_submit: ************** _submit: running task",rank
        
        theJob = self.tasks[rank].data["job"]
        self.status[rank] = "launched"
        self._statusupdate(rank)
        if self.verbose:
            print >>sys.stderr, self

        # The next line is for debug (graph walk) but not so correct ... (status etc)
        # return "ok"

        print theJob, theJob.__class__
        # theJob.run(**args)
        theJob.run(**args)
        try:
            self.tasks[rank].data["jobid"] =  theJob.jobid()
            self._statusupdate(rank)
        except:
            pass
##         try:
##             theJob = theJob.run()
##             if self.verbose:
##                 print >>sys.stderr, "MobyleWorkflow/_submit: ************** _submit: task ",rank,"jobid=", theJob.jobid()
##             #self.tasks[rank].job(theJob.jobid())
##             # self._statusupdate(rank)
##             if self.verbose:
##                 print >>sys.stderr, "MobyleWorkflow/_submit: updated, will print"
##                 print >>sys.stderr, self
##         except:
##             if self.verbose:
##                 print >>sys.stderr, "MobyleWorkflow/_submit: task ",rank,"========> Could not sumbit job!"
##             self.status[rank] = "error"
##             self.error[rank]  = "Could not sumbit job!"
##             self._statusupdate(rank)
##             return "error"

##         self.status[rank] = "running"
##         self._statusupdate(rank)
##         if self.verbose:
##             print >>sys.stderr, self

##         theJob.waitFor()


        if theJob.error(): # some problem occurred starting the application
            if self.verbose:
                print >>sys.stderr, "MobyleWorkflow/_submit: task ",rank,"========> some problem occurred starting the application task ", theJob.error_message()
            self.status[rank] = "error"
            self.error[rank]  = theJob.error_message()
            if self.verbose:
                print >>sys.stderr, "MobyleWorkflow/_submit: error_message=", theJob.error_message()
            self._statusupdate(rank)
            return "error"

        # wait for the application to complete
##         if not theJob.terminated():
##             theJob.results_type("url")
##             while not theJob.terminated():
##                 time.sleep(self.interval)

        if self.verbose:
            print >>sys.stderr, "MobyleWorkflow/_submit: **************  task ",rank," ========> The jobs has completed !"
        self.status[rank] = "done"
        self._statusupdate(rank)
        if self.verbose:
            print >>sys.stderr, self
            print >>sys.stderr, "MobyleWorkflow/_submit: **************  task ",rank," ============> exiting for task ",rank
        
##         del time
        return "ok"

    def _pipe(self, parentid, currentid, outP, inP):
        """
        We pipe methods ffrom and currentid, for parameters outP -> inP
        """
        # return
        job1 = self.tasks[parentid].data["job"]
        self._statusread()
        job1.jobState(self.tasks[parentid].data["jobid"])
        job2 = self.tasks[currentid].data["job"]
        print "_pipe: %d %d %s %s" % (parentid, currentid, outP, inP)
        print job1.__class__
        print job2.__class__
        print job1.__dict__
        if self.verbose: 
            print >>sys.stderr, "MobyleWorkflow/_pipe: xxxxxxxxxxxxxxx _pipe %s %d %s ==> %s %d %s\n" % (self.tasks[parentid].data["service"].name, parentid, outP, self.tasks[currentid].data["service"].name, currentid, inP)


        self.tasks[currentid].data["job"].makePipe(self.tasks[parentid].data["job"], outParamName = outP, inParamName = inP)

##     def __str__(self):
##         """
##         Some visualisation of the tasks.
##         """
##         html = False
##         # 1. We read the content
##         self._statusread()
        
## ##         f = open(self.statusFile, "r")
## ##         fcntl.flock(f.fileno(), fcntl.LOCK_EX)
## ##         u = cPickle.Unpickler(f)
## ##         odefs = u.load()
        
## ##         fcntl.flock(f.fileno(), fcntl.LOCK_UN)
## ##         f.close()

## 	s = "======================\n"
##         s += "Complete worklow tasks status:\n\n"
##         # 2. We print the information
##         for rank in range(0,len(self.tasks)):
##             if not html:
##                 s += self.tasks[rank]._print(self.status[rank],self.error[rank]) + "\n"
## 	s += "======================\n"
## 	return s

	
