#############################################################
#                                                           #
#   Author: Herve Menager                                   #
#   Organization:'Biological Software and Databases' Group, #
#                Institut Pasteur, Paris.                   #
#   Distributed under GPLv2 Licence. Please refer to the    #
#   COPYING.LIB document.                                   #
#                                                           #
#############################################################
"""
Validator.py

This module is used to validate a Mobyle XML file (program or else?)
against the Relax NG schema and the schematron rules
- The Validator class
"""
import os, sys, StringIO
import libxml2
from Ft.Xml.Xslt.Processor import Processor
from Ft.Xml.InputSource import DefaultFactory
from Ft.Xml.Domlette import NonvalidatingReader
from Ft.Lib.Uri import OsPathToUri
from Ft.Xml.XPath import Compile, Evaluate
from Ft.Xml.XPath.Context import Context
from Ft.Xml.Domlette import NoExtDtdReader
from Mobyle import ConfigManager

_cfg = ConfigManager.Config()

# defining paths for validation external stuff
mobrngPath = os.path.join(_cfg.mobylehome(),'mobyle.rng')
mobschPath = os.path.join(_cfg.mobylehome(),'mobyle.sch')
schprocPath = os.path.join(_cfg.mobylehome(),'Tools','validation','iso_svrl.xsl')
rbPath = os.path.join(_cfg.mobylehome(),'Tools','validation','remove_base.xsl')

# pre-validation processing
# generating relax NG validator
rng_xml = libxml2.parseFile(mobrngPath)
rngp = rng_xml.relaxNGNewDocParserCtxt()
rngs = rngp.relaxNGParse()
mobrngVal = rngs.relaxNGNewValidCtxt()
# hack:xsl pre-processing between xinclude and relax-NG validation to
# compensate the lack of support for xml:base in libxml2/relax-NG
xpRb = Processor()
xpRb.setDocumentReader(NoExtDtdReader)
xpRb.appendStylesheet(DefaultFactory.fromUri(OsPathToUri(rbPath)))  
# generating schematron validation xsl from its xsl generator
xsltproc = Processor()
xsltproc.appendStylesheet(DefaultFactory.fromUri(OsPathToUri(schprocPath)))
xsltproc.setDocumentReader(NoExtDtdReader)
mobschValIS = xsltproc.run(DefaultFactory.fromUri(OsPathToUri(mobschPath)))

# this is additional cleanup code executed on unload.
def _clean():
    del sys.modules[__name__].rngp
    del sys.modules[__name__].mobrngVal
    del sys.modules[__name__].rngs
import atexit
atexit.register(_clean)

class Validator(object):
    
    def __init__(self, docPath, publicURI=None,runRNG=True, runSCH=True):
        self.docPath = docPath
        if publicURI:
            self.publicURI = publicURI
        else:
            self.publicURI = self.docPath
        self.runRNG = runRNG
        self.runSCH = runSCH
        if self.runRNG:
            self.rngSrc = libxml2.readFile(docPath,None,libxml2.XML_PARSE_NOENT)
            self.rngSrc.xincludeProcess()
            rbProcRes = xpRb.run(DefaultFactory.fromString(str(self.rngSrc),OsPathToUri(docPath)))
            self.rngSrc = libxml2.parseDoc(rbProcRes)
        if self.runSCH:
            self.schSrc = DefaultFactory.fromUri(OsPathToUri(docPath))

    def run(self):
        if self.runRNG:
            self.rngErrors = []
            #desactivate error messages from the validation
            def noerr(ctx, str):
                self.rngErrors.append(str.strip())
                pass
            libxml2.registerErrorHandler(noerr, None)
            self.rngOk = not(bool(self.rngSrc.relaxNGValidateDoc(mobrngVal)))

        if self.runSCH:
            xsltproc = Processor()
            xsltproc.setDocumentReader(NoExtDtdReader)
            xsltproc.appendStylesheet(DefaultFactory.fromString(mobschValIS,OsPathToUri(mobschPath)))
            schValReport = xsltproc.run(self.schSrc,topLevelParams={'fileNameParameter':os.path.basename(self.publicURI)})
            resultDoc = NonvalidatingReader.parseString(schValReport,"urn:bogus")
            nss = {u'svrl': u'http://purl.oclc.org/dsdl/svrl'}
            failedAsserts = Evaluate(u'//svrl:failed-assert', Context(resultDoc, processorNss=nss))
            self.schErrors = []
            for failedAssert in failedAsserts:
                self.schErrors.append(str(Evaluate(u'svrl:text/text()', Context(failedAssert, processorNss=nss))[0].nodeValue).strip())
        self.valOk = not(self.runRNG and not(self.rngOk)) and not(self.runSCH and len(self.schErrors)>0)
