########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under LGPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################


import logging , logging.handlers
from string import upper

import sys , os , os.path
import re
import glob , copy 
import types


#append Mobyle Home to the search modules path

MOBYLEHOME = None
if os.environ.has_key('MOBYLEHOME'):
    MOBYLEHOME = os.environ['MOBYLEHOME']
if not MOBYLEHOME:
    sys.exit('MOBYLEHOME must be defined in your environment')

if ( MOBYLEHOME ) not in sys.path:
    sys.path.append( MOBYLEHOME )
if ( os.path.join( MOBYLEHOME , 'Src' ) ) not in sys.path:    
    sys.path.append( os.path.join( MOBYLEHOME , 'Src' ) )

import Local.Config.Config

import Mobyle.MobyleError 

MOBYLEHTDOCS = None

class Config( object ):
    """
    this class is designed as the singleton pattern
    (ref: Programmation Python , Tarek Ziade .p 483).
    there is only one instance at once.
    this class parse the file local/Config/Config.py
    raise error fix default values raise error if needed. 
    all other Mobyle classes must use this class to access
    to a configuration information. 
    """
    _ref = None
    
    def __new__( self ):
        
        if self._ref is None:
            self._ref = super( Config,self ).__new__( self )

            #############################
            #
            # logging
            #
            ##############################
            
            try:
                self._logdir = Local.Config.Config.LOGDIR
            except AttributeError:
                self._logdir = "/dev/null"
                msg = "LOGDIR not found in  Local/Config/Config.py"
                print >> sys.stderr,  "%s . It sets to %s" %( msg , self._logdir ) 

            self.log = logging.getLogger('Mobyle.Config')
            self.log.propagate = False
            try:
                defaultHandler = logging.FileHandler( os.path.join( self._logdir , 'error_log' ) , 'a')
            except :
                print >> sys.stderr , " WARNING : can't access to logfile. Logs will redirect to /dev/null"
                defaultHandler = logging.FileHandler( '/dev/null' , 'a' )

            defaultFormatter = logging.Formatter(
                '%(name)-10s : %(levelname)-8s : L %(lineno)d : %(asctime)s : %(message)s' ,
                '%a, %d %b %Y %H:%M:%S'
                )

            defaultHandler.setLevel( logging.DEBUG )
            defaultHandler.setFormatter( defaultFormatter )
            self.log.addHandler( defaultHandler )

            if  os.path.exists( MOBYLEHOME ):
                self._mobylehome = os.path.realpath( MOBYLEHOME )
            else:
                raise MobyleError , "MOBYLEHOME is not defined"
  
            ######################
            #
            # Mandatory values
            #
            ######################

            try: #old style config
                if os.path.exists( Local.Config.Config.DOCUMENT_ROOT ):
                    self._mobyle_htdocs = os.path.realpath( Local.Config.Config.DOCUMENT_ROOT )
                else:
                    msg = "DOCUMENT_ROOT: %s does not exist" % Local.Config.Config.DOCUMENT_ROOT
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
            except AttributeError: 
                if MOBYLEHTDOCS: #define by new installer
                    if os.path.exists( MOBYLEHTDOCS ):
                        self._mobyle_htdocs = os.path.realpath( MOBYLEHTDOCS )
                    else: #never happen ?
                        msg = "bad installation option --install-htdocs : %s no such directory " %MOBYLEHTDOCS
                        raise Mobyle.MobyleError.ConfigError , msg
                else:
                    try: #define in Config.py witout installer but with new style config
                        if os.path.exists( Local.Config.Config.MOBYLEHTDOCS ):
                            self._mobyle_htdocs = os.path.realpath( Local.Config.Config.MOBYLEHTDOCS )
                        else:
                            msg= "MOBYLEHTDOCS: %s no such directory" % Local.Config.Config.MOBYLEHTDOCS
                            self.log.error( msg )
                            raise Mobyle.MobyleError.ConfigError , msg
                    except AttributeError:
                        msg= "error during installation or configuration MOBYLEHTDOCS not found"
                        self.log.error( msg )
                        raise Mobyle.MobyleError.ConfigError , msg
                
            try:
                self._root_url = Local.Config.Config.ROOT_URL.strip( '/' )
            except AttributeError:
                msg = "ROOT_URL not found in Local/Config/Config.py"
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg
            
            try:
                self._mobyleroot_cgi_url = Local.Config.Config.MOBYLEROOT_CGI_URL.strip( '/' )
            except AttributeError:
                self._mobyleroot_cgi_url = None
                self.log.info( "MOBYLEROOT_CGI_URL not found in Local/Config/Config.py" )
            
            try:
                self._mobyleroot_htdocs_url = Local.Config.Config.MOBYLEROOT_HTDOCS_URL.strip( '/' )
            except AttributeError:
                self._mobyleroot_htdocs_url = None
                self.log.info( "MOBYLEROOT_HTDOCS_URL not found in Local/Config/Config.py" )

            try:
                self._cgi_prefix = Local.Config.Config.CGI_PREFIX.strip( '/' )
            except AttributeError:
                if self._mobyleroot_cgi_url is None :
                    msg = "CGI_PREFIX not found in Local/Config/Config.py"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
                else:
                    self._cgi_prefix = None
            try:
                self._htdocs_prefix = Local.Config.Config.HTDOCS_PREFIX.strip( '/' )
            except AttributeError:
                if self._mobyleroot_htdocs_url is None:
                    msg = "HTDOCS_PREFIX not found in Local/Config/Config.py" 
                    self.log.error( "HTDOCS_PREFIX not found in Local/Config/Config.py" )
                    raise Mobyle.MobyleError.ConfigError , msg
                else:
                    self._htdocs_prefix = None
                    
            if  self._htdocs_prefix is not None :
                 if self._mobyleroot_htdocs_url is not None :
                     msg = "HTDOCS_PREFIX and MOBYLEROOT_HTDOCS_URL are defined in same time, you must defined only one"
                     self.log.error( msg )
                     raise Mobyle.MobyleError.ConfigError , msg
                 else:
                     self._portal_prefix = self._htdocs_prefix + '/portal'
                     self._portal_prefix = self._portal_prefix.lstrip( '/' )
                     self._portal_path = os.path.realpath( os.path.join( self._mobyle_htdocs , 'portal' ) )
            else:
                self._portal_prefix = self._mobyleroot_htdocs_url
                self._portal_path = os.path.realpath( os.path.join( self._mobyle_htdocs , self._mobyleroot_htdocs_url ) )

            
            if self._cgi_prefix is not None :
                 if self._mobyleroot_cgi_url is not None :
                     msg = "CGI_PREFIX and MOBYLEROOT_CGI_URL are defined in same time, you must defined only one"
                     self.log.error( msg )
                     raise Mobyle.MobyleError.ConfigError , msg
                 else:
                     pass
            else:
                self._cgi_prefix = self._mobyleroot_cgi_url
                
            ######################
            #
            #  default values
            #
            #######################
            
            self._debug = 0
            self._particular_debug = {}
            
            self._accounting = False
            self._session_debug = False
            
            if self._htdocs_prefix:
                self._programs_url  =  "%s/%s/%s" %( self._root_url , self._htdocs_prefix , 'data/programs' )
                self._results_url = "%s/%s/%s" %( self._root_url , self._htdocs_prefix , 'data/jobs' )
                self._user_sessions_url = "%s/%s/%s" %( self._root_url , self._htdocs_prefix , 'data/sessions' )
            else:
                self._programs_url  =  "%s/%s" %( self._root_url , 'data/programs' )
                self._results_url = "%s/%s" %( self._root_url , 'data/jobs' )
                self._user_sessions_url = "%s/%s" %( self._root_url , 'data/sessions' )
                
            self._programs_path = os.path.realpath( os.path.join( self._mobyle_htdocs , 'data' , 'programs' ) )
            self._programs_imports_path = os.path.realpath( os.path.join( self._programs_path ,'imports' ) )
            self._programs_index_path = os.path.realpath( os.path.join( self._programs_path ,'index' ) )
            self._results_path = os.path.realpath( os.path.join( self._mobyle_htdocs ,'data' , 'jobs' ) )
            self._user_sessions_path = os.path.realpath( os.path.join( self._mobyle_htdocs , 'data' , 'sessions' ) )
            
            self._binary_path = []  
            self._format_detector_cache_path = None
            
            self._databanks_config = {}
            
            self._dns_resolver = False 
            self._opt_email = False
            self._particular_opt_email = {}
            
            self._anonymous_session = "captcha"
            self._authenticated_session = 'email'
            
            self._refresh_frequency = 240
            
            self._dont_email_result = False
            
            self._filelimit = 2147483648     #  2 Gib
            self._sessionlimit = 52428800    # 50 Mib
            self._previewDataLimit = 1048576 #  1 Mib
            
            self._result_remain = 10 # in day
            
            self._lang = 'en'
            self._timeout = 60
            
            self._seqconverter = {}
            
            self._batch = 'Sys'
            self._particular_batch = {}
            
            self._default_Q = 'default_q'
            self._particular_Q = {}
            self._Qproperties = {}
            
            self._programs_deployment_include = ['*']
            self._programs_deployment_exclude = []
            self._programs_deployment_order = [ 'include' , 'exclude' ]
            
            self._disable_all = False
            self._disabled_services =[]
            
            self._authorized_services = {}
            self._all_portals = {}
            self._exported_programs = []
            
            
            try:
                self._accounting = Local.Config.Config.ACCOUNTING
            except AttributeError:
                self.log.info( "ACCOUNTING not found in  Local/Config/Config.py, set ACCOUNTING= %s" %self._accounting )
            try:
                self._session_debug = Local.Config.Config.SESSION_DEBUG
            except AttributeError:
                pass
            ######################
            #
            #  Directories
            #
            ######################
            
            ## in htdocs ##
            
            try:
                if os.path.exists( Local.Config.Config.PROGRAMS_PATH ): 
                    self._programs_path = os.path.realpath( Local.Config.Config.PROGRAMS_PATH )
                else:
                    msg = "PROGRAMS_PATH: %s does not exit" % Local.Config.Config.PROGRAMS_PATH
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
            except AttributeError:
                pass
            
            try:
                if os.path.exists( Local.Config.Config.PROGRAMS_CACHE_PATH):
                    self._programs_imports_path = os.path.realpath( Local.Config.Config.PROGRAMS_CACHE_PATH )
                    self._programs_index_path = os.path.realpath( Local.Config.Config.PROGRAMS_CACHE_PATH )
                else:
                    msg = "PROGRAMS_CACHE_PATH: %s does not exit" % Local.Config.Config.PROGRAMS_CACHE_PATH
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg                    
            except AttributeError:
                pass
            
            try:           
                if os.path.exists( Local.Config.Config.RESULTS_PATH ): 
                    self._results_path = os.path.realpath( Local.Config.Config.RESULTS_PATH )
                else:
                    msg = "RESULTS_PATH: %s does not exit" % Local.Config.Config.RESULTS_PATH
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg                     
            except AttributeError:
                pass
              
            self._admindir =  os.path.join( self._results_path , 'ADMINDIR' )

            try:
                if os.path.exists( Local.Config.Config.USER_SESSIONS_PATH ): 
                    self._user_sessions_path = os.path.realpath( Local.Config.Config.USER_SESSIONS_PATH )
                else:
                    msg = "USER_SESSIONS_PATH: %s does not exit" % Local.Config.Config.USER_SESSIONS_PATH
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg   
            except AttributeError, err:
                self.log.debug( "USER_SESSIONS_PATH not found in Local/Config/Config.py default value used : " +str( self._user_sessions_path) )
 
            ## out web ##
            try:
                binary_path = []
                for path in Local.Config.Config.BINARY_PATH:
                    if not os.path.exists( path ):
                       msg = "the path \"%s\" defined in BINARY_PATH doesn't exist" % path
                       self.log.warning( msg )
                       continue
                    binary_path.append( path )
                self._binary_path  = binary_path
            except AttributeError:
                self.log.warning( "BINARY_PATH not found in Local/Config/Config.py the default web server path will be used" )
                
            try:
                if os.path.exists( Local.Config.Config.FORMAT_DETECTOR_CACHE_PATH ):
                    self._format_detector_cache_path = os.path.realpath( Local.Config.Config.FORMAT_DETECTOR_CACHE_PATH )
                else:
                    msg = "FORMAT_DETECTOR_CACHE_PATH: %s does not exit" % Local.Config.Config.FORMAT_DETECTOR_CACHE_PATH
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
            except AttributeError:
                pass
                
            try:
                self._databanks_config = Local.Config.Config.DATABANKS_CONFIG
            except AttributeError:
                self.log.info( "No databanks are found in Local/Config/Config.py" )
                    
                    
            #######################
            #
            # portal
            #
            #######################
            
            try:
                self._programs_url = "%s/%s" %( self._root_url , 
                                              Local.Config.Config.PROGRAMS_URL.strip( '/' ) 
                                              )
            except AttributeError:
                self.log.debug( "PROGRAMS_URL not found in Local/Config/Config.py" )
                pass
            
            try:
                self._results_url = "%s/%s" %( self._root_url ,
                                             Local.Config.Config.RESULTS_URL.strip( '/' )
                                             )
            except AttributeError:
                self.log.debug( "RESULTS_URL not found in Local/Config/Config.py" )
                pass
            try:
                self._user_sessions_url = "%s/%s" %( self._root_url ,
                                                    Local.Config.Config.SESSIONS_URL.strip( '/' )
                                                    )
            except AttributeError:
                self.log.debug( "SESSIONS_URL not found in Local/Config/Config.py" )
                pass  
            
            #######################
            #
            #    debug
            #
            #######################

            try:
                debug = Local.Config.Config.DEBUG
                if debug >= 0 or debug <= 3:
                    self._debug = debug
                else:
                    self.log.warning( "DEBUG must be >= 0 and <= 3 I found DEBUG =%s, DEBUG is set to %i" % ( debug , self._debug ) ) 
            except AttributeError:
                self.log.info( "DEBUG not found in Local/Config/Config.py, DEBUG is set to %i" % self._debug )

            try:
                self._particular_debug = Local.Config.Config.PARTICULAR_DEBUG
            except AttributeError:
                self.log.info( "PARTICULAR_DEBUG not found in Local/Config/Config.py" )
                                        
            #######################
            #
            #  Mail
            #
            #######################

            try:
                if type( Local.Config.Config.MAINTAINER ) == types.ListType or type( Local.Config.Config.MAINTAINER ) == types.TupleType :
                    self._maintainer = Local.Config.Config.MAINTAINER
                else:
                    self._maintainer = [ Local.Config.Config.MAINTAINER ]
                    
                if type( Local.Config.Config.HELP ) == types.ListType or type( Local.Config.Config.HELP ) == types.TupleType :    
                    self._help = " , ".join( Local.Config.Config.HELP )
                else:
                    self._help = Local.Config.Config.HELP

                self._mailhost = Local.Config.Config.MAILHOST
            except AttributeError, err:
                raise err
                msg = str(err).split()[-1] + "not found in Local/Config/Config.py"
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg         
                
            try:
                self._sender = Local.Config.Config.SENDER
            except AttributeError, err:
                self._sender = self._help
                self.log.info( "SENDER not found in  Local/Config/Config.py , set SENDER to HELP ")
                
            try:
                self._dns_resolver = Local.Config.Config.DNS_RESOLVER
            except AttributeError:
                self.log.info( "DNS_RESOLVER not found in  Local/Config/Config.py, set DNS_RESOLVER to False" )
                
            #######################
            #
            #     Authentication
            #
            #######################

            try:
                self._opt_email = Local.Config.Config.OPT_EMAIL
            except AttributeError:
                self.log.info( "OPT_EMAIL not found in  Local/Config/Config.py, set OPT_EMAIL to %s" % self._opt_email )
                               
            try:
                self._particular_opt_email = Local.Config.Config.PARTICULAR_OPT_EMAIL
            except AttributeError:
                self.log.info( "PARTICULAR_OPT_EMAIL not found in  Local/Config/Config.py, use OPT_EMAIL for all services" )
                
            try:
                self._anonymous_session = Local.Config.Config.ANONYMOUS_SESSION
                try:
                    self._anonymous_session = self._anonymous_session.lower()
                except AttributeError:
                    self._anonymous_session = "captcha"
                    self.log.warning( "ANONYMOUS_SESSION have a wrong value: " + str( Local.Config.Config.ANONYMOUS_SESSION ) + " , set to \"%s\"" %self._anonymous_session)
               
                if self._anonymous_session == "yes" or self._anonymous_session == "y":
                    self._anonymous_session = "yes"
                elif self._anonymous_session == "no" or self._anonymous_session == "n":
                    self._anonymous_session = "no"
                elif self._anonymous_session != "captcha":
                    self._anonymous_session = "captcha"
                    self.log.warning( "ANONYMOUS_SESSION have a wrong value: " + str( Local.Config.Config.ANONYMOUS_SESSION ) + " , set to \"captcha\"" )

            except AttributeError:
                self.log.info( "ANONYMOUS_SESSION not found in  Local/Config/Config.py, set to  default value: \"%s\"" %self._anonymous_session )
            try:
                self._authenticated_session = Local.Config.Config.AUTHENTICATED_SESSION
                    
                if self._authenticated_session not in ( "email" ,"no" ,"yes" ):
                    self.log.warning( "AUTHENTICATED_SESSION have a wrong value: %s set to \"%s\""%( Local.Config.Config.AUTHENTICATED_SESSION ,
                                                                                                      self._authenticated_session ) )
                        
                if self._anonymous_session == "no" and not self._authenticated_session :
                    msg = "anonymous session AND authenticated session are disabled. you can't disabled both"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError , msg
                
            except AttributeError , err :
                self.log.info( "AUTHENTICATED_SESSION not found in  Local/Config/Config.py , set to default value: \"%s\"" %self._authenticated_session )         
                
            try:
                self._refresh_frequency = Local.Config.Config.REFRESH_FREQUENCY
            except AttributeError:
                self.log.info( "REFRESH_FREQUENCY not found in Local/Config/Config.py, set to %s" % self._refresh_frequency )               
                
            ########################
            #
            #      results
            #
            ########################
               
            try:
                self._dont_email_result = Local.Config.Config.DONT_EMAIL_RESULTS
            except AttributeError:
                self.log.info( "DONT_EMAIL_RESULTS not found in  Local/Config/Config.py, set DONT_EMAIL_RESULTS to %s" %self._dont_email_result)

            try:
                self._maxmailsize = Local.Config.Config.MAXMAILSIZE
            except AttributeError:
                if not self._dont_email_result :
                    self.log.info("MAXMAILSIZE not found in  Local/Config/Config.py but DONT_EMAIL_RESULTS is False, I set MAXMAILSIZE = 2 Mo")
                    self._maxmailsize = 2097152 
                else:
                    self._maxmailsize = None

            try:
                self._filelimit = int( Local.Config.Config.FILELIMIT )
            except AttributeError:
                self.log.info( "FILELIMIT not found in  Local/Config/Config.py, set FILELIMIT to %d Gib" %( self._filelimit / 1073741824 ) )
            except ValueError:
                msg = "FILELIMIT have an invalid value : %s .\nIt must be an integer" % self._filelimit 
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg
 
            try:
                self._sessionlimit = int( Local.Config.Config.SESSIONLIMIT )
            except AttributeError:
                self.log.info( "SESSIONLIMIT not found in  Local/Config/Config.py, set SESSIONLIMIT to %d Mib" %( self._sessionlimit / 1048576 ) )
            except ValueError:
                msg = "SESSIONLIMIT have an invalid value : %s .\nIt must be an integer" % self._sessionlimit 
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg

            try:
                self._previewDataLimit = int( Local.Config.Config.PREVIEW_DATA_LIMIT )
            except AttributeError:
                self.log.info( "PREVIEW_DATA_LIMIT not found in  Local/Config/Config.py, set PREVIEW_DATA_LIMIT to %d Mib"%( self._previewDataLimit / 1048576 ) )
            except ValueError:
                msg = "PREVIEW_DATA_LIMIT have an invalid value : %s .\nIt must be an integer" % self._sessionlimit 
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg

            try:
                self._result_remain = int( Local.Config.Config.RESULT_REMAIN )
            except AttributeError:
                self.log.info( "RESULT_REMAIN not found in  Local/Config/Config.py, set RESULT_REMAIN to %d days" %( self._result_remain ))
            except ValueError:
                msg = "RESULT_REMAIN have an invalid value : %s .\nIt must be an integer" % Local.Config.Config.RESULT_REMAIN
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg

            #############################
            #
            #       SEQCONVERTER
            #
            #############################
            
            try:
                for conv in Local.Config.Config.SEQCONVERTER:
                    convUp = conv.upper()
                    if convUp not in ( 'SQUIZZ', 'READSEQ' ):
                        msg = self.log.error( conv + " is not an allowed value for SEQCONVERTER")
                        self.log.error( msg )
                        raise Mobyle.MobyleError.ConfigError , msg
                    else:
                        converter = Local.Config.Config.SEQCONVERTER[ conv ]
                        if os.path.exists( converter ):
                            self._seqconverter[ convUp ] = Local.Config.Config.SEQCONVERTER[ conv ]
                        else:
                            msg = str( converter ) + " : is not a path toward squizz or readseq"
                            self.log.error( msg )
                            raise Mobyle.MobyleError.ConfigError ,msg
            except AttributeError , err :
                msg = "SEQCONVERTER not found in  Local/Config/Config.py, it must be defined "
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg


            ##############################
            #
            #      Misc
            #
            ##############################


            try:
                self._lang = Local.Config.Config.LANG
            except AttributeError:
                self.log.info( "LANG not found in  Local/Config/Config.py, set LANG to %s" % self._lang)

            try:
                self._timeout = Local.Config.Config.TIMEOUT
            except AttributeError:
                self.log.info( "TIMEOUT not found in  Local/Config/Config.py, set TIMEOUT to %d" % self._timeout)

            ##############################
            #
            #      BATCH
            #
            ##############################
            
            try:
                self._batch = upper( str( Local.Config.Config.BATCH ) )
            except AttributeError:
                msg = "BATCH not found in  Local/Config/Config.py, it is set to %s " % self._batch
                self.log.info( msg )

            if self._batch not in ('SYS', 'SGE', 'PBS'):
                msg = "BATCH can take only SYS , SGE , PBS values"
                self.log.error( msg )
                raise Mobyle.MobyleError.ConfigError , msg            

            try:
                self._particular_batch = Local.Config.Config.PARTICULAR_BATCH
            except AttributeError:
                self.log.info( "PARTICULAR_BATCH not found in  Local/Config/Config.py" )

            try:
                self._default_Q = Local.Config.Config.DEFAULT_Q
            except AttributeError:
                if self._batch != 'SYS':
                    self.log.info( "DEFAULT_Q not found in  Local/Config/Config.py" )

            try:
                self._particular_Q  = Local.Config.Config.PARTICULAR_Q
            except AttributeError:
                self.log.info( "PARTICULAR_Q not found in  Local/Config/Config.py" )

            try:
                self._Qproperties = Local.Config.Config.Q_PROPERTIES
            except AttributeError:
                self.log.info( "Q_PROPERTIES not found in  Local/Config/Config.py" )
                
            try:
                self._sge_root = Local.Config.Config.SGE_ROOT
                self._sge_cell = Local.Config.Config.SGE_CELL

            except AttributeError ,err:
                if self._batch == "SGE" or "SGE" in self._particular_batch.values() :

                    msg = str( err ).split()[ -1 ]+" not found in Local/Config/Config.py\nIf you use SGE queueing system you must defined SGE_ROOT and SGE_CELL"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError ,msg

            try:
                self._pbs_root = Local.Config.Config.PBS_ROOT
                self._pbs_wait_time = Local.Config.Config.DEFAULT_PBS_WAIT_TIME
                self._pbs_priority  = Local.Config.Config.DEFAULT_PBS_PRIORITY

            except AttributeError ,err:
                if self._batch == "PBS" or "PBS" in self._particular_batch.values() :

                    msg = str( err ).split()[ -1 ]+" not found in Local/Config/Config.py\nIf you use PBS queueinq system you must defined PBS_ROOT, DEFAULT_PBS_WAIT_TIME, DEFAULT_PBS_PRIORITY"
                    self.log.error( msg )
                    raise Mobyle.MobyleError.ConfigError ,msg


            ##########################
            #
            # programs publication
            #
            ###########################
            try:
                self._programs_deployment_exclude = Local.Config.Config.LOCAL_DEPLOY_EXCLUDE
            except ( AttributeError , NameError ):
                self.log.info( "LOCAL_DEPLOY_ORDER not found in  Local/Config/Config.py" )
                
            try:
                self._programs_deployment_include = Local.Config.Config.LOCAL_DEPLOY_INCLUDE
            except ( AttributeError , NameError ):
                self.log.info( "LOCAL_DEPLOY_INCLUDE not found in  Local/Config/Config.py" )

            try:
                self._programs_deployment_order = Local.Config.Config.LOCAL_DEPLOY_ORDER
            except ( AttributeError , NameError ):
                self.log.info( "LOCAL_DEPLOY_EXCLUDE not found in  Local/Config/Config.py" )

            #############################
            #
            # disabled service
            #
            #############################

            try:
                self._disable_all = Local.Config.Config.DISABLE_ALL
            except AttributeError:
                msg = "DISABLE_ALL not found in  Local/Config/Config.py,set to %s" %self._disable_all
                self.log.info( msg )
            try:
                self._disabled_services = Local.Config.Config.DISABLED_SERVICES
            except AttributeError:
                msg = "DISABLED_SERVICES not found in  Local/Config/Config.py,"
                self.log.info( msg )
                
            #########################################
            #
            # restriction services access
            #
            #########################################

            try:
                self._authorized_services = Local.Config.Config.AUTHORIZED_SERVICES
            except AttributeError:
                self.log.info( "AUTHORIZED_SERVICES not found in  Local/Config/Config.py" )
            
            #########################################
            #
            # Grid aspects
            #
            #########################################    

            try:
               self._all_portals = Local.Config.Config.PORTALS 
               for portal in self._all_portals .keys():
                   self._all_portals[ portal ][ 'url' ] = self._all_portals[ portal ][ 'url' ].rstrip( '/')
                   self._all_portals[ portal ][ 'repository' ] = self._all_portals[ portal ][ 'repository' ].rstrip( '/')
                   self._all_portals[ portal ][ 'jobsBase' ] = self._all_portals[ portal ][ 'jobsBase' ].rstrip( '/')
            except KeyError , err:
                msg = "error PORTALS  : %s is not properly defined : " %( portal , err )
                self.log.warning( msg )
                try:
                    del self._all_portals[ portal ]
                except KeyError:
                    pass
                self.log.warning( " the portal: %s will not be imported" %portal )
            except AttributeError:
               pass

            try:
              self._exported_programs = Local.Config.Config.EXPORTED_SERVICES
            except AttributeError:
                self.log.info( "EXPORTED_SERVICES not found in  Local/Config/Config.py" )
            
        return self._ref                



    ###############################
    #
    #    accessors
    #
    ##############################

    def debug( self , jobName = None ):
        """
        Returns the debug level for a job or the default debug level if jobName is not specified
        @param jobName:
        @type jobName: string
        @return: an int >=0 and <= 3
        @rtype: int
        """
        try:
            debug = self._particular_debug[ jobName ]
        except KeyError:
            debug = self._debug
        if debug < 0 or debug > 3:
            if jobName is None:
                msg =" DEBUG must be >= 0 and <= 3 I found DEBUG =%i, DEBUG is set to %i" % ( debug , self._debug )
            else:
                msg = jobName + " debug must be >= 0 and <= 3 I found %i,  PARTICULAR_DEBUG[ '%s' ] is set it to %i" % ( debug ,
                                                                                                                         jobName , 
                                                                                                                         self._debug )
            self.log.warning( msg )
        return debug
    
    
    def mobylehome( self ):
        """
        @return: the absolute path to home of the project
        @rtype: string
        """
        return self._mobylehome
    
    
    def document_root( self ):
        """
        @return: the absolute path to the mobyle htdocs directory
        @rtype: string
        """
        return self._mobyle_htdocs
    
    
    def root_url( self ):
        """
        @return: the base url of the web server
        @rtype: string
        """
        return self._root_url


    def results_url( self ):
        """
        @return: the complete url of the jobs
        @rtype: string
        """
        return self._results_url

    def results_path( self ):
        """
        @return: the directory absolute path  where are located the jobs
        @rtype: string
        """
        return self._results_path
    
    def admindir(self):
        """
        @return: the absolute path of the ADMINDIR
        @rtype: string
        """
        return self._admindir
    
    def programs_url( self ):
        """
        @return: the complete url of the programs 
        @rtype: string
        """
        return self._programs_url

    def programs_path( self ):
        """
        @return: the directory absolute path  where are located the programs
        @rtype: string
        """
        return self._programs_path
    
    
    def binary_path( self ):
        """
        @return: the list of path where binaries must be search
        @rtype: list of strings
        """
        return self._binary_path
    
    def programs_imports_path( self ):
        """
        @return: the directory absolute path where are located the imported services
        @rtype: string
        """
        return self._programs_imports_path

    def programs_index_path( self ):
        """
        @return: the directory absolute path  where are located the indexes for this portal.
        @rtype: string
        """
        return self._programs_index_path

    def format_detector_cache_path( self ):
        """
        @return: the absolute path where the invalid sequences and alignment are dumped for analysis
        @rtype: string
        """
        return self._format_detector_cache_path


    def user_sessions_path( self ):
        """
        @return: the directory absolute path  where are located the sessions
        @rtype: string
        """
        return self._user_sessions_path
    
    def user_sessions_url( self ):
        """
        @return: the complete url of the sessions
        @rtype: string
        """
        return self._user_sessions_url 
    
    def log_dir( self ):
        """
        @return: the absolute path where located the log files
        @rtype: string
        """
        return self._logdir
       
    
    def portal_url( self , relative = False ):
        """
        @param relative:
        @type relative: boolean
        @return: if relative return the relative url of the portal otherwise return the absolute 
                 url of the portal.
        @rtype: string
        """
        if relative:
            return self._portal_prefix
        else:
            return "%s/%s" %( self._root_url , self._portal_prefix )
    
    def portal_path(self):
        """
        @return: the absolute path to the portal
        @rtype: string
        """
        return  self._portal_path
    
    
    def cgi_url( self ):
        """
        @return: the absolute url to the cgis
        @rtype: string
        """
        return "%s/%s" %( self._root_url , self._cgi_prefix )
    

    def programs( self , category  ):
        """
        @return:
        @rtype: 
        """
        try:
            return self._programs[ category ]
        except KeyError:
            return None


    def categories( self ):
        """
        @return:
        @rtype:
        """
        return self._programs.keys()
    

    def response( self ):
        """
        @return:
        @rtype:
        """
        return [ datatype['datatype'] for datatype in self._response['datatypes'] ]


    def getDatabanksConfig(self):
        """
        @return: the list of databanks
         along with their typing information and label
        @rtype: dictionary
        """
        return self._databanks_config
        
    
    def maintainer( self ):
        """
        @return: the email address of the Mobyle server  maintainer
        @rtype: string
        """
        return self._maintainer

    def mailHelp( self ):
        """
        @return: The email address to the hot line for this Mobyle server
        @rtype: string
        """
        return self._help
    
    
    def sender( self ):
        """
        the sender address is used as From: for mail sent by mobyle
        - notification of long job
        - results
        - session confirmation  
        @return: the sender email address ( From: ) of this Mobyle server
        @rtype: string
        """
        return self._sender
    

    def mailhost( self ):
        """
        @return: the mail transfert agent used to send emails
        @rtype: string
        """
        return self._mailhost


    def opt_email( self , jobName = None ):
        """
        @return: True if the user email is not mandatory to run a job, False otherwise
        @rtype: Boolean
        """
        if jobName is None:
            return self._opt_email
        else:
            try:
                return self._particular_opt_email[ jobName ]
            except KeyError:
                return self._opt_email
            
    
    def anonymousSession( self ):
        """
        @return: 
          - 'no'       if the anonymous sessions are not allowed.
          - 'yes'      if the anonymous sessions are allowed, without any restrictions.
          - 'captcha'  if the anonymous sessions are allowed, with a captcha challenge.
        @rtype: string
        """
        return self._anonymous_session
    
    
    def authenticatedSession( self ):
        """
        @return: 
          - 'no'     if authenticated session are not allowed.
          - 'yes'    if authenticated session are allowed and activated without any restriction.
          - 'email'  if authenticated session are allowed but an email confirmation is needed to activate it.
        @rtype: string
        """
        return self._authenticated_session

    def refreshFrequency( self ):
        """
        @return: portal refresh frequency, in seconds
        @rtype: integer
        """
        return self._refresh_frequency
    
    
    def dnsResolver( self ):
        """
        @return: True if the user email domain name is checked to have a mail server , False otherwise 
        @rtype: boolean
        """
        return self._dns_resolver

    
    def mailResults( self ):
        """
        @return: dont_email_result , mailmaxsize in a tuple
        @rtype: ( boolean , int )
        """
        return ( self._dont_email_result , self._maxmailsize )


    def remainResults( self ):
        """
        @return: how long ( in days ) the results files remains on the server.
        @rtype: int
        """
        return self._result_remain

        
    def lang( self ):
        """
        @return: The default language used by this Mobyle server (used to internationalise the form) 
        (2 letter code )(default = "en")
        @rtype: string
        """
        return self._lang


    def filelimit( self ):
        """
        @return: the max size for one file generated by a service ( in byte ). 
        @rtype: int
        """
        return self._filelimit

    def sessionlimit( self ):
        """
        @return: the max size for a session directory ( in byte )
        @rtype: int
        """
        return self._sessionlimit
    
    def previewDataLimit( self ):
        """
        @return: the max size for a result to be preview in job result ( in Byte ).
        @rtype: int
        """
        return self._previewDataLimit   

    
    def seqconverter( self , prgName = None ):
        """
        @return: the path to the sequence converter name prgName. if prgName is None  
          return the list of progName defined ( squizz has a higher priority than redaseq )
        @rtype: string
        """
        if prgName is None:
            l = []
            if 'SQUIZZ' in self._seqconverter:
                l.append( 'SQUIZZ' )
            if 'READSEQ' in self._seqconverter:
                l.append( 'READSEQ' )
            return l
        else:
            try:
                return self._seqconverter[ prgName.upper() ]
            except KeyError:
                return None
    
    def timeout( self ):
        """
        @return: the time during the father process wait its child  ( in sec ).
        @rtype: int 
        """
        return self._timeout

    
    def batch( self , jobName = None ):
        """
        @return: the batch system used by this jobName (Sys ,PBS ,SGE ...). if jobName is None return the default value
        @rtype: string
        """
        try:
            batch = upper(self._particular_batch[jobName])
        except KeyError:
            batch = self._batch
        return batch


    def queue( self , jobName = None ):
        """
        @return: the queue name for this jobName. if jobName is None return the default queue
        @rtype: string
        """
        try:
            queue  = self._particular_Q[ jobName ]
        except KeyError:
            queue = {"queue": self._default_Q}
        return queue


    def queueProperties( self, Qname ):
        """
        @param Qname = the name of the queue
        @type Qname: string
        @return: the properties associated with a queue
        @rtype: dict
        """
        try:
            rs = self._Qproperties[ Qname ]
            return rs
        except KeyError:
            rc_log.critical("queueProperties problem for %s" % Qname)
            return None
            
    def sge( self ):
        """
        @return: the values of variables SGEROOT , SGECELL
        @rtype: tuple (string , string )
        """
        return ( self._sge_root , self._sge_cell )


    def pbs( self ):
        """
        @return: the value of variables PBSROOT
        @rtype: string
        """
        return (self._pbs_root, self._pbs_wait_time, self._pbs_priority)


    def accounting( self ):
        """
        @return: True if the accounting is on, False otherwise.
        @rtype: boolean
        """
        return self._accounting

    def session_debug( self ):
        """
        @return: True if the debug log on session is on, False otherwise.
        @rtype: boolean
        """
        return self._session_debug
    
    def isDisabled( self , programName = None ):
        """
        @param programName: the name of the program (e.g., topppred)
        @type: string
        @return: True if the service is disabled, False otherwise
        @rtype: boolean
        """
        
        if programName is None:
            return self._disable_all
        else:
            if self._disable_all :
                return True
            else:
                return programName in self._disabled_services 


    def restrictedServices( self ):
        """
        @return: The list of service which have a restrictive access
        @rtype: list of strings
        """
        return self._authorized_services.keys()


    def isAuthorized( self , serviceName , addr = None ):
        """
        @param service: the name of the service
        @type service: type
        @param addr: the ip address to test. if addr is None the environment REMOTE_ADDR will be used
        @type addr: string
        @return: True if the user is authorized to used the service (based on IP), False otherwise.
        @rtype: boolean
        """
        if addr is None:
            try: 
                addr = os.environ[ 'REMOTE_ADDR' ]
            except KeyError:
                return True
        try:
            pattern = self._authorized_services[ serviceName ]
        except KeyError:
            # by default if no pattern are specified there is no restriction
            # to access to this service
            return True
        
        if pattern :
            pattern = '|'.join( pattern )
            pattern = pattern.replace('.' , '\.' )
            pattern = pattern.replace('*' , '.*')
            pattern = "^(%s)$" % pattern
            auto = re.compile( pattern )
        else:
            # by default if no pattern are specified there is no restriction
            # to access to this service
            return True
        
        if auto.search( addr ):
            return True
        else:
            return False

    def getPrivilegeTable( self ):
        """
        @return: a dict where keys are the service names and values a list of Ip mask which are authorized to access to this service
        @rtype: dict { string : [ strings ] }
        """
        return self._authorized_services


    def imported_programs( self ):
        """
        @return: the urls of the programs imported by this server
        @rtype: list of strings
        """
        return self._imported_programs
    
    
    def exported_programs( self ):
        """
        @return: the name of the programs which are exported by this server
        @rtype: list of strings
        """
        return self._exported_programs
    
    
    def portals( self ):
        """
        @return: all the Mobyle portals belonging to the "Mobyle grid"
        @rtype: list of dictionaries  [ { 'server' : 'url of Mobyle server' ,
                                            'email' : 'the contact email of this server' }
                                        ]
        """
        return self._all_portals
    
    
    def programs_deployment_include(self):
        """
        @return: the programs deployment include masks list
        @rtype: list of strings
        """
        return self._programs_deployment_include
    
    
    def programs_deployment_exclude(self):
        """
        @return: the programs deployment exclude masks list
        @rtype: list of strings
        """
        return self._programs_deployment_exclude    
    
    
    def programs_deployment_order(self):
        """
        @return: the programs deployment include/exclude priority order
        @rtype: list of strings
        """
        return self._programs_deployment_order

    
    
    
class PassChecker:
    """

    """

    _cfg = Config()


    def __init__( self ):
        """
        build a dictionary with all IP masks found in Config.Config.AUTHORIZED_SERVICES as keys and the services authorized for each mask as values
        """
        
        serviceInfo = {}
    
        for path in glob.glob( os.path.join ( _cfg.mobylehome() , "Local" , "Programs" ,"*.xml" ) ):

            serviceInfo [ os.path.basename( path )[:-4] ] = None

        for path in  glob.glob( os.path.join ( _cfg.mobylehome()  , "Programs" ,"*.xml" ) ):
            service = os.path.basename( path )[:-4]

            if not serviceInfo.has_key( service ):
                serviceInfo [ service ] = None

        self.maskService = { '*.*.*.*' : None }
        self.authorizedServices = PassChecker._cfg.getPrivilegeTable()
        restrictedServices = self.authorizedServices.keys()
        
        for service in restrictedServices :
            for mask in self.authorizedServices[ service ]:
                if self.maskService.has_key( mask ):
                    continue
                else:
                    self.maskService[ mask ] = None

        errors = []
        for mask in self.maskService.keys():
            newServiceInfo = copy.deepcopy( serviceInfo )

            for service in restrictedServices :
                if not PassChecker._cfg.isAuthorized( service , mask ):
                    try:
                        del newServiceInfo[ service ]
                    except KeyError:
                        msg = "the service :\"%s\" is in your Local.Config.Config.AUTHORIZED_SERVICES but the xml description of this service does no exist" %service
                        if msg not in errors :
                            errors.append( msg )
            self.maskService[mask] = newServiceInfo.keys() 

        for error in errors:
            PassChecker._cfg.log.warning( error )
            

    def getMask( self , addr ):
        """
        @param addr: an ip addresse (ipv4) like '192.168.3.1'
        @type addr: string
        @return: the mask which fit the best ( permit to access at the most services )to the addr. or None if any mask match 
        @rtype: string
        """
        masks = []
        
        for mask in self.maskService.keys():
            pattern = mask.replace( '.' , '\.' )
            pattern = pattern.replace( '*' , '.*')
            pattern = "^(%s)$" % pattern
            auto = re.compile( pattern )
            if auto.search( addr ):
                masks.append( mask )

        if len( masks ) == 1:
            return masks[ 0 ]
        else:
            #celui qui permet d'acceder aux maximum de services.
            bestmask = None
            bestscore = 0
            
            for mask in masks:
                score = len( self.maskService[ mask ] )
                if score > bestscore:
                    bestscore = score
                    bestmask = mask

            return bestmask


    def getAllMasks( self ):
        """
        @return: the list of all mask
        @rtype: list of strings
        """
        return self.maskService.keys()


    def itermasks( self ):
        """
        @return: an iterator on all masks
        """
        for m in self.maskService.keys() :
            yield m


    def __getitem__( self , mask ):
        """
        @param mask: an ip mask (ipv4) like '192.168.*.*'
        @type mask: string
        @return: the list of services accessible for an ip address witch match to the mask
        @rtype: list of string
        """
        return self.maskService[ mask ]


    def has_mask( self , mask ):
        """
        @param mask: an ip mask (ipv4) like '192.168.*.*'
        @type mask: string
        @return: True if the mask is in this PassChecker instance, False othrewise
        @rtype: boolean
        """
        return self.maskService.has_key( mask )

    
