
########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################


"""
Classes executing the command and managing the results  
"""
import os ,os.path
from subprocess import Popen, PIPE
import signal 
import sys
import time

import logging 


import logging
_log = logging.getLogger(__name__)

import Mobyle.JobState
import Mobyle.ConfigManager
import Mobyle.Utils
from Mobyle.Execution.Batch import _Batch

import Local.Policy
from Mobyle.MobyleError import *


_cfg = Mobyle.ConfigManager.Config()
__extra_epydoc_fields__ = [('call', 'Called by','Called by')]




class SYS(_Batch):
    """
    Run the commandline by a system call
    """

    def run(self):
        """
        Run the commandLine in a subshell (system call) and redirect the standard error and output on service.name.out and service.name.err, then restore the sys.stderr and sys.stdout
        @return: the L{Mobyle.Utils.Status} of this job and a message
        @rtype: ( int , string )
        @call: in _Batch.__init__
        """
        if (os.getcwd() != os.path.abspath(self.dirPath) ):
            msg = "the process is not in the working directory"

            self._logError( admMsg = msg ,
                            userMsg = "Mobyle internal server error" ,
                            logMsg = msg )

            raise MobyleError , msg

        else:
            jobKey = self.getKey()
            fout = open( self.serviceName + ".out" , 'w' )
            ferr = open( self.serviceName + ".err" , 'w' )

            try:
                #the new process launch by popen must be a session leader 
                # because it launch in background and if we send a siggnal to the shell 
                # this signal will no propagate to all child process
                # and if we kill the process leader we kill the python 
                # thus we execute our setsid which transform the new process as a session leader
                # and execute the command prior to return.
                # in these conditions we could kill the process leader and the signal is propagate to
                # all processes belonging to this session. 

                setsidPath = '%s/Tools/setsid' %_cfg.mobylehome()
                command = [ setsidPath , setsidPath , '/bin/sh' , '-c' , self.commandLine ]
                # _log.debug("SYS env : %s\n" % str(self.xmlEnv) )

                pipe = Popen( command ,
                              shell     = False ,
                              stdout    = fout ,
                              stdin     = None ,
                              stderr    = ferr ,
                              close_fds = True ,
                              env       = self.xmlEnv
                              )

            except OSError, err:
                msg= "System execution failed: " + str(err)

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   jobKey ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg 
                
            jobPid = pipe.pid
            
            self._adm.setBatch( 'SYS' )
            self._adm.setNumber( jobPid )
            self._adm.commit()

            linkName = ( "%s/%s.%s" %( _cfg.admindir() ,
                                       self.serviceName ,
                                       jobKey
                                    )
                         )
            
            try:
                os.symlink(
                    os.path.join( self.dirPath , '.admin') ,
                    linkName
                    )
            except OSError , err:
                msg = "can't create symbolic link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            #pipe.wait() doesn't work with signal.signal SIGUSR1 & SIGUSR2 (until python 2.5)
            #OSError: [Errno 4] Interrupted system call
            #RESOLVED with used siginterrupt.siginterrupt( signal , False )

            pipe.wait()

            try:
                os.unlink( linkName )
            except OSError , err:
                msg = "can't remove symbolic link %s in ADMINDIR: %s" %( linkName , err )

                self._logError( admMsg = msg ,
                                userMsg = "Mobyle internal server error" ,
                                logMsg = None )
                
                _log.critical( "%s/%s : %s" %( self.serviceName ,
                                                   self.getKey() ,
                                                   msg
                                                   )
                                 )

                raise MobyleError , msg

            fout.close()
            ferr.close()
            self._returncode = pipe.returncode
            
            self._adm.refresh()
            oldStatus = self._adm.getStatus()
 
            if oldStatus.isEnded():
                return oldStatus
            else:
                if pipe.returncode == 0 :# finished
                    status = Mobyle.Utils.Status( code = 4 ) #finished
                elif pipe.returncode == -15 or pipe.returncode == -9: #killed
                    status = Mobyle.Utils.Status( code = 6 , message = "Your job has been cancelled" ) 
                elif pipe.returncode < 0 or pipe.returncode > 128:
                    status = Mobyle.Utils.Status( code = 6 , message = "Your job execution failed ( %s )" %pipe.returncode )
                else:
                    status = Mobyle.Utils.Status( code= 4 , message = "Your job finished with an unusual status code ( %s ), check your results carefully." % pipe.returncode )
    
                return status
        

    @staticmethod
    def getStatus( pid ):
        """
        @param pid:
        @type pid:
        @return: the status of job with pid pid
        @rtype: string
        """
        pid = int( pid )
        try:
            message = None
            os.kill( pid , signal.SIG_DFL )
        except OSError , err:
            if str( err ).find( 'No such process' ) != -1:
                return Mobyle.Utils.Status( code = -1 ) #unknown
            else:
                raise MobyleError , err 

        return Mobyle.Utils.Status( code = 3 , message = message ) #running

    @staticmethod
    def kill( job_pid ):
        """
        kill a job
        @param job_pid: the pid of the job
        @type job_pid: int
        @raise MobyleError: if can't kill the job
        @call: by L{Utils.Mkill}
        """
        try:
            job_pid = int( job_pid )
            job_pgid = os.getpgid( job_pid )
        except ValueError:
            raise MobyleError , "can't kill job  - %s" % err 
        try:
            os.killpg( job_pgid , signal.SIGTERM )
        except OSError , err:
            raise MobyleError , "can't kill job  - %s" % err 
        
               
        try:
            time.sleep(0.2)
            os.killpg( job_pgid , signal.SIG_DFL )
            try:
                os.killpg( job_pgid , signal.SIGKILL )
            except OSError , err:
                raise MobyleError , "can't kill job - %s" % err
                
        except OSError , err:
            return
        
