########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.        #
#                                                                                      #
########################################################################################

import os ,os.path ,random , string 
import resource
from hashlib import md5
import time 
import glob
import logging 
j_log = logging.getLogger( __name__ )

import Mobyle.JobState 
import Mobyle.ConfigManager
import Mobyle.Utils

from Mobyle.MobyleError import *


__extra_epydoc_fields__ = [('call', 'Called by','Called by')]



class Job:

    """
    create the environment to run a job
      - working directory creation
    """
 
    
    def __init__(self, service , cfg , session = None , userEmail = None , synchron = False):
        """
        @param service:
        @type service: a {Service} instance
        @param cfg: a config object
        @type cfg: L{ConfigManager.Config} instance
        @param session:
        @type session: a L{Session} instance
        @param userEmail: the email of the user 
        @type userEmail: a L{Mobyle.Net.EmailAddress} instance
        @param synchron: 
        @type synchron: boolean

        """
        self.cfg = cfg
        self.tmpdir = None
        self.jobState = None
        self.service = service
        self.key = self._newKey()
        self.runner = None
        self.session = session
        self.userEmail = userEmail
        self.synchron = synchron
        self.adm = None 
        self.makeEnv()



    def makeEnv( self, uri = None ):
        """
        create the environment to run a job
          - working directory creation
          - fixing permission
          - creation of the xml file index.xml
        """
        basedir =  self.cfg.results_path() 
        if not os.path.exists( basedir ):
            msg = "directory " + str( basedir ) + " does not exist"
            j_log.error(msg)
            raise MobyleError, msg

        if uri is None:            
            workdir = os.path.join( basedir , self.service.getName() , self.key )
            if os.path.exists( workdir ):
                msg = 'cannot make directory '+ str( workdir )+" : File exists"
                
                j_log.error( msg )
                raise MobyleError , msg

            os.makedirs( workdir , 0755 ) #create parent directory
        try:    
            os.chdir( workdir )
        except OSError, err:
            msg = "unable to change directory to:" + str( err )
            j_log.error( msg )
            raise MobyleError , msg 

        os.umask(0022)
        self.tmpdir = workdir

        resource.setrlimit( resource.RLIMIT_CORE ,(0 , 0) )
        maxFileSize = self.cfg.filelimit()
        resource.setrlimit( resource.RLIMIT_FSIZE ,( maxFileSize , maxFileSize ) )

        self.date = time.localtime()
        
        #be careful the order of setting informations in state is important
        
        if self.service.isService:
            self.jobState = Mobyle.JobState.JobState( uri= workdir , create= "SERVICE" )
            #self.jobState.createState( "SERVICE" )
        else:
            self.jobState = Mobyle.JobState.JobState( uri= workdir , create= "PIPELINE" )
            #self.jobState.createState( "PIPELINE" )

        serviceName = self.service.getName()
        jobID = self.getURL()
        
        self.jobState.setName( self.service.getUrl() )
        self.jobState.setHost( self.cfg.root_url() )
        self.jobState.setID( jobID )
        self.jobState.setDate( self.date )
        self.jobState.setStatus( Mobyle.Utils.Status( code= 0 ) ) # building  
        if self.userEmail is not None:
            self.jobState.setEmail( self.userEmail )
        if self.session is not None:
            sessionKey = self.session.getKey()
            self.jobState.setSessionKey( sessionKey )

        self.jobState.commit()
   
        #create the .admin file
        self.adm = Mobyle.Utils.Admin( self.tmpdir )
        self.adm.setDate()
        if self.userEmail is not None:
            self.adm.setEmail( self.userEmail )

        try:
            remote_host = os.environ['REMOTE_HOST']
        except KeyError :
            remote_host = "UNKNOWN"
        try:
            ip_addr = os.environ['REMOTE_ADDR']
        except KeyError :
            ip_addr = 'local'
        
        self.adm.setRemote( ip_addr+"\\"+remote_host )

        ## Attention I'm building the job instance
        ## during the MobyleJob instanciation
        ## thus the job is not already add to the session
        ## this is done explicitly in the CGI when a value is submitted

        self.adm.setJobName( serviceName )
        self.adm.setJobID( jobID )
        self.adm.setStatus( Mobyle.Utils.Status( code = 0 ) )
        if self.session is not None:
            self.adm.setSession( sessionKey )
        self.adm.commit()
        
       
    def _newKey(self):
        """
        @return: a unique key which serve to indentify a job
        @rtype: string
        """

        #for PBS/SGE jobname, the first char of tmp dir must begin by a letter
        letter = string.ascii_uppercase[ random.randrange( 0 , 26 ) ]
        #max 15 chars for PBS jobname
        strTime = "%.9f" % time.time()
        strTime = strTime[-9:]
        strPid = "%05d" %os.getpid()
        strPid = strPid.replace( '.' , '' )[ -5 : ]
        return letter + strPid + strTime

    def setSession(self , session ):
        self.session = session
        sessionKey = self.session.getKey() 
        self.jobState.setSessionKey( sessionKey )
        self.jobState.commit()
        self.adm.setSession( sessionKey )
        self.adm.commit()
        
        
    def getKey( self ):
        """
        @return: the unique key of this job
        @rtype: string
        """
        return  self.key

    def getDate( self ):
        """
        @return: the submission date of this job
        @rtype: string
        """
        return  self.date
    

    def getDir(self):
        """
        @return: the absolute path to the directory where the job is executed
        @rtype: string
        """
        
        return self.tmpdir

   
    def getURL(self):
        """
        @return: the URL to the directory where the job is executed
        @rtype: string
        """
        url = Mobyle.JobState.path2url( self.tmpdir )
        return url

    def getService( self ):
        """
        @return: the instance of the service used for this job
        @rtype: Service
        """
        return self.service

    def getEmail( self ):
        """
        @return: the email of the user or None if there is no email
        @rtype: -L{Net.EmailAddress} instance
        """
        return self.userEmail


    def isLocal(self):
        """
        @return: True if this job is on this server , False otherwise
        @rtype: boolean
        """ 
        return self.jobState.isLocal()
        
        
    def run( self ):
        """
        Run the Job (instanciate a CommandRunner)
        @call: L{MobyleJob.run <MobyleJob>}
        """
        import Mobyle.RunnerFather
        
        if self.jobState is None:
            
            msg = "you must make an environment for the job before running it"
            self._logError( admMsg = "Job.toFile : " + msg ,
                            logMsg = msg ,
                            userMsg = "Mobyle Internal server error"
                            )
            
            raise MobyleError, msg

        self.jobState.update()
        self.jobState.setStatus( Mobyle.Utils.Status( code = 1 ) )#submitted
        self.jobState.commit()
        self.adm = Mobyle.Utils.Admin( self.tmpdir )
        self.adm.setStatus( Mobyle.Utils.Status( code = 1 ) )
        self.adm.commit()
        
        evaluator = self.service.getEvaluator()

        self.runner = Mobyle.RunnerFather.CommandRunner( self , synchron= self.synchron  )

        cmdLine = self.runner.getCmdLine()

        debug = self.cfg.debug( self.service.getName() )

        if debug > 0 and debug < 3:# the job will not run
            self.jobState.setStatus( Mobyle.Utils.Status( code = 4 ) )#finished
            self.jobState.commit()
            self.adm.setStatus( Mobyle.Utils.Status( code = 4 ) )
            self.adm.commit()
        else:
            if self.userEmail is not None:
                jobExist , msg = self.isExists( self.userEmail , cmdLine )
            else:
                jobExist = False
            
            if jobExist :
                #msg = "you have already submitted a similar job" 
                userMsg = "a similar job have been already submitted the %s, and is not finished yet. Please wait for the end of this job before you resubmit." %( time.strftime( "%x %X" , self.adm.getDate() ) )
                self._logError( admMsg = "Job.run : " + msg + " : run aborted ",
                                logMsg = msg + " : run aborted " ,
                                userMsg = userMsg
                                )

                raise UserValueError( parameter = None, msg = userMsg )

            else:
                self.runner.run()
        

    def kill( self ) :
        """@todo: kill the job"""
        raise NotImplementedError , "ToDo"


    def isExists( self, userEmail , cmdLine ):
        """
        @param userEmail: the email address of the user
        @type userEmail: -L{Net.EmailAddress} instance
        @param cmdLine:
        @type cmdLine: string
        """
        newMd5 = md5()
        newMd5.update( str( userEmail ) )

        try:
            remote = os.environ[ 'REMOTE_HOST' ]
            if not remote :
                try:
                    remote = os.environ[ 'REMOTE_ADDR' ]
                except KeyError :
                    remote = 'local'
        except KeyError:
            try:
                remote = os.environ[ 'REMOTE_ADDR' ]
            except KeyError:
                remote = 'local'
                
        newMd5.update( remote )
        newMd5.update( cmdLine )
        newDigest = newMd5.hexdigest()
        self.adm.refresh() #est ce utile ?
        self.adm.setMd5( newDigest )
        self.adm.commit()

        #batch.number.name.md5
        mask = os.path.normpath( "%s/%s.*" %(
            self.cfg.admindir() ,
            self.service.getName()
            )
                                 )
        jobs = glob.glob( mask )
        exist = False
        msg = None
        
        for job in jobs:
            oldAdm = Mobyle.Utils.Admin( job )
            oldDigest = oldAdm.getMd5()

            if newDigest == oldDigest :
                oldStatus = oldAdm.getStatus() 

                if not oldStatus.isEnded() :
                    exist = True
                    date = time.strftime( "%x %X" , self.adm.getDate() )
                    now = time.localtime()
                    msg = "a similar job (%s) have been already submitted the %s. it's status is: %s" % (
                        os.path.basename( job ),
                        date ,
                        oldStatus
                        )
                    break
                
        return ( exist , msg )



    def _logError( self , admMsg = None , userMsg = None , logMsg = None ):

        if  admMsg :
            self.adm.setStatus( Mobyle.Utils.Status( code = 5 , message = admMsg ) )
            self.adm.commit()

        if userMsg :
            self.jobState.setStatus( Mobyle.Utils.Status( code = 5 , message = userMsg ) )
            self.jobState.commit()

        if logMsg :
            j_log.error( "%s : %s : %s" %( self.service.getName() ,
                                          self.getKey() ,
                                          logMsg
                                          )
                        )
            
