########################################################################################
#                                                                                      #
#   Author: Bertrand Neron,                                                            #
#   Organization:'Biological Software and Databases' Group, Institut Pasteur, Paris.   #  
#   Distributed under GPLv2 Licence. Please refer to the COPYING.LIB document.         #
#                                                                                      #
########################################################################################

"""
 Tools to parse and build services for Mobyle

"""
import sys
import os.path
import Ft.Xml.Domlette
import logging

b_log = logging.getLogger('Mobyle.builder')

import Mobyle.ConfigManager
import Mobyle.Service
from  Mobyle.Classes.DataType import DataTypeFactory
from Mobyle.Evaluation import Evaluation
from Mobyle.Registry import registry

from Mobyle.MobyleError import *


__extra_epydoc_fields__ = [('call', 'Called by','Called by')]



class ServiceParser( object ) :
    
    logger = logging.getLogger( __name__ )
    def __init__(self , cfg = None , debug = 0 ):
        
        self._debug = debug
        if cfg:
            self._cfg = cfg
        else:
            self._cfg = Mobyle.ConfigManager.Config()
        
    
    def parse(self , serviceUrl ):
        """
        @param serviceUrl: the url of a Mobyle Service definition
        @type serviceUrl: string
        @return: a service
        @rtype: service instance
        """
        
        try:
            servicePath = registry.programsByUrl[serviceUrl].path
        except KeyError:
            raise MobyleError , "the service %s doesn't exist" % serviceUrl
        
        doc = Ft.Xml.Domlette.NoExtDtdReader.parseUri( servicePath )
        
        mobyleNode = doc.xpath( './mobyle')
        if mobyleNode:
            try:
                serviceNode = doc.xpath( './mobyle/program')[0]
            except IndexError:
                try:
                    pipeNode = doc.xpath( './mobyle/pipeline')[0]
                except IndexError:
                    raise ParserError , "a service must be a program or a pipeline"
                else:     
                    service = self.parsePipeline( pipeNode )
            else:
                service = self.parseProgram( serviceNode )
        else:            
            try:
                serviceNode = doc.xpath( './program')
                serviceNode = doc.xpath( './program')[0]
            except MobyleError:
                try:
                    pipeNode = doc.xpath( './pipeline')[0]
                except IndexError:
                    raise ParserError , "a service must be a program or a pipeline"
                else:     
                    service = self.parsePipeline( pipeNode )
            else:
                service = self.parseProgram( serviceNode )
                
        service.setUrl( serviceUrl )   
        return service
    
    
    def parsePipeline( self , pipelineNode ):
        raise NotImplementedError, "parsePipeline is not yet implemented todo"

    
    def parseProgram(self , programNode ):
        ######################################
        #
        #  validation 
        #
        ######################################
        self.evaluator = Evaluation() 
        #self.dataTypeFactory = dataTypeFactory
        self.dataTypeFactory = DataTypeFactory()
        service = Mobyle.Service.Service( self.evaluator )
        
        try:
            name = programNode.xpath( "./head/name/text()" )[0].data
            service.setName( name )
        except IndexError:
            raise ParserError , "program has no tag \"name\""
                  
        try:
            version = programNode.xpath( "./head/version/text()" )[0].data
            service.setVersion( version )
        except IndexError:
            pass
          
        try:
            title = programNode.xpath( "./head/doc/title/text()" )[0].data
            service.setTitle( title )
        except IndexError:
            raise MobyleError , "service has no title"
        
        desctiptionNodes = programNode.xpath( "./head/doc/description/text" )
        for descriptionNode in desctiptionNodes :
            content , proglang , lang , href  = ServiceParser.toText( descriptionNode )
            if lang:
                service.addDescription( content , lang = lang)
            elif href:
                service.addDescription( content , href = href)
            elif proglang:
                service.addDescrition( content , proglang = proglang )
        
        try:
            authors = programNode.xpath( "./head/doc/authors/text()" )[0].data
            service.setAuthors( authors )
        except IndexError:
            pass

        references = programNode.xpath( "./head/doc/reference/text()" )
        for reference in references:
            service.addReference( reference.data )
        
        doclinks = programNode.xpath( "./head/doc/doclink/text()" )
        for doclink in doclinks:
            service.addDoclink( doclink.data )
        
        helpNodes = programNode.xpath( "./head/doc/help/text" )
        for helpNode in helpNodes :
            content , proglang , lang , href  = ServiceParser.toText( helpNode )
            if lang:
                service.addHelp( content , lang = lang)
            elif href:
                service.addHelp( content , href = href)
            elif proglang:
                service.addHelp( content , proglang = proglang )
        
        categoryNodes = programNode.xpath( "./head/category" )
        categories = []
        for categoryNode in categoryNodes:
            categories.append( categoryNode.xpath( "./text()" )[0].data )
        
        if categories:
            service.addCategories( categories )
        
        try:
            command = programNode.xpath( "./head/command/text()" )[0].data
            try:
                type = programNode.xpath( "./head/command/@type" )[0].value
            except IndexError:
                type = None
            try:
                path = programNode.xpath( "./head/command/@path" )[0].value
            except IndexError:
                path = None
                
            if type and path:
                service.setCommand( command , type = type , path = path )
            elif type:
                service.setCommand( command , type = type )
            elif path:
                service.setCommand( command , path = path )
            else:
                service.setCommand( command )
        except IndexError:
            pass

        envNodes = programNode.xpath( "./head/env" )
        for envNode in envNodes:
            try:
                envName = envNode.xpath( './@name')[0].value
                envValue = envNode.xpath( './text()')[0].data
                service.addEnv( envName , envValue )
            except IndexError:
                raise ParserError , "invalid env element in head"
                      
        allParameterNodes = programNode.xpath( './parameters/parameter')
        for parameterNode in allParameterNodes:
            try:
                service.addParameter( ServiceParser.parseParameter( parameterNode , self.evaluator, self.dataTypeFactory ) )
            except MobyleError , err :
                raise ParserError , "%s : %s : %s" %( service.getName() ,
                                                      parameterNode.xpath( "./name/text()")[0].data ,
                                                      err
                                                      )
         
        allParagraphNodes = programNode.xpath( './parameters/paragraph')
        
        for paragraphNode in allParagraphNodes:
            service.addParagraph( ServiceParser.parseParagraph( paragraphNode , self.evaluator, self.dataTypeFactory )  )
        

        if self._debug > 1:
            b_log.debug("""
            \t##########################################
            \t#                                        #
            \t#              vdefs filling             #
            \t#                                        #
            \t##########################################
            """)
        #service.resetAllParam() #tous les parametres sont dans l'evaluateur
        for paramName in service.getAllParameterNameByArgpos() :
            #all parameters must be in evaluation space
            #in some output parameters there is a format element eg to rename the output file
            # see outfile parameter in protdist.xml ( this renaming is mandatory to avoid conflict 
            # when 2 phylip job are piped
            try:
                service.reset( paramName )
            except MobyleError , err :
                msg = "%s.%s : invalid vdef : %s" %( service.getName(),
                                                 paramName ,
                                                 err
                                                 )
                if self._debug < 2 :
                    ServiceParser.logger.critical( msg )
                else:
                    ServiceParser.logger.error( msg )
                raise MobyleError , msg
        return service
    

    @staticmethod
    def parseParagraph( paragraphNode , evaluator , dataTypeFactory = None ):
        paragraph = Mobyle.Service.Paragraph( evaluator )
        if dataTypeFactory is None:
            dataTypeFactory = DataTypeFactory()
        try:
            name = paragraphNode.xpath( "name/text()" )[0].data
            paragraph.setName( name )
        except IndexError:
            raise MobyleError , "paragraph has no tag \"name\""
        
        for promptNode in paragraphNode.xpath( './prompt' ): # plusieurs prompts dans des lang differents
            try:
                promptLang = promptNode.xpath( './@lang' )[0].value
            except IndexError:
                try:
                    paragraph.addPrompt( promptNode.xpath( './text()' )[0].data )
                except IndexError:
                    continue #the element prompt is empty
            else:
                try:
                    paragraph.addPrompt( promptNode.xpath( './text()' )[0].data , lang = promptLang )
                except IndexError:
                    continue #the element prompt is empty
        precondNodes = paragraphNode.xpath( './precond/code' )
        for precondNode in precondNodes :
            precond = precondNode.xpath( './text()' )[0].data
            try:
                proglang = precondNode.xpath( './@proglang' )[0].value
                paragraph.addPrecond( precond , proglang = proglang )
            except IndexError:
                paragraph.addPrecond( precond )
         
        try:
            argpos = paragraphNode.xpath( './argpos/text()' )[0].data
            paragraph.setArgpos( int( argpos ) )
        except IndexError:
            pass
        except ValueError:
            raise ParserError , "Argpos must be an integer"
 
        format = paragraphNode.xpath( './format' ) 
        if format :
            for codeNode in format[0].xpath( './code'):
                try:
                    proglang = codeNode.xpath( './@proglang')[0].value
                except IndexError:
                    raise ParserError ,"comment est ce possible"
                
                code = codeNode.xpath( "./text()" )[0].data
                paragraph.addFormat( code , proglang)
 
        comment = paragraphNode.xpath( './comment' )
        if comment:
            for text in comment[0].xpath( './text' ):
                oneComment = ServiceParser.toText( text )
                
                if oneComment[1]:
                    paragraph.addComment( oneComment[0] , proglang = oneComment[1])
                elif oneComment[2] :
                    paragraph.addComment( oneComment[0] , lang = oneComment[2])
                elif oneComment[3]:
                    paragraph.addComment( oneComment[0] , href = True )
                else:
                    paragraph.addComment( oneComment[0] , lang = 'en')
              
        ##################################
        #
        #   descente recursive dans l'arbre des paragraphes et paramettres
        #
        ##################################
 
        paragraphChildNodes = paragraphNode.xpath( './parameters/paragraph')
        
        for paragraphChildNode in paragraphChildNodes:
            
            paragraphChild = ServiceParser.parseParagraph( paragraphChildNode , evaluator , dataTypeFactory )
            paragraph.addParagraph( paragraphChild )
    
        parameterChildNodes = paragraphNode.xpath( './parameters/parameter')
        for parameterChildNode in parameterChildNodes:
            try:
                parameterChild = ServiceParser.parseParameter( parameterChildNode , evaluator , dataTypeFactory )
            except MobyleError , err :
                raise ParserError , "error while parsing parameter %s : %s" %( parameterChildNode.xpath( "./name/text()")[0].data ,
                                                      err
                                                      )

            paragraph.addParameter( parameterChild )
        
        return paragraph 
    
    
    @staticmethod
    def parseType( typeNode , dataTypeFactory = None ):
        
        if not dataTypeFactory:
            dataTypeFactory = DataTypeFactory()
        try:
            bioTypes = typeNode.xpath( "./biotype/text()" )
            bioTypes = [ bt.data for bt in bioTypes ]
        except IndexError :
            bioTypes = None
        
        #to be compliant with the actual grammar of type in service definition 
        #and the future version ( without node acceptedDataFormats ) which is already in use in session 
        #formatList = typeNode.xpath( "./acceptedDataFormats/dataFormat/text()" )
        formatList = typeNode.xpath( ".//dataFormat/text()" )
        
        if formatList:
            formats = [ fmt.data for fmt in formatList ]
            try:
                forceFormat = typeNode.xpath( "./acceptedDataFormats/@force" )[0].value 
            except IndexError:
                forceFormat = False
        else:
            formats = []
            forceFormat = False
        try:    
            card = str( typeNode.xpath( "./type/card/text()" )[0].data )
            try:
                min , max = card.split(",")
            except ValueError:
                if len( card ) == 2:
                    min = max = card 
                else:
                    raise ParserError , "invalid card: %s .the card element must be a string of 2 integer or \"n\" separate by a comma : "% card 
            try:
                min = int( min )
            except ValueError :
                if min != "n":
                    raise ParserError , "invalid card: %s .the card element must be a string of 2 integer or \"n\" separate by a comma : "% card 
            try:
                max = int( max )
            except ValueError :
                if max != "n":
                    raise ParserError , "invalid card: %s .the card element must be a string of 2 integer or \"n\" separate by a comma : "% card 
            card = ( min , max )
        except IndexError :
            card = ( 1 , 1 )
        try:
            superKlass = typeNode.xpath( "./datatype/superclass/text()" )[0].data
        except IndexError:
            superKlass = None
        try:
            klass = typeNode.xpath( "./datatype/class/text()" )[0].data
        except IndexError:
            if superKlass is None :
                raise ParserError , typeNode.xpath( "./name/text()")[0].data + " must have either a \"class\" element either a \"class\" and \"superclass\" element"
            else:
                raise ParserError , typeNode.xpath( "./name/text()")[0].data +" if the \"superclass\" is specified the the \"class\" element must be also specified"
        try:
            if superKlass:
                dataType = dataTypeFactory.newDataType( superKlass , xmlName = klass )
            else:
                dataType = dataTypeFactory.newDataType( klass )
        except MobyleError , err :

            raise ParserError , err
        
        mobyleType = Mobyle.Service.MobyleType( dataType , bioTypes = bioTypes , formats = formats , forceFormat = forceFormat , card = card )
        
        return  mobyleType  

    @staticmethod
    def parseParameter( parameterNode , evaluator = None , dataTypeFactory = None ):
        
        try:
            typeNode = parameterNode.xpath( "./type")[0]
            mobyleType = ServiceParser.parseType(  typeNode , dataTypeFactory = dataTypeFactory )
            
        except ( ParserError , MobyleError ), err :
            try:
                serviceName = parameterNode.xpath( '/program/head/name/text()')[0].data 
            except IndexError:
                try:
                    serviceName = parameterNode.xpath( '/mobyle/program/head/name/text()')[0].data
                except IndexError:
                    serviceName = "??"
            try:
                paramName= parameterNode.xpath('name/text()')[0].data
            except IndexError:
                paramName = "??"
            
            ServiceParser.logger.critical( "an error occured during a %s.%s parameter parsing: %s" %( serviceName ,
                                                                                                      paramName ,
                                                                                                      err )
                                                                                                    )
            raise ParserError , err

        parameter = Mobyle.Service.Parameter( mobyleType )
        
        #############################
        #                           #
        #   parsing des attributs   #
        #                           #
        #############################
        
        attrs = parameterNode.xpath( "./@*" )
        for attr in attrs:
        
            if attr.name == 'ismandatory' and attr.value == "1":
                parameter.setMandatory( True )

            elif attr.name == 'ismaininput' and attr.value == "1":
                parameter.setMaininput( True )
        
            elif attr.name == 'iscommand' and attr.value == "1":
                parameter.setCommand( True )
        
            elif attr.name == 'ishidden' and attr.value == "1":
                parameter.setHidden( True )
                
            elif attr.name == 'issimple' and attr.value == "1": # a t'il encore une signification
                parameter.setSimple( True )
        
            elif attr.name == 'isout' and attr.value == "1":
                parameter.setOut( True )
                
            elif attr.name == 'isstdout' and attr.value == "1":
                parameter.setStdout( True )
                parameter.setOut( True )
                  
            elif attr.name == 'bioMoby' :
                parameter.setBioMoby( attr.value )
        
            elif attr.name ==  'formfield':
                parameter.setFormField( attr.value )
        try:
            name = parameterNode.xpath( "name/text()" )[0].data
            parameter.setName( name )
        except IndexError:
            raise ParserError , "parameter has no tag \"name\""
        
        for promptNode in parameterNode.xpath( './prompt' ): # plusieurs prompts dans des lang differents
            try:
                promptLang = promptNode.xpath( './@lang' )[0].value
            except IndexError:
                try:
                    parameter.addPrompt( promptNode.xpath( './text()' )[0].data )
                except IndexError:
                    continue #the prompt element is empty
            else:
                try:
                    parameter.addPrompt( promptNode.xpath( './text()' )[0].data , lang = promptLang )
                except IndexError:
                    continue #the prompt element is empty
                
        format = parameterNode.xpath( './format' ) 
        if format :
            for codeNode in format[0].xpath( './code'):
                try:
                    proglang = codeNode.xpath( './@proglang')[0].value
                except IndexError:
                    raise MobyleError
                try:
                    code = codeNode.xpath( "./text()" )[0].data
                except IndexError , e:
                   code = "" 
                   pname = codeNode.xpath( "/program/head/name/text()" )[0].data
                   ServiceParser.logger.warning( "find empty element code in %s.%s parameter. The code value is set to \"\" " %( pname , name )  )
                
                parameter.addFormat( code , proglang)
        
        vdefs = [] # #dans clustalW hgapresidue est un MultipleChoice et la vdef est une liste de valeurs
        for vdefNode in parameterNode.xpath( './vdef/value/text()' ):
            vdefs.append( vdefNode.data )
        if vdefs:
            parameter.setVdef( vdefs )
        #the vdef can't be code anymore the service and apidoc must be updated
        #getVdef could return always list 
        
        try:
            argpos = parameterNode.xpath( './argpos/text()' )[0].data
            parameter.setArgpos( int( argpos ) )
        except IndexError:
            pass
        except ValueError:
            raise ParserError , "Argpos must be an integer"
        
        vlist = parameterNode.xpath( './vlist' )
        if vlist:
            elems  = vlist[0].xpath('./velem')

            for elem in elems:
              try:
                  label = elem.xpath('./label/text()')[0].data
              except IndexError:
                  label = "" 
              try:  
                  value = elem.xpath('./value/text()')[0].data
              except IndexError:
                  value = ""
              try:
                  undef = elem.xpath( './@undef')[0].value 
              except IndexError:
                  undef = 0
              if undef:
                  parameter.setListUndefValue( value )
              else:    
                  parameter.addElemInVlist(  label , value ) 
        flist = parameterNode.xpath( './flist' )      

        if flist:
            elems  = flist[0].xpath('./felem')
            for elem in elems:
                try:
                    label = elem.xpath( './label/text()')[0].data 
                except IndexError:
                    label = ""
                try:
                    value = elem.xpath( './value/text()')[0].data
                except IndexError:
                    value = ""
                codes = {}
                for codeNode in elem.xpath( './code' ):
                    try:
                        code = codeNode.xpath( './text()')[0].data
                    except IndexError:
                        code = "" 
                        pname = codeNode.xpath( "/program/head/name/text()" )[0].data
                        ServiceParser.logger.warning( "find empty felem code in %s.%s parameter. The code value is set to \"\" " %( pname , name )  )
                    try:
                        proglang = codeNode.xpath( './@proglang')[0].value
                    except IndexError:
                        pname = codeNode.xpath( "/program/head/name/text()" )[0].data
                        msg = "find felem code in %s.%s parameter without proglang"%( pname , name ) 
                        ServiceParser.logger.critical(msg)
                        raise ParserError , msg
                    
                    codes [ proglang ] = code
                
                try:
                    undef = elem.xpath( './@undef')[0].value 
                except IndexError:
                    undef = 0
              
                if undef:
                    parameter.setListUndefValue( value )
                else:                   
                    parameter.addElemInFlist( value, label , codes )
                
        comment = parameterNode.xpath( './comment' )
        if comment:
            for text in comment[0].xpath( './text' ):
                oneComment = ServiceParser.toText( text )
                
                if oneComment[1]:
                    parameter.addComment( oneComment[0] , proglang = oneComment[1])
                elif oneComment[2] :
                    parameter.addComment( oneComment[0] , lang = oneComment[2])
                elif oneComment[3]:
                    parameter.addComment( oneComment[0] , href = True )
                else:
                    parameter.addComment( oneComment[0] , lang = 'en')
                
        ctrls = parameterNode.xpath( './ctrl' )
        for ctrl in ctrls:
            messages = []
            codes = []
            messageNodes = ctrl.xpath( './message/text' )
            for messageNode in messageNodes:
                message = ServiceParser.toText( messageNode)
                messages.append( message )
            
            for codeNode in ctrl.xpath( './code'):
                code = codeNode.xpath( './text()' )[0].data
                proglang = codeNode.xpath( './@proglang' )[0].value
                codes.append( ( code , proglang ) )
            
            parameter.addCtrl( ( messages , codes ) ) #pourquoi passer un tuple et non pas 2 param messages et codes ??
       
        precondNodes = parameterNode.xpath( './precond/code' )
        for precondNode in precondNodes :
            precond = precondNode.xpath( './text()' )[0].data
            try:
                proglang = precondNode.xpath( './@proglang' )[0].value
                parameter.addPrecond( precond , proglang = proglang )
            except IndexError:
                parameter.addPrecond( precond )

        try:    
            paramfile= parameterNode.xpath( './paramfile/text()' )[0].data
            parameter.setParamfile( paramfile )
        except IndexError:
            pass
        try:
            filenamesNode = parameterNode.xpath( './filenames' )[0]
            for codeNode in filenamesNode.xpath( './code' ):
                parameter.setFilenames( codeNode.xpath( './text()' )[0].data , codeNode.xpath( './@proglang' )[0].value )
        except IndexError:
            pass
        
       
        try:
            scaleNode = parameterNode.xpath( './scale')[0]
            try:
                minNode = scaleNode.xpath('./min' )[0]
                maxNode = scaleNode.xpath('./max' )[0]
            except IndexError:
                raise MobyleError, "parameter named:\"%s\" have a malformed element scale" % parameter.getName()
            try:
                inc = scaleNode.xpath('./inc/text()')[0].data
            except IndexError:
                inc = None
            try:
                max = maxNode.xpath( './value/text()')[0].data
                maxProglang = None
                min = minNode.xpath( './value/text()')[0].data
                minProgLang = None
                parameter.setScale( min , max , inc = inc )
                
            except IndexError:
                try:
                    minCodes = {}
                    maxCodes = {}
     
                    maxCodeNodes = maxNode.xpath( './code' )
                    minCodeNodes = minNode.xpath( './code' )
                    
                    for codeNode in maxCodeNodes:
                        maxProglang = codeNode.xpath( './@proglang')[0].value                    
                        maxCode = CodeNodes[ i ].xpath( './text()' )[0].data
                        maxCodes[ maxProglang ] = maxCode
    
                    for codeNode in minCodeNodes:
                        minProglang = codeNode.xpath( './@proglang')[0].value                    
                        minCode = CodeNodes[ i ].xpath( './text()' )[0].data
                        minCodes[ minProglang ] = minCode
                        
                    for minProglang in minCodes.keys():
                        parameter.setScale( minCodes[ minProglang ] , maxCodes[ minProglang ] , inc = inc , proglang = minProglang )
                
                except IndexError , keyError :
                    raise ParserError, "parameter named:\"%s\" have a malformed element scale" % parameter.getName()
        except IndexError:
            pass
        
        try:
            separator = parameterNode.xpath( './separator/text()' )[0].data
            parameter.setSeparator( separator )
        except IndexError:
            pass
        try:
            width = parameterNode.xpath( './width/text()' )[0].data
            parameter.setWidth( int( width ) )
        except IndexError:
            pass
        try:
            height= parameterNode.xpath( './height/text()' )[0].data
            parameter.setHeight( int( height ) )
        except IndexError:
            pass
        try:
            interfaceNode = parameterNode.xpath( './interface' )[0]
            #pas implementer dans le service
        except IndexError:
            pass
           
        return parameter
    
    @staticmethod
    def toText( textNode ):
        try:
            content = textNode.xpath('./text()' )[0].data
        except IndexError:
            content = None
        try:
            lang = textNode.xpath('./@lang' )[0].value
            return ( content , None , lang , None )
        except IndexError:
            pass
    
        try:
            href = textNode.xpath('./@href' )[0].value
            return ( content , None , None , href )
        except IndexError:
            return ( content , None , 'en' , None )
        
        try:
            proglang = textNode.xpath('./@proglang' )[0].value
            return ( content , proglang , None , None )
        except IndexError:
            raise ParserError
    
    

if __name__ == "__main__":

    import sys
    sys.path.append( '/home/bneron/Mobyle' )
    sys.path.append( '/home/bneron/Mobyle/Src' )
    
    servicePath = os.path.normpath( os.path.abspath( sys.argv[1] ) )
    #servicePath = "/home/bneron/Mobyle/Src/Mobyle/Test/param2.xml"
    print servicePath
    parser = ServiceParser()
    
    s = parser.parse( servicePath )
    
