#############################################################
#                                                           #
#   Author: Herve Menager                                   #
#   Organization:'Biological Software and Databases' Group, #
#                Institut Pasteur, Paris.                   #
#   Distributed under GPLv2 Licence. Please refer to the    #
#   COPYING.LIB document.                                   #
#                                                           #
#############################################################
"""
Mobyle.Index

This module manages the list of available programs to:
 - search them (using search fields)
 - classify them (building the categories tree)
 - cache this information (on disk as a JSON file)
"""
import os
import string
import Mobyle.ConfigManager
from Mobyle.Registry import *
from Mobyle.MobyleError import MobyleError
from Ft.Xml.Domlette import NoExtDtdReader
from Ft.Xml.Domlette import Print
from Ft.Xml.XPath import Compile, Evaluate
from Ft.Lib import UriException
import pickle
import re

import logging
r_log = logging.getLogger(__name__)

from Mobyle import IndexBase

_cfg = Mobyle.ConfigManager.Config()

queries = {
           u'head': Compile(u'/program/head'),
           u'name': Compile(u'name//text()'),
           u'title': Compile(u'doc/title/text()'), 
           u'description': Compile(u'doc/description/text/text()'),
           u'categories': Compile(u'category/text()'),
           u'comment': Compile(u'doc/comment/text/text()'),
           u'authors': Compile(u'doc/authors/text()'),
           u'references': Compile(u'doc/reference/text()'),
           u'parameter': Compile(u'.//parameter'),
           u'parameter_name': Compile(u'name/text()'),
           u'parameter_prompt': Compile(u'prompt/text()'),
           u'parameter_comment': Compile(u'comment/text/text()'),
           u'parameter_type': Compile(u"type"),
           u'parameter_biotype': Compile(u"biotype/text()"),
           u'parameter_datatype': Compile(u"datatype"),
           u'parameter_class': Compile(u"class/text()"),
           u'paragraph': Compile(u'//paragraph'),
           u'paragraph_name': Compile(u'name/text()'),
           u'paragraph_prompt': Compile(u'prompt/text()'),
           u'paragraph_comment': Compile(u'comment/text/text()')
          }

class SearchIndex(IndexBase.Index):

    indexPath = os.path.join(IndexBase.Index.indexFolder, 'search.dat')
    wheretowrite = os.path.join(IndexBase.Index.tempdir, 'search.dat')

    def filterRegistry(self, keywordList):
        keywordsRe = re.compile('(%s)' % '|'.join(keywordList), re.I)
        programsList = registry.programs[:]
        for program in programsList:
            program.searchMatches = []
            if self.index.has_key(program.url):
                for field, value in self.index[program.url].items():
                    if isinstance(value, basestring):
                       self._searchFieldString(field, value, keywordsRe, program)
                    if isinstance(value, list):
                        for valueItem in value:
                            self._searchFieldString(field, valueItem, keywordsRe, program)
            if len(program.searchMatches) == 0:
                registry.pruneProgram(program)     

    def _searchFieldString(self, fieldName, fieldValue, rx, program):
        if len(rx.findall(fieldValue))>0:
            program.searchMatches.append((fieldName,rx.sub('<b>\\1</b>',fieldValue)))
        
    @classmethod
    def getIndexEntry(cls, doc, index):
        """
        Return an search index entry value
        @return: the index entry: value
        @rtype: object
        """
        head = IndexBase._XPathQuery(doc, queries['head'], 'rawResult')[0]
        fields = {}
        fields['name'] =         IndexBase._XPathQuery(head, queries['name'])
        fields['title'] =        IndexBase._XPathQuery(head, queries['title'])
        fields['description'] =  IndexBase._XPathQuery(head, queries['description'])
        fields['categories'] =   IndexBase._XPathQuery(head, \
                                              queries['categories'], \
                                              'valueList')
        fields['comment'] =      IndexBase._XPathQuery(head, queries['comment'])
        fields['authors'] =      IndexBase._XPathQuery(head, queries['authors'])
        fields['references'] =   IndexBase._XPathQuery(head, queries['references'])
        fields['parameter name'] = []
        fields['parameter prompt'] = []
        fields['parameter comment'] = []
        fields['parameter bioTypes'] = []
        fields['parameter class'] = []
        pars = IndexBase._XPathQuery(doc, queries['parameter'], 'rawResult')
        for p in pars:
            fields['parameter name'].append(IndexBase._XPathQuery(p, queries['parameter_name']))
            fields['parameter prompt'].append(IndexBase._XPathQuery(p, queries['parameter_prompt']))
            fields['parameter comment'].append(IndexBase._XPathQuery(p, queries['parameter_comment']))
            parType = IndexBase._XPathQuery(p, \
                                  queries['parameter_type'], 
                                  'rawResult')[0]
            fields['parameter bioTypes'].append(IndexBase._XPathQuery(parType, \
                                         queries['parameter_biotype']))
            parDataType = IndexBase._XPathQuery(parType, \
                                      queries['parameter_datatype'],\
                                      'rawResult')[0]
            fields['parameter class'].append(IndexBase._XPathQuery(parDataType, \
                                      queries['parameter_class']))
        fields['paragraphs'] = []
        pars = IndexBase._XPathQuery(doc, queries['paragraph'], 'rawResult')
        fields['paragraph name'] = []
        fields['paragraph prompt'] = []
        fields['paragraph comment'] = []
        for p in pars:
            fields['paragraph name'].append(IndexBase._XPathQuery(p, queries['paragraph_name']))
            fields['paragraph prompt'].append(IndexBase._XPathQuery(p, queries['paragraph_prompt']))
            fields['paragraph comment'].append(IndexBase._XPathQuery(p, queries['paragraph_comment']))
        return fields
