#############################################################
#                                                           #
#   Author: Herve Menager                                   #
#   Organization:'Biological Software and Databases' Group, #
#                Institut Pasteur, Paris.                   #
#   Distributed under GPLv2 Licence. Please refer to the    #
#   COPYING.LIB document.                                   #
#                                                           #
#############################################################
"""
Registry.py

This module holds:
- The Registry class that describes the list of installed/imported programs 
  and their servers
- The DefNode class, an abstract class that describes an element (node or
  leaf) of the programs tree
- The ProgramDef class that describes a program (leaf in the programs tree)
- The CategoryDef class that describes a category (node in the programs tree)
- The ServerDef class that describes a server (node in the programs tree)
"""
import os
from hashlib import md5
import glob

import Mobyle.ConfigManager

import logging
from Mobyle import MobyleLogger
MobyleLogger.MLogger()
r_log = logging.getLogger('mobyle.registry' )

_cfg = Mobyle.ConfigManager.Config()

class DefNode(object):
    """
    DefNode is the class that provides the tree structure to the 
    server/categories/program hierarchy
    """

    def __init__(self):
        self.name = None
        self.children = []
        self.parents = []
              
    def getDescendants(self):
        """
        Returns the descendants of the node recursively
        @return: the list of descendant nodes
        @rtype: list
        """
        r = []
        r.extend(self.children)
        for p in self.children:
            r.extend(p.getDescendants())
        return r

    def getAncestors(self):
        """
        Returns the ancestors of the node recursively
        @return: the list of ancestor nodes
        @rtype: list
        """
        r = []
        r.extend(self.parents)
        for p in self.parents:
            r.extend(p.getAncestors())
        return r

    def addDescendant(self, subpath):
        """
        Adds a descendant to the node, by recursively adding the descendants from the provided subpath if they do not exist
        @param subpath: the subpath, a list of descendant objects 
                        (e.g.: [CategoryDef, CategoryDef, ProgramDef])
        @type subpath: string
        """
        if len(subpath)==0:
            return
        else:
            c = [c for c in self.children if (c.name == subpath[0].name and (not(hasattr(c,'server')) or c.server == subpath[0].server))]
            if not(c):
                c = subpath[0]
                self.children.append(c)
                c.parents.append(self)
            else:
                c = c[0]
            c.addDescendant(subpath[1:])
            
    def __cmp__(self, other):
        """
        Comparison operator that allows to classify the tree structure alphabetically
        based on node names
        @param other: the other node to which self is compared
        @type other: DefNode
        @return: the comparison result
        @rtype: boolean
        """
        return cmp(self.name.lower(), other.name.lower())
    
    def getChildCategories(self):
        """
        Returns child categories, i.e. child nodes which happen to be CategoryDef instances
        @return: a list of category nodes
        @rtype: list
        """
        l = [c for c in self.children if not(isinstance(c, ProgramDef))]
        l.sort()
        return l

    def getChildPrograms(self):
        """
        Returns child programs, i.e. child nodes which happen to be ProgramDef instances
        @return: a list of program nodes
        @rtype: list
        """
        l = [c for c in self.children if isinstance(c, ProgramDef)]
        l.sort()
        return l

class Registry(DefNode):
    
    __shared_state = {}
    
    def __init__(self):
        self.__dict__ = self.__shared_state
        DefNode.__init__(self)
        self.load()
    
    def load(self, selectedPrograms=None, fromTmpDir=False):
        self.programs = []
        self.viewers = []
        self.servers = []
        self.serversByName = {}
        self.serversByUrl = {}
        self.programsByUrl = {}
        self.viewersByUrl = {}
        self.viewersByName = {}
        #variable to know if we are in case of prepublication
        self.fromTmpDir = fromTmpDir
           
        serverProperties = self._getServerProperties()
        for name, properties in serverProperties.items():
            server = ServerDef( name = name,
                                url = properties['url'] ,
                                help = properties['help'],
                                repository = properties['repository'],
                                jobsBase = properties['jobsBase']
                               )
            self._addServer(server)
            for name in properties['programs']:
                try:                        
                    url = self.getProgramUrl(name, server.name)
                    if not(selectedPrograms) or (selectedPrograms and url in selectedPrograms.keys()):
                        path = self.getProgramPath(name, server.name)
                        program = ProgramDef( name=name,
                                              url=url,
                                              path=path,
                                              server=server
                                              )
                        # program disabled
                        if _cfg.isDisabled(programName=url): 
                            program.disabled = True
                        # program with restricted access
                        if not(_cfg.isAuthorized(url)):
                            program.authorized = False
                        self._addProgram(program)
                except IndexError, e:
                    r_log.error("Error while loading %s: %s" % (name, e))
            
            if server.name == 'local':
                for name in properties['viewers']:
                    try:                        
                        url = self.getViewerUrl(name)
                        path = self.getViewerPath(name)
                        v = ViewerDef( name= name,
                                       url= url,
                                       path=path,
                                       server=server
                                              )
                        self._addViewer(v)
                    except IndexError, e:
                        r_log.error("Error while loading %s: %s" % (name, e))

    def _addServer(self, server):
        self.servers.append(server)
        self.serversByName[server.name] = server
        self.serversByUrl[server.url] = server

    def _addProgram(self, program):
        self.programs.append(program)
        self.programsByUrl[program.url] = program
        self.serversByName[program.server.name].programs.append(program)
        self.serversByName[program.server.name].\
            programsByName[program.name] = program

    def _addViewer(self, viewer):
        self.viewers.append(viewer)
        self.viewersByUrl[viewer.url] = viewer
        self.viewersByName[viewer.name] = viewer

    def pruneProgram(self, program):
        if self.programsByUrl.has_key(program.url):
            self.programs.remove(program)
            del self.programsByUrl[program.url]
            self.serversByName[program.server.name].programs.remove(program)
            del self.serversByName[program.server.name].\
                programsByName[program.name]
            if (len(program.server.programs)==0):
                del self.serversByName[program.server.name]
                self.servers.remove(program.server)

    def _getServerProperties(self):
        #local programs
        p = _cfg.programs_path(self.fromTmpDir)
        localPaths = glob.glob(os.path.join(p, "*.xml"))   
        programNames = [os.path.basename(programPath)[:-4] \
                      for programPath in localPaths]
        #viewers
        p = _cfg.viewers_path(self.fromTmpDir)
        localPaths = glob.glob(os.path.join(p, "*.xml"))   
        viewerNames = [os.path.basename(viewerPath)[:-4] \
                      for viewerPath in localPaths]
        #imported programs
        imported_p = _cfg.portals()
        #check that the xml is effectively available 
        if self.fromTmpDir:
            for name, properties in imported_p.items():
                if name !='local':
                    dir = properties['repository']
                    for n in properties['programs']:
                        url = '%s/%s.xml' % (dir, n)
                        if not os.access(os.path.join(_cfg.programs_path(self.fromTmpDir), "imports", md5(url).hexdigest()+'.xml'), os.F_OK):
                            properties['programs'].remove(n)
        serverProperties = imported_p
        
        serverProperties['local'] = {
           'url': _cfg.cgi_url(),
           'help' : _cfg.mailHelp(),
           'repository': _cfg.programs_url(),
           'programs': programNames,
           'viewers': viewerNames,
           'jobsBase': _cfg.results_url()
        }
        return serverProperties
    
    def getProgramUrl(self, name, server='local'):
        return "%s/%s.xml" % \
            ( self.serversByName[server].repository , name )
            
    def getViewerUrl(self, name):
        return "%s/%s.xml" % \
            ( _cfg.viewers_url(), name )
            
    def getJobPID(self, url):
        server = self.getServerByJobId(url)
        jobPID = url.replace(server.jobsBase,'').lstrip('/').replace('/','.')
        if server.name != 'local':
            jobPID = server.name + jobPID
        return jobPID

    def getJobURL(self, pid):
        if (len(pid.split('/'))==1):
            return self.serversByName['local'].jobsBase + '/' + pid.replace('.','/')
        else:
            return self.serversByName[pid.split('/')[0]].jobsBase + '/' + pid.replace('.','/')

    def getViewerPath(self, name):
        p = _cfg.viewers_path(self.fromTmpDir)
        return os.path.join(p, name+'.xml')
        
    def getProgramPath(self, name, server='local'):
        if server != 'local':
            p = os.path.join(_cfg.programs_path(self.fromTmpDir), "imports")
            url = self.getProgramUrl(name, server)
            return os.path.join(p, \
                                md5(url).hexdigest()+'.xml')
        else:
            p = _cfg.programs_path(self.fromTmpDir)
            return os.path.join(p, name+'.xml')
        
    def getServerByJobId(self,jobId):
        for server in self.servers:
            if jobId.startswith(server.jobsBase):
                return server


class ServiceDef(DefNode):
    """
    ServiceDef is the superclass that provides the service information to the registry
    """
    def __init__(self, url, name=None, path=None, server=None):
        
        DefNode.__init__(self)
        self.url = url
        self.name = name
        self.path = path
        self.server = server
        self.categories = []
        

class ProgramDef(ServiceDef):
    """
    ProgramDef is the class that provides the program information to the registry
    """
        
    def __init__(self, url, name=None, path=None, server=None):
        """
        @param url: the url of the definition of the program on the server where it is executed
        @type url: String
        @param name: the name of the program 
        @type name: String
        @param path: the os path to the local version of the file (local file definition or local cache of distant program)
        @type path: String
        @param server: the execution server definition
        @type server: ServerDef
        """
        ServiceDef.__init__(self, url, name, path, server)
        self.disabled = False
        self.authorized = True
        
        if (self.server.name == 'local'):
            self.pid = self.name
        else:
            self.pid = '%s.%s' % (self.server.name, self.name)
        
    def isExported(self):
        return self.server.name == 'local' and \
            self.name in _cfg.exported_programs()


class ViewerDef(ServiceDef):
    """
    ViewerDef is the class that provides the viewer information to the registry
    """
    
    def __init__(self, url, name=None, path=None, server=None):
        """
        @param url: the url of the definition of the viewer on the server where it is executed
        @type url: String
        @param name: the name of the viewer 
        @type name: String
        @param path: the os path to the file
        @type path: String
        @param server: the execution server definition
        @type server: ServerDef
        """
        ServiceDef.__init__(self, url, name, path, server)



class ServerDef(DefNode):
    """
    ServerDef is the class that provides the server information to the registry
    """

    def __init__(self, name, url, help, repository, jobsBase):
        """    
        @param name: the short name, as displayed for instance in the programs tree menu
        @type name: String
        @param url: the url of the server (cgi-bin base url for now)
        @type url: String
        @param help: the help email contact
        @type help: String
        @param repository: the repository url
        @type repository: String
        @param jobsBase: the url of the directory that contains all of the server's jobs
        @type jobsBase: String
        """
        DefNode.__init__(self)
        self.name = name
        self.url = url
        self.help = help
        self.repository = repository
        self.programs = [] # top-level programs
        self.programsByName = {}
        self.categories = [] # top-level categories
        if self.name == 'local':
            self.path = _cfg.programs_path()
        else:
            self.path = _cfg.programs_imports_path()
        self.jobsBase = jobsBase


class CategoryDef(DefNode):
    """
    CategoryDef is the class that provides the category information to the registry
    """

    def __init__ (self, name, parentCategory=None, server=None):
        """    
        @param name: the name of the category
        @type name: String
        @param parentCategory: the parent category (if any)
        @type parentCategory: CategoryDef
        @param server: the server to which the Category belongs
        @type server: ServerDef
        """
        DefNode.__init__(self)
        self.name = name
        self.parentCategory = parentCategory
        self.server = server
        self.programs = [] # top-level programs
        self.categories = [] # top-level categories
        
registry = Registry()
